// -*-c++-*-

/*!
  \file gzfstream.h
  \brief gzip file stream Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCTOOLS_RCSC_GZ_GZFSTREAM_H
#define RCTOOLS_RCSC_GZ_GZFSTREAM_H

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <iostream>
#include <string>
#include <boost/scoped_ptr.hpp>

namespace rcsc {

struct gzfilebuf_impl;

/*!
  \brief gzip file stream buffer class.

  This class implements basic_filebuf for gzipped files.
  It doesn't yet support seeking (allowed by zlib but slow/limited),
  putback and read/write access(tricky). Otherwise, it attempts
  to be a drop-in replacement for the standard file streambuf.
*/
class gzfilebuf
    : public std::streambuf {
public:
    /*!
      The list of compresion level defined in zlib.h
      - Z_NO_COMPRESSION
      - Z_BEST_SPEED
      - Z_BEST_COMPRESSION
      - Z_DEFAULT_COMPRESSION
    */
    enum Compression {
        DEFAULT_COMPRESSION = -1,
        NO_COMPRESSION = 0,
        BEST_SPEED = 1,
        BEST_COMPRESSION = 9,
    };

    /*!
      The list of strategy defined in zlib.h
      - Z_DEFAULT_STRATEGY
      - Z_FILTERED
      - Z_HUFFMAN_ONLY
      - Z_RLE  // run-length

      For more details, see deflateInit2 in zlib.h.
    */
    enum Strategy {
        DEFAULT_STRATEGY = 0,
        FILTERED = 1,
        HUFFMAN_ONLY = 2,
        RLE = 3,
    };

private:

    //! Pimpl ideom. the instance of a file buffer.
    boost::scoped_ptr< gzfilebuf_impl > M_impl;

    //! buffer size (default: 8192)
    std::size_t M_buf_size;
    //! pointer to the stream buffer. This is used as array.
    char_type * M_buf;

    //! used in underflow
    int M_remained_size;
    //! used in underflow
    char_type M_remained_char;


    //! not used
    gzfilebuf( const gzfilebuf & );
    //! not used
    gzfilebuf & operator=( const gzfilebuf & );

public:
    //! default constructor.
    gzfilebuf();

    //! destructor.
    virtual
    ~gzfilebuf();

    //! check if file is open.
    bool is_open();

    //! open the file.
    gzfilebuf * open( const char * path,
                      std::ios_base::openmode mode,
                      int level = DEFAULT_COMPRESSION,
                      int strategy = DEFAULT_STRATEGY );

    //! close the file
    gzfilebuf * close() throw();


private:
    //! flush current buffer
    bool flushBuf();

    //! make open mode string for gzopen
    std::string makeModeString( std::ios_base::openmode mode,
                                int level,
                                int strategy );

    //! release the buffer and set pointers to NULL.
    void destroyInternalBuffer() throw();

protected:
    //virtual
    //void imbue( const locale& loc );
    //virtual
    //int_type pbackfail( int_type c );
    //virtual
    //std::streambuf* setbuf( char_type* p, std::streamsize n );

    virtual
    std::streampos seekoff( std::streamoff off,
                            std::ios_base::seekdir way,
                            std::ios_base::openmode mode );

    virtual
    std::streampos seekpos( std::streampos pos,
                            std::ios_base::openmode mode );

    virtual
    std::streamsize showmanyc();

    virtual
    int sync();

    virtual
    int_type overflow( int_type c );

    // Not need to override because super class works sufficiently.
    //virtual
    //int_type uflow();

    virtual
    int_type underflow();

    // Not need to override because super class works sufficiently.
    //virtual
    //std::streamsize xsgetn( char_type* s, std::streamsize n );

    // Not need to override because super class works sufficiently.
    //virtual
    //std::streamsize xsputn( char_type * s, std::streamsize n );

    /*   All virtual methods defined in std::streambuf.
         See: http://www.cplusplus.com/ref/iostream/
      XXX imbue Imbue locale [virtual]
      OK overflow Put character at current position [virtual]
      XXX pbackfail Put character back [virtual]
      N/A seekoff Set relative position of internal position pointer [virtual]
      OK seekpos  Set absolute position of internal position pointer [virtual]
      XXX setbuf  Set buffer [virtual]
      OK showmanyc  Get number of characters availbale in input sequence [virtual]
      OK sync Synchronize stream buffer [virtual]
      OK uflow  Get current character [virtual]
      OK underflow  Get current character [virtual]
      OK xsgetn Get some characters [virtual]
      OK xsputn Write some characters [virtual]
    */
};


/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*/


/*!
  Gzipped file input stream class.
  This class implements ifstream for gzipped files. Seeking and putback
  is not supported yet.
*/
class gzifstream
    : public std::istream {
private:
    //! underlying stream buffer.
    gzfilebuf M_file_buf;
public:
    //! default constructor
    gzifstream();

    //! init stream buffer & open file.
    explicit
    gzifstream( const char * path );

    //! returns underlying stream buffer.
    gzfilebuf * rdbuf() const
      {
          return const_cast< gzfilebuf * >( &M_file_buf );
      }

    /*!
      check if file is open.
      \return  True if file is open.
    */
    bool is_open()
      {
          return M_file_buf.is_open();
      }

    //! open gzipped file.
    void open( const char * name );

    //! close gzipped file.
    void close();
};

/*****************************************************************************/

/*!
  gzipped file output stream class.
  This class implements ofstream for gzipped files. Seeking and putback
  is not supported yet.
*/
class gzofstream
    : public std::ostream {
private:
    //! underlying stream buffer.
    gzfilebuf M_file_buf;

public:
    //  default constructor
    gzofstream();

    //! construct streambuf with file name.
    explicit
    gzofstream( const char* path,
                int level = gzfilebuf::DEFAULT_COMPRESSION,
                int strategy = gzfilebuf::DEFAULT_STRATEGY );

    //! get const_cast<> pointer to the underlying stream buffer.
    gzfilebuf * rdbuf() const
      {
          return const_cast< gzfilebuf * >( &M_file_buf );
      }

    //! check if file is open
    bool is_open()
      {
          return M_file_buf.is_open();
      }

    //! open file and go into fail() state if unsuccessful
    void  open( const char * name,
                int level = gzfilebuf::DEFAULT_COMPRESSION,
                int strategy = gzfilebuf::DEFAULT_STRATEGY );

    //! close file.
    void close();
};

} // end namespace

#endif // include guard
