// -*-c++-*-

/*!
  \file loader.h
  \brief rcg data loader Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCTOOLS_RCSC_RCG_LOADER_H
#define RCTOOLS_RCSC_RCG_LOADER_H

#include <string>
#include <rcsc/rcg/types.h>
#include <rcsc/rcg/handler.h>
#include <rcsc/rcg/holder.h>

namespace rcsc {
namespace rcg {

/*!
  \brief rcg data loader interface class.
*/
class Loader
    : public Handler {
private:

    //! reference to the repository instance.
    Holder & M_holder;

    //! default constructor must not be used.
    Loader();
    //! copy constructor must not be used.
    Loader( const Loader & );
    //! substitution operator must not be used.
    Loader& operator=( const Loader & );

public:
    //! constructor.
    /*!
      It is assumed that only this is used to construct this class.
      \param r reference to the holder instance.
     */
    explicit
    Loader( Holder & holder )
        : M_holder( holder )
      { }

    //! destructor
    virtual
    ~Loader()
      { }

    //! adds distpinfo_t to the holder.
    /*!
      This method is used for rcg v1 format.
      \param info const reference of dispinfo_t struct.
      \return registry status.
    */
    virtual
    bool handleDispInfo( const dispinfo_t & info )
      {
          return M_holder.addDispInfo( info );
      }

    //! adds showinfo_t to the holder.
    /*!
      This method is used for rcg v2 format.
      \param info const reference of showinfo_t struct.
      \return registry status.
    */
    virtual
    bool handleShowInfo( const showinfo_t & info )
      {
          return M_holder.addShowInfo( info );
      }

    //! adds short_showinfo_t2 to the holder.
    /*!
      This method is used for rcg v3 format.
      \param info const reference of short_showinfo_t struct.
      \return registry status.
    */
    virtual
    bool handleShortShowInfo2( const short_showinfo_t2 & info )
      {
          return M_holder.addShortShowInfo2( info );
      }

    //! adds msginfo_t to the holder.
    /*!
      \param board board parametor of msginfo_t
      \param msg converted std::string message.
      \return registry status.
    */
    virtual
    bool handleMsgInfo( short board,
                        const std::string & msg )
      {
          return M_holder.addMsgInfo( board, msg );
      }

    //! adds latest playmode to the holder.
    /*!
      \param playmode type id(char) of playmode.
      \return registry status.
    */
    virtual
    bool handlePlayMode( char playmode )
      {
          return M_holder.addPlayMode( playmode );
      }

    //! adds two team info to the holder.
    /*!
      \param team_left team_t for the left team
      \param team_right team_t for the right team
      \return registry status.
    */
    virtual
    bool handleTeamInfo( const team_t & team_left,
                         const team_t & team_right )
      {
          return M_holder.addTeamInfo( team_left, team_right );
      }

    //! adds player_type_t to the holder.
    /*!
      \param type player_type_t info
      \return registry status.
    */
    virtual
    bool handlePlayerType( const player_type_t & type )
      {
          return M_holder.addPlayerType( type );
      }

    //! adds server_params_t to the holder.
    /*!
      \param param server_params_t info
      \return registry status.
    */
    virtual
    bool handleServerParam( const server_params_t & param )
      {
          return M_holder.addServerParam( param );
      }

    //! adds player_params_t to the holder.
    /*!
      \param param player_params_t info
      \return registry status.
    */
    virtual
    bool handlePlayerParam( const player_params_t & param )
      {
          return M_holder.addPlayerParam( param );
      }

    //! called when stream reaches the end of file.
    /*!
      Please override this in the derived classes.
      \return always true.
    */
    virtual
    bool handleEOF()
      {
          return true;
      }

};

} // end of namespace
} // end of namespace

#endif
