// -*-c++-*-

/*!
  \file parser.cpp
  \brief abstract rcg parser & holder classes Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "types.h"
#include "handler.h"
#include "parser.h"

namespace rcsc {
namespace rcg {

/*-------------------------------------------------------------------*/
/*!
  static method for singleton pattern.
  \return reference to the local static variable.
*/
ParserHolder&
ParserHolder::instance()
{
    static ParserHolder s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!
  register parser object.
  \param smart pointer to the parser object.
*/
bool
ParserHolder::add( ParserPtr p )
{
    if ( ! p )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " Parser pointer is NULL. Failed to register."
                  << std::endl;
        return false;
    }
    ParserMap::const_iterator it = M_parser_map.find( p->version() );
    if ( it != M_parser_map.end() )
    {
        std::cerr << "already registered. version = " << p->version() << std::endl;
        return false;
    }

    //std::cerr << p->version() << ": pointer count = " << p.use_count()
    //<< std::endl;
    M_parser_map.insert( std::make_pair( p->version(), p ) );
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  After checking, the ptr point of istreambuf becomes 4.
  \param istrm reference to the imput stream.
  \return rcg version number.
*/
int
ParserHolder::parseVersion( std::istream & istrm ) const
{
    char header[4];
    int ver = REC_OLD_VERSION;

    istrm.read( header, 4 ); // read 'U', 'L', 'G', <version>

    if ( ! istrm.good() )
    {
        return -1;
    }
    //std::cerr << "rcg header ["
    //          << header[0] << header[1] << header[2] << (int)header[3]
    //          << "]"
    //          << std::endl;
    if ( header[0] == 'U'
         && header[1] == 'L'
         && header[2] == 'G' )
    {
        ver = static_cast< int >( header[3] );
    }

    return ver;
}

/*-------------------------------------------------------------------*/
/*!
  This method creates a new parser.
  If parser is not found, NULL pointer is returned.
  \param istrm reference to the input stream.
  \return pointer to a new parser instance. if not found, return NULL.
*/
const
ParserPtr
ParserHolder::get( std::istream & istrm ) const
{
    return this->get( this->parseVersion( istrm ) );
}

/*-------------------------------------------------------------------*/
/*!
  This method creates a new parser.
  if parser is not foudn, return NULL pointer.
  \param ver parser version.
  \return pointer to new parser instance. if not found, return NULL.
*/
const
ParserPtr
ParserHolder::get( const int ver ) const
{
    ParserMap::const_iterator it = M_parser_map.find( ver );
    if ( it != M_parser_map.end() )
    {
        return it->second;
    }

    std::cerr << __FILE__ << ':' << __LINE__
              << " parser not found. version = " << ver << std::endl;
    ParserPtr ptr( static_cast< Parser* >( 0 ) );
    return ptr;
}

} // end of namespace
} // end of namespace
