// -*-c++-*-

/*!
  \file parser_v2.cpp
  \brief rcg v2 parser Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#include "types.h"
#include "handler.h"
#include "parser_v2.h"

namespace rcsc {
namespace rcg {

/*-------------------------------------------------------------------*/
/*!
  \param istrm reference to the imput stream (usually ifstream).
  \param handler reference to the rcg data handler.
  \retval true if successfuly parsed.
  \retval false if incorrect format is detected.
*/
bool
ParserV2::parse( std::istream & istrm,
                 Handler & handler ) const
{
    // streampos must be the first point!!!
    istrm.seekg( 0 );

    if ( ! istrm.good() )
    {
        return false;
    }

    // skip header
    char header[4];
    istrm.read( header, 4 ); // read 'U', 'L', 'G', <version>

    // register log version
    handler.handleLogVersion( REC_VERSION_2 );

    // main loop
    while ( istrm.good() )
    {
        // read data
        if ( ! parseData( istrm, handler ) )
        {
            return false;
        }
    }

    if ( istrm.eof() )
    {
        return handler.handleEOF();
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  This method is called in the loop of main parse method.
  First, check the type of data mode.
  Second, call each data item parsing method.
  \param istrm reference to the imput stream (usually ifstream).
  \param handler reference to the rcg data handler.
  \retval true if successfuly parsed.
  \retval false if incorrect format is detected.
*/
bool
ParserV2::parseData( std::istream & istrm,
                     Handler & handler ) const
{
    // chedk data mode.
    dispinfo_t info;
    istrm.read( reinterpret_cast< char* >( &info.mode ), sizeof( short ) );

    if ( ! istrm.good() )
    {
        if ( istrm.eof() )
        {
            return true;
        }
        return false;
    }

    // read each data block
    switch ( ntohs( info.mode ) ) {
    case NO_INFO:
        return true;
    case SHOW_MODE:
        istrm.read( reinterpret_cast< char* >( &info.body.show ),
                    sizeof( showinfo_t ) );
        if ( istrm.gcount() == sizeof( showinfo_t ) )
        {
            return handler.handleShowInfo( info.body.show );
        }
        break;
    case MSG_MODE:
        istrm.read( reinterpret_cast< char* >( &info.body.msg ),
                    sizeof( msginfo_t ) );
        if ( istrm.gcount() == sizeof( msginfo_t ) )
        {
            return handler.handleShowInfo( info.body.show );
        }

        break;
    case DRAW_MODE:
        istrm.read( reinterpret_cast< char* >( &info.body.draw ),
                    sizeof( drawinfo_t ) );
        if ( istrm.gcount() == sizeof( drawinfo_t ) )
        {
            return true;
        }
        break;
    default:
        std::cerr << __FILE__ << ':' << __LINE__
                  << "Unknown mode" << htons( info.mode )
                  << std::endl;;
    }

    return false;
}

} // end of namespace
} // end of namespace
