// -*-c++-*-

/*!
  \file parser_v3.cpp
  \brief rcg v3 parser Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#include "types.h"
#include "handler.h"
#include "parser_v3.h"

namespace rcsc {
namespace rcg {

/*-------------------------------------------------------------------*/
/*!
  \param istrm reference to the imput stream (usually ifstream).
  \param handler reference to the rcg data handler.
  \retval true if successfuly parsed.
  \retval false if incorrect format is detected.
*/
bool
ParserV3::parse( std::istream & istrm,
                 Handler & handler ) const
{
    // streampos must be the first point!!!
    istrm.seekg( 0 );

    if ( ! istrm.good() )
    {
        return false;
    }

    // skip header
    char header[4];
    istrm.read( header, 4 ); // read 'U', 'L', 'G', <version>
    //std::cerr << "skip header " << header[0] << header[1]
    //<< header[2] << (int)header[3] << std::endl;
    // register log version
    handler.handleLogVersion( REC_VERSION_3 );

    while ( istrm.good() )
    {
        //std::cerr << "loop... " << std::endl;
        // read data
        if ( ! parseData( istrm, handler ) )
        {
            return false;
        }
    }

    if ( istrm.eof() )
    {
        return handler.handleEOF();
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  \brief First, check the type of data mode.
  Second, call each data item parsing method.
  \param istrm reference to the imput stream (usually ifstream).
  \param handler reference to the rcg data handler.
  \retval true, if successfuly parsed.
  \retval false, if incorrect format is detected.
*/
bool
ParserV3::parseData( std::istream & istrm,
                     Handler & handler ) const
{
    // chedk data mode.
    short mode;
    istrm.read( reinterpret_cast< char* >( &mode ),
                sizeof( short ) );
    if ( ! istrm.good() )
    {
        //std::cerr << "istream become bad " << std::endl;
        if ( istrm.eof() )
        {
            return true;
        }
        return false;
    }

    //std::cerr << "read mode = " << (int)ntohs( mode ) << std::endl;
    // read each data block
    switch ( ntohs( mode ) ) {
    case NO_INFO:
        return true;
    case SHOW_MODE:
        return parseShowInfo( istrm, handler );
    case MSG_MODE:
        return parseMsgInfo( istrm, handler );
    case PM_MODE:
        return parsePlayMode( istrm, handler );
    case TEAM_MODE:
        return parseTeamInfo( istrm, handler );
    case PT_MODE:
        return parsePlayerType( istrm, handler );
    case PARAM_MODE:
        return parseServerParam( istrm, handler );
    case PPARAM_MODE:
        return parsePlayerParam( istrm, handler );
    case BLANK_MODE:
    case DRAW_MODE:
    default:
        throw std::string( "Unknown mode" );
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse SHOW_MODE info
  \param istrm reference to the input stream
  \param handler reference to the data handler object
  \retval true if successfully parsed.
  \retval false if failed to parse.
*/
bool
ParserV3::parseShowInfo( std::istream & istrm,
                         Handler & handler ) const
{
    short_showinfo_t2 short_show;
    istrm.read( reinterpret_cast< char* >( &short_show ),
                sizeof( short_showinfo_t2 ) );
    if ( istrm.gcount() == sizeof( short_showinfo_t2  ) )
    {
        return handler.handleShortShowInfo2( short_show );
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse MSG_MODE info
  \param istrm reference to the input stream
  \param handler reference to the data handler object
  \retval true if successfully parsed.
  \retval false if failed to parse.
*/
bool
ParserV3::parseMsgInfo( std::istream & istrm,
                        Handler & handler ) const
{
    bool result = false;

    short board;
    istrm.read( reinterpret_cast< char* >( &board ), sizeof( short ) );
    if ( istrm.gcount() != sizeof( short ) )
    {
        return false;
    }

    short len;
    istrm.read( reinterpret_cast< char* >( &len ), sizeof( short ) );
    if ( istrm.gcount() != sizeof( short ) )
    {
        return false;
    }
    len = ntohs( len );

    char * msg = new char[len];
    istrm.read( msg, len );
    if ( istrm.gcount() == len )
    {
        if ( msg[len - 1] == 0 )
        {
            len = std::strlen( msg );
        }
        std::string msgstr( msg, len );
        result = handler.handleMsgInfo( board, msgstr );
    }

    delete [] msg;
    return result;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse PM_MODE info
  \param istrm reference to the input stream
  \param handler reference to the data handler object
  \retval true if successfully parsed.
  \retval false if failed to parse.
*/
bool
ParserV3::parsePlayMode( std::istream & istrm,
                         Handler & handler ) const
{
    char pmode;
    istrm.read( reinterpret_cast< char* >( &pmode ),
                sizeof( char ) );
    if ( istrm.gcount() == sizeof( char ) )
    {
        return handler.handlePlayMode( pmode );
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse TEAM_MODE info
  \param istrm reference to the input stream
  \param handler reference to the data handler object
  \retval true if successfully parsed.
  \retval false if failed to parse.
*/
bool
ParserV3::parseTeamInfo( std::istream & istrm,
                         Handler & handler ) const
{
    team_t team[2];
    istrm.read( reinterpret_cast< char* >( team ),
                sizeof( team_t ) * 2 );
    if ( istrm.gcount() == sizeof( team_t ) * 2 )
    {
        return handler.handleTeamInfo( team[0], team[1] );
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse PT_MODE info
  \param istrm reference to the input stream
  \param handler reference to the data handler object
  \retval true if successfully parsed.
  \retval false if failed to parse.
*/
bool
ParserV3::parsePlayerType( std::istream & istrm,
                           Handler & handler ) const
{
    player_type_t ptinfo;
    istrm.read( reinterpret_cast< char* >( &ptinfo ),
                sizeof( player_type_t ) );
    if ( istrm.gcount() == sizeof( player_type_t ) )
    {
        return handler.handlePlayerType( ptinfo );
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse PARAM_MODE info
  \param istrm reference to the input stream
  \param handler reference to the data handler object
  \retval true if successfully parsed.
  \retval false if failed to parse.
*/
bool
ParserV3::parseServerParam( std::istream & istrm,
                            Handler & handler ) const
{
    server_params_t sparams;
    istrm.read( reinterpret_cast< char* >( &sparams ),
                sizeof( server_params_t ) );
    if ( istrm.gcount() == sizeof( server_params_t ) )
    {
        return handler.handleServerParam( sparams );
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  \brief parse PPARAM_MODE info
  \param istrm reference to the input stream
  \param handler reference to the data handler object
  \retval true if successfully parsed.
  \retval false if failed to parse.
*/
bool
ParserV3::parsePlayerParam( std::istream & istrm,
                            Handler & handler ) const
{
    player_params_t pparams;
    istrm.read( reinterpret_cast< char* >( &pparams ),
                sizeof( pparams ) );
    if ( istrm.gcount() == sizeof( player_params_t ) )
    {
        return handler.handlePlayerParam( pparams );
    }

    return false;
}

} // end of namespace
} // end of namespace
