// -*-c++-*-

/*!
  \file types.h
  \brief rcg data format types.
*/

/*
 *Copyright:

 Copyright (C) 2004 Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCTOOLS_RCSC_RCG_TYPES_H
#define RCTOOLS_RCSC_RCG_TYPES_H

namespace rcsc {
namespace rcg {

//! side type id
enum Side {
    LEFT = 1,
    NEUTRAL = 0,
    RIGHT = -1,
};

//! playmode types
enum PlayMode {
    PM_Null,
    PM_BeforeKickOff,
    PM_TimeOver,
    PM_PlayOn,
    PM_KickOff_Left,
    PM_KickOff_Right,
    PM_KickIn_Left,
    PM_KickIn_Right,
    PM_FreeKick_Left,
    PM_FreeKick_Right,
    PM_CornerKick_Left,
    PM_CornerKick_Right,
    PM_GoalKick_Left,
    PM_GoalKick_Right,
    PM_AfterGoal_Left,
    PM_AfterGoal_Right, // - sserver-2.94
    PM_Drop_Ball, // - sserver-3.29
    PM_OffSide_Left,
    PM_OffSide_Right, // untill sserver-5.27
    // [I.Noda:00/05/13] added for 3D viewer/commentator/small league
    PM_PK_Left,
    PM_PK_Right,
    PM_FirstHalfOver,
    PM_Pause,
    PM_Human,
    PM_Foul_Charge_Left,
    PM_Foul_Charge_Right,
    PM_Foul_Push_Left,
    PM_Foul_Push_Right,
    PM_Foul_MultipleAttacker_Left,
    PM_Foul_MultipleAttacker_Right,
    PM_Foul_BallOut_Left,
    PM_Foul_BallOut_Right, // until sserver-7.11
    PM_Back_Pass_Left, // after rcssserver-8.05-rel
    PM_Back_Pass_Right,
    PM_Free_Kick_Fault_Left,
    PM_Free_Kick_Fault_Right,
    PM_CatchFault_Left,
    PM_CatchFault_Right,
    PM_IndFreeKick_Left, // after rcssserver-9.2.0
    PM_IndFreeKick_Right,
    PM_PenaltySetup_Left,  // after rcssserver-9.3.0
    PM_PenaltySetup_Right,
    PM_PenaltyReady_Left,
    PM_PenaltyReady_Right,
    PM_PenaltyTaken_Left,
    PM_PenaltyTaken_Right,
    PM_PenaltyMiss_Left,
    PM_PenaltyMiss_Right,
    PM_PenaltyScore_Left,
    PM_PenaltyScore_Right,
    PM_MAX
};

//! max player number in one team
const int MAX_PLAYER = 11;

//! not used
const int COLOR_NAME_MAX = 64;

//! monitor data type identifier.
enum DispInfoMode {
    NO_INFO     = 0, //!< dummy type
    SHOW_MODE   = 1, //!< showinfo_t
    MSG_MODE    = 2, //!< msg info
    DRAW_MODE   = 3, //!< drawinfo_t
    BLANK_MODE  = 4, //!< dummy. until sserver-6.07
    PM_MODE     = 5, //!< playmode
    TEAM_MODE   = 6, //!< team_t (team name & score)
    PT_MODE     = 7, //!< player_type_t
    PARAM_MODE  = 8, //!< server_params_t
    PPARAM_MODE = 9  //!< player_params_t
};

//! message data type id.
enum MsgInfoMode {
    MSG_BOARD = 1,
    LOG_BOARD = 2
};

//! player status bit mask.
enum PlayerStatus {
    DISABLE         = 0x0000,
    STAND           = 0x0001,
    KICK            = 0x0002,
    KICK_FAULT      = 0x0004,
    GOALIE          = 0x0008,
    CATCH           = 0x0010,
    CATCH_FAULT     = 0x0020,
    BALL_TO_PLAYER  = 0x0040,
    PLAYER_TO_BALL  = 0x0080,
    DISCARD         = 0x0100,
    LOST            = 0x0200,
    BALL_COLLIDE    = 0x0400,
    PLAYER_COLLIDE  = 0x0800,
    TACKLE          = 0x1000,
    TACKLE_FAULT    = 0x2000,
    BACK_PASS       = 0x4000,
    FREE_KICK_FAULT = 0x8000
};


/////////////////////////////////////////
// monitor protocol version 1
// game log format versin 1 & version 2

const double SHOWINFO_SCALE = 16.0; //!< scaling parameter
const int REC_OLD_VERSION = 1; //!< version number of rcg v1
const int REC_VERSION_2 = 2; //!< recorded value of rcg v2

//! object data
struct pos_t {
    short enable; //!< determines if this object is in field.
    short side;   //!< side type of this object
    short unum;   //!< uniform number
    short angle;  //!< player's body angle. This is degree data.
    short x;      //!< x-coordinate
    short y;      //!< y-coordinate
};

//! team data
struct team_t {
    char name[16]; //!< team name string
    short score;   //!< score
};

//! view data
struct showinfo_t {
    char pmode;     //!< playmode id
    team_t team[2]; //!< team date
    pos_t pos[MAX_PLAYER * 2 + 1]; //!< all movable objects
    short time; //!< game time
};

//! message data
struct msginfo_t {
    short board;  //!< message type id
    char message[2048]; //!< message body
};

//! point data in drawinfo_t
struct pointinfo_t {
    short x;
    short y;
    char color[COLOR_NAME_MAX];
};

//! circle data in drawinfo_t
struct circleinfo_t {
    short x;
    short y;
    short r;
    char color[COLOR_NAME_MAX];
};

//! line data in drawinfo_t
struct lineinfo_t {
    short x1;
    short y1;
    short x2;
    short y2;
    char color[COLOR_NAME_MAX];
};

//! drawing data
struct drawinfo_t {
    short mode; //!< data type id
    union {
        pointinfo_t pinfo;
        circleinfo_t cinfo;
        lineinfo_t linfo;
    } object;
};

//! data block for rcg v1/v2
struct dispinfo_t {
    short mode; //!< data type id
    union {
        showinfo_t show;
        msginfo_t msg;
        drawinfo_t draw;
    } body;
};

/////////////////////////////////////////
// monitor protocol version 2
// game log format version 3

//! scaling variable
const double SHOWINFO_SCALE2 = 65536.0;
//! version number
const int REC_VERSION_3 = 3;

//! ball data
struct ball_t {
    long x; //!< pos x scaled by SHOWINFO_SCALE2
    long y; //!< pos y scaled by SHOWINFO_SCALE2
    long deltax; //!< velocity x scaled by SHOWINFO_SCALE2
    long deltay; //!< velocity y scaled by SHOWINFO_SCALE2
};

//! player data
struct player_t {
    short mode; //!< status flag
    short type; //!< player type id
    long x; //!< scaled pos x
    long y; //!< scaled pos y
    long deltax; //!< scaled velocity x
    long deltay; //!< scaled velocity y
    long body_angle; //!< scaled body angle. radian data
    long head_angle; //!< scaled heading angle relative to body. radian data
    long view_width; //!< scaled view width. radian data
    short view_quality; //!< view quality id
    long stamina; //!< scaled stamina value.
    long effort; //!< scaled effort value
    long recovery; //!< scaled recover value
    short kick_count; //!< executed kick command count
    short dash_count; //!< executed dash command count
    short turn_count; //!< executed turn command count
    short say_count; //!< executed say command count
    short turn_neck_count; //!< executed turn_neck command count
    short catch_count; //!< executed catch command count
    short move_count; //!< executed move command count
    short change_view_count; //!< executed change_view command count
};

//! view data
struct showinfo_t2 {
    char pmode; //!< playmode id
    team_t team[2]; //!< team info
    ball_t ball; //!< ball info
    player_t pos[MAX_PLAYER * 2]; //!< all player info
    short time; //!< game time
};

/*!
  \brief view data.

  rcg v3 includes only this as view data.
*/
struct short_showinfo_t2 {
    ball_t ball; //!< ball info
    player_t pos[MAX_PLAYER * 2]; //!< player info
    short time; //!< game time
};

//! player type parametors
struct player_type_t {
    short id; //!< type id
    long player_speed_max; //!< max speed
    long stamina_inc_max; //!< max stamina increment value
    long player_decay; //!< speed decay
    long inertia_moment; //!< inertia moment
    long dash_power_rate; //!< dash power rate
    long player_size; //!< body radius
    long kickable_margin; //!< kickable margin
    long kick_rand; //!< random factor for kick
    long extra_stamina; //!< extra stamina value when stamina is 0.
    long effort_max; //!< max(initial) effort value
    long effort_min; //!< min effort value

    long sparelong1;
    long sparelong2;
    long sparelong3;
    long sparelong4;
    long sparelong5;
    long sparelong6;
    long sparelong7;
    long sparelong8;
    long sparelong9;
    long sparelong10;
};

//! server parametors
struct server_params_t {
    long goal_width; //!< goal width
    long inertia_moment; //!< related to player's turn action
    long player_size; //!< normal player size
    long player_decay; //!< normal player speed decay
    long player_rand; //!< noize added to normal player movement
    long player_weight; //!< normal player weight
    long player_speed_max; //!< normal player speed max
    long player_accel_max; //!< normal player accel max
    long stamina_max; //!< normal player stamina max
    long stamina_inc; //!< normal player stamina inc
    long recover_init; //!< normal player recovery init
    long recover_dec_thr; //!< normal player recovery decriment threshold
    long recover_min; //!< normal player recovery min
    long recover_dec; //!< normal player recovery decriment
    long effort_init; //!< normal player dash effort init
    long effort_dec_thr; //!< normal player dash effort decriment threshold
    long effort_min; //!< normal player dash effrot min
    long effort_dec; //!< normal player dash effort decriment
    long effort_inc_thr; //!< normal player dash effort incriment threshold
    long effort_inc; //!< normal player dash effort incriment
    long kick_rand;  //!< noise added directly to normal player's kick
    short team_actuator_noise; //!< flag whether to use team specific actuator noise
    long player_rand_factor_l; //!< factor to multiple prand for left team
    long player_rand_factor_r; //!< factor to multiple prand for right team
    long kick_rand_factor_l; //!< factor to multiple kick_rand for left team
    long kick_rand_factor_r; //!< factor to multiple kick_rand for right team
    long ball_size; //!< ball size
    long ball_decay; //!< ball speed decay
    long ball_rand; //!< noise added to ball movement
    long ball_weight; //!< ball weight
    long ball_speed_max; //!< ball speed max
    long ball_accel_max; //!< ball acceleration max
    long dash_power_rate; //!< normal player's dash power rate
    long kick_power_rate; //!< normal player's kick power rate
    long kickable_margin; //!< normal player's kickable margin
    long control_radius; //!< control radius
    long control_radius_width; //!< (control radius) - (plyaer size)
    long max_power; //!< max power
    long min_power; //!< min power
    long max_moment; //!< max moment
    long min_moment; //!< min moment
    long max_neck_moment; //!< max neck moment
    long min_neck_moment; //!< min neck moment
    long max_neck_angle; //!< max neck angle
    long min_neck_angle; //!< min neck angle
    long visible_angle; //!< visible angle
    long visible_distance; //!< visible distance
    long wind_dir; //!< wind direction
    long wind_force; //!< wind force
    long wind_ang; //!< wind angle for rand
    long wind_rand; //!< wind noise for force
    long kickable_area; //!< kickable_area
    long catch_area_l; //!< goalie catchable area length
    long catch_area_w; //!< goalie catchable area width
    long catch_probability; //!< goalie catchable possibility
    short goalie_max_moves; //!< goalie max moves after a catch
    long corner_kick_margin; //!< corner kick margin
    long offside_active_area; //!< offside active area size
    short wind_none; //!< wind factor is none
    short use_wind_random; //!< wind factor is random
    short coach_say_count_max; //!< max count of coach's freeform say
    short coach_say_msg_size; //!< max length of coach's freeform say message
    short clang_win_size;
    short clang_define_win;
    short clang_meta_win;
    short clang_advice_win;
    short clang_info_win;
    short clang_mess_delay; //!< CLang advise is delayed this cycle
    short clang_mess_per_cycle; //!< online coach can send CLang advise per this cycle
    short half_time; //!<  half time cycle
    short simulator_step; //!< simulator step interval msec
    short send_step; //!< udp send step interval msec
    short recv_step; //!< udp recv step interval msec
    short sense_body_step; //!< sense_body interval step msec
    short lcm_step; //!< lcm of all the above steps msec
    short player_say_msg_size; //!< string size of say message
    short player_hear_max; //!< player hear_capacity_max
    short player_hear_inc; //!< player hear_capacity_inc
    short player_hear_decay; //!< player hear_capacity_decay
    short catch_ban_cycle; //!< goalie catch ban cycle
    short slow_down_factor; //!< factor to slow down simulator and send intervals
    short use_offside; //!< flag for using off side rule
    short kickoff_offside; //!< flag for permit kick off offside
    long offside_kick_margin; //!< offside kick margin
    long audio_cut_dist; //!< audio cut off distance
    long dist_quantize_step; //!< quantize step of distance
    long landmark_dist_quantize_step; //!< quantize step of distance for landmark
    long dir_quantize_step; //!< quantize step of direction
    long dist_quantize_step_l; //!< team right quantize step of distance
    long dist_quantize_step_r; //!< team left quantize step of distance
    long landmark_dist_quantize_step_l; //!< team right quantize step of distance for landmark
    long landmark_dist_quantize_step_r; //!< team left quantize step of distance for landmark
    long dir_quantize_step_l; //!< team left quantize step of direction
    long dir_quantize_step_r; //!< team right quantize step of direction
    short coach_mode; //!< coach mode
    short coach_with_referee_mode; //!< coach with referee mode
    short use_old_coach_hear; //!< old format for hear command (coach)
    short online_coach_look_step; //!< online coach's look interval step
    long slowness_on_top_for_left_team;  //!< flag for left team dash power rate reduction on field top area
    long slowness_on_top_for_right_team; //!< flag for right team dash power rate reduction on field top area

    long ka_length; //!< keep away region length
    long ka_width;  //!< keep away region width
    long sparelong5;
    long sparelong6;
    long sparelong7;
    long sparelong8;
    long sparelong9;
    long sparelong10;

    short start_goal_l; //!< initial left team score
    short start_goal_r; //!< initial right team score
    short fullstate_l; //!< flag for left team's fullstate info receive
    short fullstate_r; //!< flag for right team's fullstate info receive
    short drop_ball_time; //!< automatically drop_ball waiting cycle
    short synch_mode; //!< flag for using synchronized mode
    short synch_offset; //!< wheh sync_mode, server decide the message sent to clients before this number[ms]
    short synch_micro_sleep; //!< the number of microseconds to sleep while waiting for players */
    short point_to_ban; //!< player continue to point to the same point at least during this cycle
    short point_to_duration; //!< point_to command has effectiveness during this cycle

    /*
      short pen_before_setup_wait;
      short pen_setup_wait;
      short pen_ready_wait;
      short pen_taken_wait;
      short pen_nr_kicks;
      short pen_max_extra_kicks;
      long pen_dist_x;
      short pen_random_winner;
      long pen_max_goalie_dist_x;
      short pen_allow_mult_kicks;
    */
};

//! heterogenious player trade-off parametors
struct player_params_t {
    short player_types; //!< the number of player types
    short substitute_max; //!< max player substitution
    short pt_max; //!< max each player type on field
    long player_speed_max_delta_min;
    long player_speed_max_delta_max;
    long stamina_inc_max_delta_factor;
    long player_decay_delta_min;
    long player_decay_delta_max;
    long inertia_moment_delta_factor;
    long dash_power_rate_delta_min;
    long dash_power_rate_delta_max;
    long player_size_delta_factor;
    long kickable_margin_delta_min;
    long kickable_margin_delta_max;
    long kick_rand_delta_factor;
    long extra_stamina_delta_min;
    long extra_stamina_delta_max;
    long effort_max_delta_factor;
    long effort_min_delta_factor;
    long random_seed;
    long new_dash_power_rate_delta_min;
    long new_dash_power_rate_delta_max;
    long new_stamina_inc_max_delta_factor;

    long sparelong5;
    long sparelong6;
    long sparelong7;
    long sparelong8;
    long sparelong9;
    long sparelong10;

    short spareshort1;
    short spareshort2;
    short spareshort3;
    short spareshort4;
    short spareshort5;
    short spareshort6;
    short spareshort7;
    short spareshort8;
    short spareshort9;
    short spareshort10;
};

/*!
  \brief data block for monitor v2 protocol.

  This data format is used only for monitor protocol and not used in rcg.
*/
struct dispinfo_t2 {
    short mode; //!< data type id
    union {
        showinfo_t2 show; //!< view data
        msginfo_t msg; //!< message data
        player_type_t ptinfo; //!< player type data
        server_params_t sparams; //!< server param data
        player_params_t pparams; //!< payer trade-off param data
    } body;
};

//! default rcg version
const int DEFAULT_LOG_VERSION = REC_VERSION_3;

} // end namespace
} // end namespace

#endif
