// -*-c++-*-

/*!
  \file resultprinter.cpp
  \brief game result printer program source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#include <cstring>
#include <iostream>
#include <fstream>
#include <string>

#include <rcsc/gz/gzfstream.h>
#include <rcsc/rcg.h>
#include <rcsc/rcg/parser_v1.h>
#include <rcsc/rcg/parser_v2.h>
#include <rcsc/rcg/parser_v3.h>


class ResultPrinter
    : public rcsc::rcg::Handler {
private:
    std::string M_left_team_name; //!< left teamname string
    std::string M_right_team_name; //!< right teamname string

    int M_left_score; //!< left team score
    int M_right_score; //!< right team score
public:

    ResultPrinter();

    virtual
    bool handleDispInfo( const rcsc::rcg::dispinfo_t & )
      {
          return true;
      }
    virtual
    bool handleShowInfo( const rcsc::rcg::showinfo_t & )
      {
          return true;
      }
    virtual
    bool handleShortShowInfo2( const rcsc::rcg::short_showinfo_t2 & )
      {
          return true;
      }
    virtual
    bool handleMsgInfo( short, const std::string & )
      {
          return true;
      }

    virtual
    bool handlePlayMode( char playmode )
      {
          return true;
      }

    virtual
    bool handlePlayerType( const rcsc::rcg::player_type_t & )
      {
          return true;
      }
    virtual
    bool handleServerParam( const rcsc::rcg::server_params_t & )
      {
          return true;
      }
    virtual
    bool handlePlayerParam( const rcsc::rcg::player_params_t & )
      {
          return true;
      }

    virtual
    bool handleTeamInfo( const rcsc::rcg::team_t & team_left,
                         const rcsc::rcg::team_t & team_right );

    //! handle the end of file
    virtual
    bool handleEOF();

};


/*-------------------------------------------------------------------*/
/*!
  constructor
*/
ResultPrinter::ResultPrinter()
    : M_left_team_name( "" )
    , M_right_team_name( "" )
    , M_left_score( 0 )
    , M_right_score( 0 )
{

}

/*-------------------------------------------------------------------*/
/*!
  read team name & score.
*/
bool
ResultPrinter::handleTeamInfo( const rcsc::rcg::team_t & team_left,
                               const rcsc::rcg::team_t & team_right )
{
    if ( M_left_team_name.empty() )
    {
        char buf[18];
        std::memset( buf, '\0', 18 );
        std::strncpy( buf, team_left.name, 16 );
        M_left_team_name = buf;
    }
    if ( M_right_team_name.empty() )
    {
        char buf[18];
        std::memset( buf, '\0', 18 );
        std::strncpy( buf, team_right.name, 16 );
        M_right_team_name = buf;
    }

    M_left_score = rcsc::rcg::nstohi( team_left.score );
    M_right_score = rcsc::rcg::nstohi( team_right.score );

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  print result
  "<TeamNameL> <TeamNameR> <ScoreL> <ScoreR>
*/
bool
ResultPrinter::handleEOF()
{
    if ( M_left_team_name.empty() )
    {
        M_left_team_name = "null";
    }
    if ( M_right_team_name.empty() )
    {
        M_right_team_name = "null";
    }

    std::cout << M_left_team_name << " " << M_right_team_name << " "
              << M_left_score << " " << M_right_score
              << std::endl;
    return true;
}


////////////////////////////////////////////////////////////////////////

void
usage()
{
    std::cerr << "Usage: rclmresultprinter <RcgFile>[.gz]"
              << std::endl;
}


////////////////////////////////////////////////////////////////////////

int
main( int argc, char** argv )
{
    if ( argc != 2 )
    {
        usage();
        return 1;
    }

    rcsc::gzifstream fin( argv[1] );

    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open file : " << argv[1] << std::endl;
        return 1;
    }

    rcsc::rcg::ParserPtr parser = rcsc::rcg::make_parser( fin );

    if ( ! parser )
    {
        std::cerr << "Failed to create rcg parser." << std::endl;
        return 1;
    }

    // create rcg handler instance
    ResultPrinter printer;

    parser->parse( fin, printer );

    return 0;
}
