// -*-c++-*-

/*!
  \file global_world_model.h
  \brief noiseless world model class Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_COACH_GLOBAL_WORLD_MODEL_H
#define RCSC_COACH_GLOBAL_WORLD_MODEL_H

#include <string>
#include <vector>

#include <rcsc/types.h>
#include <rcsc/game_time.h>
#include <rcsc/game_mode.h>
#include <rcsc/coach/global_object.h>

namespace rcsc {

class GlobalVisualSensor;

/*!
  \class GlobalWorldModel
  \brief world world for coach
 */
class GlobalWorldModel {
private:
    //! last updated time
    GameTime M_time;

    SideID M_our_side; //!< if trainer, set NEUTRAL

    std::string M_team_name_left; //!< left team name string
    std::string M_team_name_right; //!< right teamname string

    //! last updated playmode
    GameMode M_game_mode;


    // Objects

    //! seen ball info
    GlobalBallObject M_ball;

    //! the container of player's instance
    GlobalPlayerList M_players;

    //! the container of player's pointer for the left team
    GlobalPlayerPtrVector M_players_left;

    //! the container of player's pointer for the right team
    GlobalPlayerPtrVector M_players_right;


    // player type management

    //! number of player change
    int M_change_count_left;
    //! number of player change
    int M_change_count_right;

    //! teammate player type Id
    HeteroID M_player_types_left[11];
    //! opponent player type Id
    HeteroID M_player_types_right[11];


    // Coach Language management

    //! the last playon start time
    long M_last_playon_start;

    //! the number of allowd freeform message
    int M_freeform_allowed_count;

    //! the number of send out freeform message
    int M_freeform_send_count;


    //! not used
    GlobalWorldModel( const GlobalWorldModel & );
    //! not used
    GlobalWorldModel & operator=( const GlobalWorldModel & );
public:
    /*!
      \brief init member variables
     */
    GlobalWorldModel();

    /*!
      \brief init with team side info);
      \param side side character
     */
    void init( const SideID side );

    /*!
      \brief init max freeform message count.

      This method is called just after server_param message.
     */
    void initFreeformCount();

    /*!
      \brief set team name
      \param side team side
      \param name team name string
     */
    void setTeamName( const SideID side,
                      const std::string & name );

    /*!
      \brief update teammate player type Id
     */
    void setPlayerType( const SideID side,
                        const int unum,
                        const HeteroID type );

    /*!
      \brief update playmode using heard referee info
      \param game_mode analyzed referee info
      \param current current game time
     */
    void updatePlayMode( const GameMode & game_mode,
                         const GameTime & current );

    /*!
      \brief update status using analyzed visual info
      \param see_global analyzed visual info
      \param current current game time
     */
    void updateAfterSeeGlobal( const GlobalVisualSensor & see_global,
                               const GameTime & current );

    /*!
      \brief increment freeform send count
     */
    void incFreeformSendCount()
      {
          ++M_freeform_send_count;
      }

    /*!
      \brief get our team side
      \return side Id
     */
    SideID ourSide() const
      {
          return M_our_side;
      }

    /*!
      \brief get left team name
      \return team name string
     */
    const
    std::string & teamNameLeft() const
      {
          return M_team_name_left;
      }

    /*!
      \brief get right team name
      \return team name string
     */
    const
    std::string & teamNameRight() const
      {
          return M_team_name_right;
      }

    /*!
      \brief get last updated time
      \return const reference to the game time object
     */
    const
    GameTime & time() const
      {
          return M_time;
      }

    /*!
      \brief get latest playmode info
      \return const reference to the GameMode object
     */
    const
    GameMode & gameMode() const
      {
          return M_game_mode;
      }

    /*!
      \brief get teammate player type Id
      \param side side ID of the target player
      \param unum uniform number of the target player
      \return player type Id
     */
    HeteroID getPlayerType( const SideID side,
                            const int unum ) const;

    /*!
      \brief get the latest ball data
      \return const reference to the ball data
     */
    const
    GlobalBallObject & ball() const
      {
          return M_ball;
      }

    /*!
      \brief get the current ball position status
      \return ball status type
     */
    BallStatus getBallStatus() const;

    /*!
      \brief get all players
      \return const reference to the data container
     */
    const
    GlobalPlayerList & players() const
      {
          return M_players;
      }

    /*!
      \brief get left players' pointer
      \return const reference to the data container
     */
    const
    GlobalPlayerPtrVector & playersLeft() const
      {
          return M_players_left;
      }

    /*!
      \brief get right players' pointer
      \return const reference to the data container
     */
    const
    GlobalPlayerPtrVector & playersRight() const
      {
          return M_players_right;
      }

    /*!
      \brief get the last playon start cycle
      \return cycle value
     */
    const
    long & lastPlayonStart() const
      {
          return M_last_playon_start;
      }

    /*!
      \brief get the allowed freeform message count
      \return the number of the allowd message count
     */
    int freeformAllowedCount() const
      {
          return M_freeform_allowed_count;
      }

    /*!
      \brief get the freeform message send count
      \return the number of the message send count
     */
    int freeformSendCount() const
      {
          return M_freeform_send_count;
      }

    /*!
      \brief check if coach can sent the freeform message now.
      \return the check result
     */
    bool canSendFreeform() const;


    /*!
      \brief check if kickable player exists or not
      \return true if kickable plaeyr exists
     */
    bool existKickablePlayer() const;

    /*!
      \brief get player pointer nearest to the specified point
      \param point target point
      \return const pointer to the player object
     */
    const
    GlobalPlayerObject * getPlayerNearestTo( const Vector2D & point ) const;

    /*!
      \brief put all data to the output stream
      \param os reference to the output stream
      \return reference to the output stream
     */
    std::ostream & print( std::ostream & os ) const;

};

}

#endif
