// -*-c++-*-

/*!
  \file param_map.cpp
  \brief parameter registry map Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "param_map.h"

namespace rcsc {

namespace {

/*-------------------------------------------------------------------*/
/*!

*/
bool
is_true( const std::string & value_str )
{
    return ( value_str == "true"
             || value_str == "on"
             || value_str == "1"
             || value_str == "yes" );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
is_false( const std::string & value_str )
{
    return ( value_str == "false"
             || value_str == "off"
             || value_str == "0"
             || value_str == "no" );
}

}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ParamGeneric< bool >::analyze( const std::string & value_str )
{
    if ( value_str.empty() )
    {
        return false;
    }

    if ( is_true( value_str ) )
    {
        *M_value_ptr = true;
    }
    else if ( is_false( value_str ) )
    {
        *M_value_ptr = false;
    }
    else
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***ERROR*** Unexpected value string: type bool. ["
                  << value_str << "]"
                  << std::endl;
        return false;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
ParamGeneric< bool >::printValue( std::ostream & os ) const
{
    return os << std::boolalpha << *M_value_ptr;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ParamSwitch::analyze( const std::string & value_str )
{
    if ( value_str.empty()
         || is_true( value_str ) )
    {
        *M_value_ptr = true;
    }
    else if ( is_false( value_str ) )
    {
        *M_value_ptr = false;
    }
    else
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***ERROR*** Unexpected value string: type switch. ["
                  << value_str << "]"
                  << std::endl;
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
ParamSwitch::printValue( std::ostream & os ) const
{
    os << ( *M_value_ptr ? "on" : "off" );
    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
ParamMap::Registrar &
ParamMap::Registrar::operator()( const BoolSwitch & value,
                                 const std::string & long_name,
                                 const std::string & short_name,
                                 const char * description )
{
    if ( value.ptr_ == static_cast< bool * >( 0 ) )
    {
        std::cerr << "***ERROR*** detected null pointer for the option "
                  << long_name << std::endl;
        return *this;
    }

    ParamPtr ptr ( new ParamSwitch( value.ptr_,
                                    long_name,
                                    short_name,
                                    description ) );
    M_param_map.add( ptr );

    return *this;
}

/*-------------------------------------------------------------------*/
/*!

*/
ParamMap &
ParamMap::add( ParamMap & param_map )
{
    if ( this == &param_map )
    {
        return *this;
    }

    for ( std::map< std::string, ParamPtr >::iterator it
              = param_map.M_long_name_map.begin();
          it != param_map.M_long_name_map.end();
          ++it )
    {
        if ( M_long_name_map.find( it->first ) != M_long_name_map.end() )
        {
            // already registered same long name
            std::cerr << " ***ERROR*** "
                      << " the option name [" << it->first
                      << "] is already registered by other map."
                      << std::endl;
            continue;
        }

        if ( ! it->second->shortName().empty()
             && getParamPtrShortName( it->second->shortName() ) )
        {
            // already registered same short name
            std::cerr << " ***ERROR*** "
                      << " the option name [" << it->first
                      << "] is already registered by other map."
                      << std::endl;
            continue;
        }

        M_long_name_map[it->second->longName()] = it->second;

        const std::string & short_name = it->second->shortName();
        for ( std::string::const_iterator c = short_name.begin();
              c != short_name.end();
              ++c )
        {
            M_short_name_map[*c] = it->second;
        }

    }


    return *this;
}

/*-------------------------------------------------------------------*/
/*!

*/
ParamMap::Registrar &
ParamMap::add( ParamPtr param )
{
    if ( ! param )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***ERROR*** ParamMap::add(). "
                  << "detected null ParamPtr."
                  << std::endl;
        return M_registrar;
    }

    if ( param->longName().empty() )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***ERROR*** ParamMap::add(). "
                  << "Empty parameter name! parameter was not registered."
                  << std::endl;
        return M_registrar;
    }

    if ( M_long_name_map.find( param->longName() ) != M_long_name_map.end() )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***ERROR*** the long name " << param->longName()
                  << " has already been used."
                  << std::endl;
        return M_registrar;
    }

    const std::string & short_name = param->shortName();
    for ( std::string::const_iterator c= short_name.begin();
          c != short_name.end();
          ++c )
    {
        if ( M_short_name_map.find( *c ) != M_short_name_map.end() )
        {
            std::cerr << __FILE__ << ':' << __LINE__
                      << " ***ERROR*** the short name " << *c
                      << " has already been used."
                      << std::endl;
            return M_registrar;
        }
    }


    M_long_name_map[param->longName()] = param;

    for ( std::string::const_iterator c = short_name.begin();
          c != short_name.end();
          ++c )
    {
        M_short_name_map[*c] = param;
    }

    return M_registrar;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ParamMap::remove( const std::string & long_name )
{
    ParamPtr ptr = getParamPtr( long_name );

    if ( ! ptr )
    {
        return;
    }

    M_long_name_map.erase( ptr->longName() );

    const std::string & short_name = ptr->shortName();
    for ( std::string::const_iterator c = short_name.begin();
          c != short_name.end();
          ++c )
    {
        M_short_name_map.erase( *c );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
ParamPtr
ParamMap::getParamPtr( const std::string & long_name )
{
    // search long name map
    std::map< std::string, ParamPtr >::iterator it
        = M_long_name_map.find( long_name );

    if ( it != M_long_name_map.end() )
    {
        return it->second;
    }

    // return NULL
    return ParamPtr( static_cast< ParamEntry * >( 0 ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
ParamPtr
ParamMap::getParamPtr( const char short_name )
{
    // search long name map
    std::map< char, ParamPtr >::iterator it
        = M_short_name_map.find( short_name );

    if ( it != M_short_name_map.end() )
    {
        return it->second;
    }

    // return NULL
    return ParamPtr( static_cast< ParamEntry * >( 0 ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
ParamPtr
ParamMap::getParamPtrShortName( const std::string & short_name )
{
    // search short name map
    for ( std::string::const_iterator c = short_name.begin();
          c != short_name.end();
          ++c )
    {
        std::map< char, ParamPtr >::iterator it
            = M_short_name_map.find( *c );

        if ( it != M_short_name_map.end() )
        {
            return it->second;
        }
    }

    // return NULL
    return ParamPtr( static_cast< ParamEntry * >( 0 ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
ParamMap::printHelp( std::ostream & os ) const
{
    const std::string desc_nl = "\n        ";

    if ( ! M_option_name.empty() )
    {
        os << M_option_name << ":\n";
    }

    const std::map< std::string, ParamPtr >::const_iterator end = M_long_name_map.end();
    for ( std::map< std::string, ParamPtr >::const_iterator it = M_long_name_map.begin();
          it != end;
          ++it )
    {
        os << "    --" << it->second->longName();

        if ( ! it->second->shortName().empty() )
        {
            os << ' ';

            char ch = '(';
            const std::string & short_name = it->second->shortName();
            for ( std::string::const_iterator c = short_name.begin();
                  c != short_name.end();
                  ++c )
            {
                os << ch << '-' << *c;
                ch = ',';
            }
            os << " short option)";
        }

        if ( ! it->second->isSwitch() )
        {
            os << " <Value>";
        }

        os << desc_nl << "(DefaultValue=\"";
        it->second->printValue( os ) << "\")";

        if ( ! it->second->description().empty() )
        {
            // format description message
            const std::string & desc = it->second->description();
            std::string::size_type nl_pos = 0;
            for ( std::string::size_type pos = desc.find( ' ' );
                  pos != std::string::npos;
                  pos = desc.find( ' ', pos + 1 ) )
            {
                if ( pos > nl_pos + 72 - 6 )
                {
                    os << desc_nl << desc.substr( nl_pos, pos - nl_pos );
                    nl_pos = pos + 1;
                }
            }
            os << desc_nl << desc.substr( nl_pos );
        }

        os << "\n";
    }
    return os << std::flush;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
ParamMap::printValues( std::ostream & os ) const
{
    const std::map< std::string, ParamPtr >::const_iterator end = M_long_name_map.end();
    for ( std::map< std::string, ParamPtr >::const_iterator it = M_long_name_map.begin();
          it != end;
          ++it )
    {
        os << it->second->longName() << '\t';
        it->second->printValue( os );
        os << '\n';
    }
    return os << std::flush;
}

}
