// -*-c++-*-

/*!
  \file audio_codec.cpp
  \brief audio message encoder/decoder Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <iostream>
#include <algorithm>

#include <rcsc/math_util.h>

#include "audio_codec.h"

/*
  namespace {
  inline
  double
  roundDouble( const double & val,
  const double & step = 0.1 )
  {
  return rint( val / step ) * step;
  }
  }
*/

namespace rcsc {

const double AudioCodec::X_NORM_FACTOR = 57.5;
const double AudioCodec::Y_NORM_FACTOR = 39.0;
const double AudioCodec::SPEED_NORM_FACTOR = 3.0;

// related to POSX_BIT_L5, POSY_BIT_L5, VEL_BIT_L5
const boost::int32_t AudioCodec::POSX_MASK_L5 = 0x000000ff;
const boost::int32_t AudioCodec::POSY_MASK_L5 = 0x0000ff00;
const boost::int32_t AudioCodec::VELX_MASK_L5 = 0x003f0000;
const boost::int32_t AudioCodec::VELY_MASK_L5 = 0x0fc00000;

/*
  long bit_8 = 0x000000ff; // base 8 bit mask
  long bit_7 = 0x0000007f; // base 7 bit mask
  long bit_6 = 0x0000003f; // base 6 bit mask
  long bit_5 = 0x0000001f; // base 5 bit mask
  long bit_4 = 0x0000000f; // base 4 bit mask

  printf("%#010x\n", bit_8); // POSX_MASK_L5
  printf("%#010x\n", (bit_8 << POSX_BIT_L5)); // POSY_MASK_L5
  printf("%#010x\n", (bit_6 << (POSX_BIT_L5 + POSY_BIT_L5))); // VELX_MASK_L5
  printf("%#010x\n", (bit_6 << (POSX_BIT_L5 + POSY_BIT_L5 + VEL_BIT_L5))); // VELY_MASK_L5
*/

const double AudioCodec::COORD_STEP_L2 = 0.2;
const double AudioCodec::SPEED_STEP_L1 = 0.1;


//[-0-9a-zA-Z ().+*/?<>_]
const std::string AudioCodec::CHAR_SET =
"0123456789"
"abcdefghijklmnopqrstuvwxyz"
"ABCDEFGHIJKLMNOPQRSTUVWXYZ"
"().+-*/?<>_";
//" ().+-*/?<>_"; remove space

const int AudioCodec::CHAR_SIZE = static_cast< int >( CHAR_SET.length() );


/*-------------------------------------------------------------------*/
/*!

*/
AudioCodec::AudioCodec()
{
    int count = 0;

    // create int <-> char map

    // [0-9]
    for ( int i = 0; i < 10; i++ )
    {
        char ch = static_cast< char >( '0' + i );
        M_char_to_int_map.insert( std::make_pair( ch, count++ ) );
        M_int_to_char_map.push_back( ch );
    }
    // [a-z]
    for ( int i = 0; i < 26; i++ )
    {
        char ch = static_cast< char >( 'a' + i );
        M_char_to_int_map.insert( std::make_pair( ch, count++ ) );
        M_int_to_char_map.push_back( ch );
    }
    // [A-Z]
    for ( int i = 0; i < 26; i++ )
    {
        char ch = static_cast< char >( 'A' + i );
        M_char_to_int_map.insert( std::make_pair( ch, count++ ) );
        M_int_to_char_map.push_back( ch );
    }

    // [ ().+-*/?<>_]
    //M_char_to_int_map.insert( std::make_pair( ' ', count++ ) );
    //M_int_to_char_map.push_back( ' ' );
    M_char_to_int_map.insert( std::make_pair( '(', count++ ) );
    M_int_to_char_map.push_back( '(' );
    M_char_to_int_map.insert( std::make_pair( ')', count++ ) );
    M_int_to_char_map.push_back( ')' );
    M_char_to_int_map.insert( std::make_pair( '.', count++ ) );
    M_int_to_char_map.push_back( '.' );
    M_char_to_int_map.insert( std::make_pair( '+', count++ ) );
    M_int_to_char_map.push_back( '+' );
    M_char_to_int_map.insert( std::make_pair( '-', count++ ) );
    M_int_to_char_map.push_back( '-' );
    M_char_to_int_map.insert( std::make_pair( '*', count++ ) );
    M_int_to_char_map.push_back( '*' );
    M_char_to_int_map.insert( std::make_pair( '/', count++ ) );
    M_int_to_char_map.push_back( '/' );
    M_char_to_int_map.insert( std::make_pair( '?', count++ ) );
    M_int_to_char_map.push_back( '?' );
    M_char_to_int_map.insert( std::make_pair( '<', count++ ) );
    M_int_to_char_map.push_back( '<' );
    M_char_to_int_map.insert( std::make_pair( '>', count++ ) );
    M_int_to_char_map.push_back( '>' );
    M_char_to_int_map.insert( std::make_pair( '_', count++ ) );
    M_int_to_char_map.push_back( '_' );
}

/*-------------------------------------------------------------------*/
/*!

*/
boost::int32_t
AudioCodec::posToLongL5( const Vector2D & pos,
                         const Vector2D & vel ) const
{
    boost::int32_t result = 0;
    double dval;
    int ival;

    // encode pos x
    //   check range[-POSX_NORMALIZE, POSX_NORMALIZE]
    //   normalize(+POSX_NORMALIZE)(*2.0)[0, 4 * POSX_NORMALIZE]
    //              <---- POSX_BIT_L5 bit
    //     (rounded[step=0.5])
    dval = rint( ( min_max( -X_NORM_FACTOR,
                            pos.x,
                            X_NORM_FACTOR )
                   + X_NORM_FACTOR ) * 2.0 );
    ival = static_cast< int >( dval );

    //cout << "e posx double " << roundDouble(pos.x, 0.5)
    //     << " to int " << ival << endl;

    result |= ival;


    // encode pos y
    //   check range[-Y_NORM_FACTOR, Y_NORM_FACTOR]
    //   normalize(+32.0)(*2.0)[0, 4 * Y_NORM_FACTOR]
    //              <--------- POSY_BIT_L5 bit
    //    (rounded[step=0.5])
    dval = rint( ( min_max( -Y_NORM_FACTOR,
                            pos.y,
                            Y_NORM_FACTOR )
                   + Y_NORM_FACTOR ) * 2.0 );
    ival = static_cast< int >( dval );

    //cout << "e posy double " << roundDouble(pos.y, 0.5)
    //     << " to int " << ival << endl;

    result |= ( ival << POSX_BIT_L5 );


    // encode vel x
    //   check range[-3, 3]
    //   normalize(+3.0)(*10.0)[0, 20 * SPEED_NORM_FACTOR]
    //          <--------- VEL_BIT_L5 bit
    //    (rounded[step=0.1])
    dval = rint( ( min_max( -SPEED_NORM_FACTOR,
                            vel.x,
                            SPEED_NORM_FACTOR )
                   + SPEED_NORM_FACTOR ) * 10.0 );
    ival = static_cast< int >( dval );

    //cout << "e velx double " << roundDouble(vel.x, 0.1)
    //     << " to int " << ival << endl;

    result |= ( ival << ( POSX_BIT_L5 + POSY_BIT_L5 ) );


    // encode vel y
    //   check range[-3, 3]
    //   normalize(+3.0)(*10.0)[0, 20 * SPEED_NORM_FACTOR]
    //           <--------- VEL_BIT_L5 bit
    //    (rounded[step=0.1])
    dval = rint( ( min_max( -SPEED_NORM_FACTOR, vel.y, SPEED_NORM_FACTOR )
                   + SPEED_NORM_FACTOR ) * 10.0 );
    ival = static_cast< int >( dval );

    //cout << "e vely double " << roundDouble(vel.y, 0.1)
    //     << " to int " << ival << endl;

    result |= ( ival << ( POSX_BIT_L5 + POSY_BIT_L5 + VEL_BIT_L5 ) );


    return result;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
AudioCodec::longToPosL5( const boost::int32_t & val,
                         Vector2D * pos,
                         Vector2D * vel ) const
{
    int ival;

    // decode pos x
    //cout << std::dec;
    ival = static_cast< int >( val & POSX_MASK_L5 );
    //cout << "d posx int " << ival << endl;
    pos->x
        = static_cast< double >( ival ) * 0.5
        - X_NORM_FACTOR;

    ival = static_cast< int >( val & POSY_MASK_L5 );
    //cout << "d posy int " << ( ival >> POSX_BIT_L5 ) << endl;
    pos->y
        = static_cast< double >( ival >> POSX_BIT_L5 ) * 0.5
        - Y_NORM_FACTOR;

    ival = static_cast< int >( val & VELX_MASK_L5 );
    //cout << "d velx int " << ( ival >> ( POSX_BIT_L5 + POSY_BIT_L5 ) ) << endl;
    vel->x
        = static_cast< double >( ival >> ( POSX_BIT_L5 + POSY_BIT_L5 ) ) * 0.1
        - SPEED_NORM_FACTOR;

    ival = static_cast< int >( val & VELY_MASK_L5 );
    //cout << "d vely int " << ( ival >> ( POSX_BIT_L5 + POSY_BIT_L5 + VEL_BIT_L5 ) ) << endl;
    vel->y
        = static_cast< double >( ival >> ( POSX_BIT_L5 + POSY_BIT_L5 + VEL_BIT_L5 ) ) * 0.1
        - SPEED_NORM_FACTOR;

}

/*-------------------------------------------------------------------*/
/*!

*/
bool
AudioCodec::encodeL5( const Vector2D & pos,
                      const Vector2D & vel,
                      std::string & to ) const
{
    //cout << "AudioCodec::encode" << endl;

    // erase
    to.erase();

    // convert to long

    const boost::int32_t ival = posToLongL5( pos, vel );

    //cout << "  original long "<< std::dec << ival << endl;

    std::vector< int > talk_values;
    boost::int32_t divided = ival;

    for ( int i = 0; i < 4; i++ )
    {
        talk_values.push_back( divided % CHAR_SIZE );
        divided /= CHAR_SIZE;
    }
    if ( divided > CHAR_SIZE )
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***ERROR*** AudioCodec::encodeL5."
                  << " over flow. pos=" << pos
                  << " vel=" << vel
                  << std::endl;
        return false;
    }
    talk_values.push_back( divided % CHAR_SIZE );

    // reverse copy
    for ( std::vector< int >::reverse_iterator it = talk_values.rbegin();
          it != talk_values.rend();
          ++it )
    {
        char ch = intToCharMap()[*it];
        //cout << " int [" << *it << "] to char [" << ch << "]" << endl;
        to += ch;
    }
    //cout << " encode ----> [" << to << "]" << endl;
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
AudioCodec::decodeL5( const std::string & from,
                      Vector2D * pos,
                      Vector2D * vel ) const
{
    if ( from.length() != 5 )
    {
        return false;
    }

    //cout << "AudioCodec::decode" << endl;
    boost::int32_t read_val = 0;
    bool success = true;
    const char *msg = from.c_str();
    int digit_count = std::strlen( msg ) - 1;
    while ( *msg != '\0' && digit_count >= 0 )
    {
        CharToIntCont::const_iterator it;
        //if ( *msg == '0' )
        //{
        //    cout << "decode ZERO" << endl;
        //}
        it = charToIntMap().find( *msg );
        if ( it == charToIntMap().end() )
        {
            std::cerr << __FILE__ << ": " << __LINE__
                      << " ***ERROR*** AudioCodec::encodeL5."
                      << " Unexpected communication message. ["
                      << from << "]"
                      << std::endl;
            success = false;
            break;
        }
        boost::int32_t add_val
            = it->second
            * static_cast< boost::int32_t >
            ( std::pow( static_cast< double >( CHAR_SIZE ),
                        static_cast< double >( digit_count ) ) );
        //cout << " char [" << *msg
        //     << "] read val " << it->second
        //     << "  add val " << add_val << endl;
        read_val += add_val;
        digit_count -= 1;
        msg++;
    }
    //cout << "  read long " << read_val << endl;
    if ( success )
    {
        longToPosL5( read_val, pos, vel );
        return true;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
AudioCodec::encodeCoordL2( const double & xy,
                           const double & norm_factor,
                           std::string & to ) const
{
    // normalize value
    double tmp = min_max( -norm_factor, xy , norm_factor );
    tmp += norm_factor;

    // round
    tmp /= COORD_STEP_L2;

    int ival = static_cast< int >( rint( tmp ) );

    int i1 = ival % CHAR_SIZE;
    ival /= CHAR_SIZE;
    int i2 = ival % CHAR_SIZE;
    //std::cout << " posx -> " << ix1 << " " << ix2 << std::endl;


    try
    {
        to += intToCharMap().at( i1 );
        to += intToCharMap().at( i2 );
    }
    catch ( ... )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::encodeCoordL2."
                  << " Exception caught! "
                  << std::endl;
        return false;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
double
AudioCodec::decodeCoordL2( const char ch1,
                           const char ch2,
                           const double & norm_factor ) const
{
    CharToIntCont::const_iterator it;
    const CharToIntCont::const_iterator end = charToIntMap().end();

    it = charToIntMap().find( ch1 );
    if ( it == end )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::decodeCoordL2."
                  << " Unsupported character [" << ch1 << "]"
                  << std::endl;
        return HUGE_VAL;
    }
    int i1 = it->second;

    it = charToIntMap().find( ch2 );
    if ( it == end )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::decodeCoordL2."
                  << " Unsupported character [" << ch2 << "]"
                  << std::endl;
        return HUGE_VAL;
    }
    int i2 = it->second;

    return
        (
         static_cast< double >( i1 + i2 * CHAR_SIZE ) * COORD_STEP_L2
         - norm_factor
         );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
AudioCodec::encodeSpeedL1( const double & val,
                           std::string & to ) const
{
    // normalize value
    double tmp = min_max( -SPEED_NORM_FACTOR, val , SPEED_NORM_FACTOR );
    tmp += SPEED_NORM_FACTOR;

    // round
    tmp /= SPEED_STEP_L1;

    int ival = static_cast< int >( rint( tmp ) );

    try
    {
        to += intToCharMap().at( ival );
    }
    catch ( ... )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::encodeSpeedL1."
                  << " Exception caught! Failed to encode"
                  << std::endl;
        return false;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
double
AudioCodec::decodeSpeedL1( const char ch ) const
{
    CharToIntCont::const_iterator it = charToIntMap().find( ch );
    if ( it == charToIntMap().end() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::decodeSpeedL1."
                  << " Unsupported character [" << ch << "]"
                  << std::endl;
        return HUGE_VAL;
    }

    return
        ( static_cast< double >( it->second ) * SPEED_STEP_L1
          - SPEED_NORM_FACTOR
          );
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
AudioCodec::encodeL6( const Vector2D & pos,
                      const Vector2D & vel ) const
{
    std::string msg( "" );
    msg.reserve( 6 );

    if ( encodeCoordL2( pos.x, X_NORM_FACTOR, msg )
         && encodeCoordL2( pos.y, Y_NORM_FACTOR, msg )
         && encodeSpeedL1( vel.x, msg )
         && encodeSpeedL1( vel.y, msg )
         )
    {
        return msg;
    }

    std::cerr << __FILE__ << ": " << __LINE__
              << " ***ERROR*** AudioCodec::encodeL6."
              << " Invalid value! pos = " << pos << " vel = " << vel
              << std::endl;

    return std::string( "" );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
AudioCodec::decodeL6( const std::string & from,
                      Vector2D * pos,
                      Vector2D * vel ) const
{
    if ( from.length() != 6 )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** AudioCodec::decodeL6."
                  << " Invalid string length! " << from.length()
                  << std::endl;
        return false;
    }

    if ( pos )
    {
        pos->x = decodeCoordL2( from[0], from[1], X_NORM_FACTOR );
        if ( pos->x == HUGE_VAL ) return false;
        pos->y = decodeCoordL2( from[2], from[3], Y_NORM_FACTOR );
        if ( pos->y == HUGE_VAL ) return false;
    }
    if ( vel )
    {
        vel->x = decodeSpeedL1( from[4] );
        if ( vel->x == HUGE_VAL ) return false;
        vel->y = decodeSpeedL1( from[5] );
        if ( vel->y == HUGE_VAL ) return false;
    }

    return true;
}

}
