// -*-c++-*-

/*!
  \file audio_codec.h
  \brief audio message encoder/decoder Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_AUDIO_CODEC_H
#define RCSC_PLAYER_AUDIO_CODEC_H

#include <cmath>
#include <string>
#include <vector>
#include <map>
#include <boost/cstdint.hpp>

#include <rcsc/geom/vector_2d.h>

namespace rcsc {

/*!
  \class AudioCodec
  \brief communication message encoder/decorder
*/
class AudioCodec {
public:
    typedef std::map< char, int > CharToIntCont;
    typedef std::vector< char > IntToCharCont;

    enum {
        POSX_BIT_L5 = 8, //!< used by posToLongL5
        POSY_BIT_L5 = 8, //!< used by posToLongL5
        VEL_BIT_L5 = 6, //!< used by posToLongL5
    };

    //! x normalize factor (field length) to limit inputed x
    static const double X_NORM_FACTOR;
    //! y normalze factor (field width) to limit inputed y
    static const double Y_NORM_FACTOR;
    //! speed normalize factor to limit inputed speed range
    static const double SPEED_NORM_FACTOR;

    static const boost::int32_t POSX_MASK_L5; //!< used by longToPosL5
    static const boost::int32_t POSY_MASK_L5; //!< used by longToPosL5
    static const boost::int32_t VELX_MASK_L5; //!< used by longToPosL5
    static const boost::int32_t VELY_MASK_L5; //!< used by longToPosL5

    static const double COORD_STEP_L2; //!< used by encodeCoordL2/decodeCoordL2
    static const double SPEED_STEP_L1; //!< used by encodeSpeedL1/decodeSpeedL1

private:

    //! map to cnvert character to integer. key: char, value int
    CharToIntCont M_char_to_int_map;

    //! map to cnvert integer to character. vector of char
    IntToCharCont M_int_to_char_map;

public:

    static const std::string CHAR_SET; //!< available character set
    static const int CHAR_SIZE; //!< size of CHAR_SET

    /*!
      \brief create convert map
    */
    AudioCodec();

private:
    /*!
      \brief encode position and velocity to the 32bits integer
      \param pos position to be converted
      \param vel velocity to be converted
      \return converted integer
    */
    boost::int32_t posToLongL5( const Vector2D & pos,
                                const Vector2D & vel ) const;

    /*!
      \brief decode 32bits integer to position and velocity
      \param val 32bits integer value to be analyzed
      \param pos variable pointer to store the converted position
      \param vel variable pointer to store the converted velocity
    */
    void longToPosL5( const boost::int32_t & val,
                      Vector2D * pos,
                      Vector2D * vel ) const;

public:

    /*!
      \brief get character to interger map object
      \return const reference to the map object
    */
    const
    CharToIntCont & charToIntMap() const
      {
          return M_char_to_int_map;
      }

    /*!
      \brief get integer to character map object
      \return const reference to the map object
    */
    const
    IntToCharCont & intToCharMap() const
      {
          return M_int_to_char_map;
      }
    /*
      \brief encode position and velocity to  5 characters.
      \param pos position value to be encoded
      \param vel velocity value to be encoded
      \param to reference to the variable to store the encoded result
      \return true if successfully encoded.

      The length of result string must be 5.
    */
    bool encodeL5( const Vector2D & pos,
                   const Vector2D & vel,
                   std::string & to ) const;

    /*!
      \brief decode 5 characters to position and velocity
      \param from string to be decoded
      \param pos variable pointer to store the decoded position value
      \param vel variable pointer to store the decoded velocity value
      \return true if successfully decoded

      The length of 'from' must be 5.
    */
    bool decodeL5( const std::string & from,
                   Vector2D * pos,
                   Vector2D * vel ) const;

    /*!
      \brief encode coordinate value( x or y ) to 2 characters.
      \param xy coordinate value to be encoded, X or Y.
      This value should be within [-norm_factor, norm_factor]
      \param norm_factor normalize factor for xy
      \param to reference to the variable to store the encoded result
      \return true if successfully encoded.

      Following condition must be satisfied:
      - (norm_factor*2)/COORD_STEP_L2 < CHARSIZE^2

      norm_factor must be same as the value used by decodeCoordL2()
      norm_factor will be changed depending on the type of target
      coordinate value, i.e. field length or field width.
    */
    bool encodeCoordL2( const double & xy,
                        const double & norm_factor,
                        std::string & to ) const;

    /*!
      \brief decode 2 characters to coordinate value( x or y )
      \param ch1 first character to be decoded
      \param ch2 second character to be decoded
      \param norm_factor normalize factor for coordinate value

      norm_factor must be same as the value used by encodeCoordL2()
      norm_factor will be changed depending on the type of target
      coordinate value, i.e. field length or field width.
    */
    double decodeCoordL2( const char ch1,
                          const char ch2,
                          const double & norm_factor ) const;

    /*!
      \brief encode speed value to 1 character.
      \param xy velocity element to be encoded, X or Y
      \param to reference to the variable to store the encoded result
      \return true if successfully encoded.

      xy value is normalized by SPEED_NORM_FACTOR
    */
    bool encodeSpeedL1( const double & xy,
                        std::string & to ) const;

    /*!
      \brief decode 1 character to speed value
      \param ch character to be decoded
      \return decoded speed value. if failed to decode, HUGE_VAL is returned
    */
    double decodeSpeedL1( const char ch ) const;

    /*!
      \brief encode position and velocity to 6 charecters.
      \param pos position to be encoded
      \param vel velocity to be encoded
      \return encoded string. if encode is failed, return the empty string
    */
    std::string encodeL6( const Vector2D & pos,
                          const Vector2D & vel ) const;

    /*!
      \brief decode 6 characters to position and velocity
      \param from string to be decoded
      \param pos variable pointer to store the decoded position value
      \param vel variable pointer to store the decoded velocity value
      \return true if successfully decoded
    */
    bool decodeL6( const std::string & from,
                   Vector2D * pos,
                   Vector2D * vel ) const;
};

}

#endif
