// -*-c++-*-

/*!
  \file audio_sensor.h
  \brief audio message analyzer Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_PLAYER_AUDIO_SENSOR_H
#define RCSC_PLAYER_AUDIO_SENSOR_H

#include <string>

#include <rcsc/types.h>
#include <rcsc/game_time.h>
#include <rcsc/geom/vector_2d.h>

#include <rcsc/player/audio_codec.h>

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!
  \struct PlayerMessage;
  \brief player message holder
 */
struct PlayerMessage {

    /*!
      \brief communication message type
    */
    enum Type {
        BALL_INFO, // "b "
        PASS_INFO, // "p "
        OPPONENT_GOALIE_INFO, // "g "
        OFFSIDE_LINE_INFO, // "o "
        DEFENSE_LINE_INFO, // "d "
        WAIT_REQUEST, // "w "
        ATTACK_REQUEST, // "a "
        NULL_INFO
    };

    GameTime time_; //!< received time
    SideID side_; //!< sender's side
    int unum_; //!< sender's uniform number
    double dir_; //!< sender's direction
    std::string message_; //!< raw message string
    Type type_;

    PlayerMessage()
        : time_( -1, 0 )
        , side_( NEUTRAL )
        , unum_( 0 )
        , dir_( 0.0 )
        , type_( NULL_INFO )
      {
          message_.reserve( 10 );
      }

};

/*-------------------------------------------------------------------*/
/*!
  \struct FreeformMessage
  \brief freeform message from coach/trainer
 */
struct FreeformMessage {
    GameTime time_; //!< received time
    std::string message_; //!< raw message string. this may be a CLang.

    FreeformMessage()
        : time_( -1, 0 )
      {
          message_.reserve( 8192 );
      }
};


/*-------------------------------------------------------------------*/
/*!
  \class AudioSensor
  \brief processor for players' communication
*/
class AudioSensor {
private:
    //! message encoder/decoder
    AudioCodec M_codec;

    //! last updated time
    GameTime M_time;

    //! last received aural message from other player
    PlayerMessage M_player_message;

    //! last received freeform message from coach;
    FreeformMessage M_freeform_message;

    //! last received aural message from trainer
    FreeformMessage M_trainer_message;


    // ball

    //! heard ball position
    Vector2D M_ball_pos;
    //! heard ball velocity
    Vector2D M_ball_vel;

    // pass

    //! heard pass receiver uniform number
    int M_receiver_number;
    //! heard pass receive position
    Vector2D M_receive_pos;

    // opponent goalie

    //! uniform number of opponent goalie
    int M_goalie_number;
    //! opponent goalie position
    Vector2D M_goalie_pos;

    // offside line
    double M_offside_line_x;

    // defense line
    double M_defense_line_x;


    // player

    // int M_player_number;
    // Vector2D M_player_pos;
    // Vector2D M_player_vel;

public:
    /*!
      \brief init member variables by default value
    */
    AudioSensor();

    /*!
      \brief analyze other player's audio message
      \param msg raw server message
      \param current game time when message is received
    */
    void parsePlayer( const char * msg,
                      const GameTime & current );

    /*!
      \brief analyze message from online coach
      \param msg raw server message
      \param current game time when message is received
     */
    void parseCoach( const char * msg,
                     const GameTime & current );

    /*!
      \brief analyze trainer's audio message
      \param msg raw server message
      \param current game time when message is received
    */
    void parseTrainer( const char * msg,
                       const GameTime & current );

public:

    /*!
      \brief get audio codec
      \return const reference to the AudioCodec instance
    */
    const
    AudioCodec & codec() const
      {
          return M_codec;
      }

    /*!
      \brief encode ball info to the message string to be sent by say command
      \param ball_pos position to be encoded
      \param ball_vel velocity to be encoded
      \param say_buf reference to the variable to store the encoded result
    */
    void encodeBall( const Vector2D & ball_pos,
                     const Vector2D & ball_vel,
                     std::string & say_buf ) const;

    /*!
      \brief encode pass info to the message string to be sent by say command
      \param receiver position to be encoded
      \param receive_pos estimated receive position
      \param say_buf reference to the variable to store the encoded result
    */
    void encodePass( const int receiver,
                     const Vector2D & receive_pos,
                     std::string & say_buf ) const;

    /*!
      \brief encode opponent goalie info to the message string to be sent by say command
      \param goalie_number uniform number of opponent goalie
      \param goalie_pos estimated goalie position
      \param say_buf reference to the variable to store the encoded result
    */
    void encodeGoalie( const int goalie_number,
                       const Vector2D & goalie_pos,
                       std::string & say_buf ) const;

    void encodeOffsideLineX( const double & offside_line_x,
                             std::string & say_buf ) const;

    void encodeDefenseLineX( const double & defense_line_x,
                             std::string & say_buf ) const;

    std::string getWaitMessage() const
      {
          return std::string( "w" );
      }

    std::string getAttackMessage() const
      {
          return std::string( "a" );
      }

    /*!
      \brief get updated time
      \return const referncd to the game time
    */
    const
    GameTime & time() const
      {
          return M_time;
      }

    /*!
      \brief get the last received player message info
      \return const reference to the message object instance
     */
    const
    PlayerMessage & playerMessage() const
      {
          return M_player_message;
      }

    const
    FreeformMessage & freeformMessage() const
      {
          return M_freeform_message;
      }

    /*!
      \brief get the last received trainer message info
      \return const reference to the message object instance
     */
    const
    FreeformMessage & trainerMessage() const
      {
          return M_trainer_message;
      }

    //////////////////////////////////////
    /*!
      \brief get heard ball position
      \return position value
    */
    const
    Vector2D & getBallPos() const
      {
          return M_ball_pos;
      }

    /*!
      \brief get heard ball velocity
      \return velocity value
    */
    const
    Vector2D & getBallVel() const
      {
          return M_ball_vel;
      }

    //////////////////////////////////////

    /*!
      \brief get heard receiver number
      \return receiver's uniform number
    */
    int getReceiverNumber() const
      {
          return M_receiver_number;
      }

    /*!
      \brief get heard receive position
      \return position value
    */
    const
    Vector2D & getReceivePos() const
      {
          return M_receive_pos;
      }

    //////////////////////////////////////
    /*!
      \brief get heard goalie number
      \return goalie's uniform number
    */
    int getGoalieNumber() const
      {
          return M_goalie_number;
      }

    /*!
      \brief get heard goalie position
      \return position value
    */
    const
    Vector2D & getGoaliePos() const
      {
          return M_goalie_pos;
      }

    const
    double & getOffsideLineX() const
      {
          return M_offside_line_x;
      }

    const
    double & getDefenseLineX() const
      {
          return M_defense_line_x;
      }

private:
    /*
      \brief analyze v8 format message from teammate
      \param msg message body contained in hear message
      \return true if successfully analyzed

      message string to be analyzed is quated by double quatation.
      msg == ["<MSG>"]
    */
    bool parseTeammateAudioV8( const char * msg );

    /*!
      \brief analyze pass message from teammate
      \param msg head of message body sent from teammate
      \return true if successfully analyzed
    */
    bool parsePassInfo( const char * msg );

    /*!
      \brief analyze opponent goalie message from teammate
      \param msg head of message body sent from teammate
      \return true if successfully analyzed
    */
    bool parseGoalieInfo( const char * msg );


    bool parseOffsideLineInfo( const char * msg );

    bool parseDefenseLineInfo( const char * msg );

    //bool parseWaitRequest( const char * msg );

    //bool parseAttackRequest( const char * msg );

    /*!
      \brief encode position value to 7 characters
      \param pos position to be encoded
      \param msg reference to the variable to store the encoded result
      \return true if successfully encoded
    */
    bool encodePosToStr7( const Vector2D & pos,
                          std::string & msg ) const;

    /*!
      \brief decode 7 characters to position value
      \param msg head of message to be decoded
      \param pos variable pointer to store the decoded result
      \return true if successfully decoded
    */
    bool decodeStr7ToPos( const char * msg,
                          Vector2D * pos );

};

}

#endif
