// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <rcsc/param/server_param.h>

#include <rcsc/player/logger.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/intercept_table.h>

#include <rcsc/action/basic_actions.h>
#include <rcsc/action/body_go_to_point.h>
#include <rcsc/action/body_intercept.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>
#include <rcsc/action/neck_turn_to_low_conf_teammate.h>

#include "bhv_basic_tackle.h"

#include "bhv_basic_move.h"

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Bhv_BasicMove::execute( rcsc::PlayerAgent * agent )
{
    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        "%s:%d: Bhv_BasicMove"
                        ,__FILE__, __LINE__ );

    // tackle

    if ( Bhv_BasicTackle( 0.9, 80.0 ).execute( agent ) )
    {
        return true;
    }

    const rcsc::WorldModel & wm = agent->world();

    // check ball owner

    int self_min = wm.getInterceptTable()->getSelfReachCycle();
    int mate_min = wm.getInterceptTable()->getTeammateReachCycle();
    int opp_min = wm.getInterceptTable()->getOpponentReachCycle();

    if ( ! wm.existKickableTeammate()
         && ( self_min <= 3
              || ( self_min < mate_min + 3
                   && self_min < opp_min + 4 )
              )
         )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: intercept"
                            ,__FILE__, __LINE__ );
        rcsc::Body_Intercept().execute( agent );
        if ( wm.ball().distFromSelf()
             < rcsc::ServerParam::i().visibleDistance() )
        {
            agent->setNeckAction( new rcsc::Neck_TurnToLowConfTeammate() );;
        }
        else
        {
            agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        }
        return true;
    }

    // go to home position

    const double dash_power = getDashPower( agent, M_home_pos );

    double dist_thr = wm.ball().distFromSelf() * 0.1;
    if ( dist_thr < 1.0 ) dist_thr = 1.0;

    agent->debugClient().addMessage( " BasicMove%.0f", dash_power );
    agent->debugClient().setTarget( M_home_pos );

    if ( ! rcsc::Body_GoToPoint( M_home_pos, dist_thr, dash_power
                                 ).execute( agent ) )
    {
        rcsc::Body_TurnToBall().execute( agent );
    }

    if ( wm.existKickableOpponent()
         && wm.ball().distFromSelf() < 18.0 )
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBall() );
    }
    else
    {
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
double
Bhv_BasicMove::getDashPower( const rcsc::PlayerAgent * agent,
                             const rcsc::Vector2D & target_point )
{
    static bool s_recover_mode = false;

    const rcsc::WorldModel & wm = agent->world();
    /*--------------------------------------------------------*/
    // check recover

    if ( wm.self().stamina() < rcsc::ServerParam::i().staminaMax() * 0.5 )
    {
        s_recover_mode = true;
    }
    else if ( wm.self().stamina() > rcsc::ServerParam::i().staminaMax() * 0.7 )
    {
        s_recover_mode = false;
    }

    /*--------------------------------------------------------*/
    double dash_power = 100.0;
    const double my_inc
        = wm.self().playerType().staminaIncMax()
        * wm.self().recovery();

    if ( wm.getDefenseLineX() > wm.self().pos().x
         && wm.ball().pos().x < wm.getDefenseLineX() + 20.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: correct DF line. keep max power"
                            ,__FILE__, __LINE__ );
        // keep max power
        dash_power = rcsc::ServerParam::i().maxPower();
    }
    else if ( s_recover_mode )
    {
        dash_power = my_inc - 25.0; // preffered recover value
        if ( dash_power < 0.0 ) dash_power = 0.0;

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: recovering"
                            ,__FILE__, __LINE__ );
    }
    // exist kickable teammate
    else if ( wm.existKickableTeammate()
              && wm.ball().distFromSelf() < 20.0 )
    {
        dash_power = std::min( my_inc * 1.1,
                               rcsc::ServerParam::i().maxPower() );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: exist kickable teammate. dash_power= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }
    // in offside area
    else if ( wm.self().pos().x > wm.getOffsideLineX() )
    {
        dash_power = rcsc::ServerParam::i().maxPower();
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: in offside area. dash_power= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }
    // normal
    else
    {
        dash_power = std::min( my_inc * 1.7,
                               rcsc::ServerParam::i().maxPower() );
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: normal mode dash_power= %f"
                            ,__FILE__, __LINE__,
                            dash_power );
    }

    return dash_power;
}
