// -*-c++-*-

/*!
	\file formation.cpp
	\brief formation data classes Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sstream>
#include <algorithm>
#include <boost/random.hpp>

#include "formation.h"

const double FormationParam::PITCH_LENGTH = 105.0 + 10.0;
const double FormationParam::PITCH_WIDTH = 68.0 + 10.0;


/*-------------------------------------------------------------------*/
/*!

*/
FormationParam::FormationParam()
    : M_param( 0.3, 0.9 )
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
FormationParam::randomize()
{
    static boost::mt19937 gen( std::time( 0 ) );
    boost::uniform_real<> dst( -0.5, 0.5 );
    boost::variate_generator< boost::mt19937 &, boost::uniform_real<> >
        rng( gen, dst );

    M_param.randomize( rng );
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
FormationParam::getPosition( const rcsc::Vector2D & ball_pos,
                             const Type type ) const
{
    double msign = 1.0;
    if ( type == MIRROR ) msign =  -1.0;

    if ( type == CENTER && ball_pos.y > 0.0 ) msign = -1.0;


    PosNet::input_array input;

    input[0] = std::max( 0.0,
                         std::min( ball_pos.x / PITCH_LENGTH + 0.5,
                                   1.0 ) );
    input[1] = std::max( 0.0,
                         std::min( (ball_pos.y * msign) / PITCH_WIDTH + 0.5,
                                   1.0 ) );

    PosNet::output_array output;

    M_param.propagate( input, output );
    //std::cerr << "getPosition. raw output = "
    //<< output[0] << " ,  " << output[1]
    //<< std::endl;
    return rcsc::Vector2D( ( output[0] - 0.5 ) * PITCH_LENGTH,
                           ( output[1] - 0.5 ) * PITCH_WIDTH * msign );
}

/*-------------------------------------------------------------------*/
/*!
  Role <RoleNameStr>
*/
bool
FormationParam::readRoleName( std::istream & is )
{
    std::string line_buf;
    if ( ! std::getline( is, line_buf ) )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    std::istringstream istr( line_buf );
    if ( ! istr.good() )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    std::string tag;
    istr >> tag;
    if ( tag != "Role" )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    istr >> M_role_name;
    if ( M_role_name.empty() )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read role name" << std::endl;
        return false;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FormationParam::readParam( std::istream & is )
{
    std::string line_buf;
    if ( ! std::getline( is, line_buf ) )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    std::istringstream istr( line_buf );
    if ( ! istr.good() )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read" << std::endl;
        return false;
    }

    if ( ! M_param.read( istr ) )
    {
        std::cerr  << __FILE__ << ":" << __LINE__
                   << " Failed to read position param" << std::endl;
        return false;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FormationParam::read( std::istream & is )
{
    // read role name
    if ( ! readRoleName( is ) )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Failed to read role name" << std::endl;
        return false;
    }

    if ( ! readParam( is ) )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Failed to read parameters" << std::endl;
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
FormationParam::printRoleName( std::ostream & os ) const
{
    if ( M_role_name.empty() )
    {
        os << "Role Default\n";
    }
    else
    {
        os << "Role " << M_role_name << '\n';
    }
    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
FormationParam::printParam( std::ostream & os ) const
{
    M_param.print( os ) << '\n';
    return os;
}

/*-------------------------------------------------------------------*/
/*!
  Role <role name>
  <bko.x> <bkoy>
  <offense playon>
  <defense playon>
  ...

*/
std::ostream &
FormationParam::print( std::ostream & os ) const
{
    printRoleName( os );
    printParam( os );

    return os << std::flush;
}

///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////

/*-------------------------------------------------------------------*/
/*!

*/
Formation::Formation()
    : M_name( "" )
{
    for ( int i = 0; i < 11; ++i )
    {
        M_mirror_number[i] = -1;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
Formation::Formation( const std::string & name )
    : M_name( name )
{
    for ( int i = 0; i < 11; ++i )
    {
        M_mirror_number[i] = -1;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Formation::clear()
{
    M_name = "";

    for ( int i = 0; i < 11; ++i )
    {
        M_mirror_number[i] = -1;
    }

    M_param_map.clear();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Formation::setCenter( const int unum )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! invalid unum " << unum
                  << std::endl;
        return;
    }

    M_mirror_number[unum - 1] = 0;

    std::map< int, boost::shared_ptr< FormationParam > >::iterator
        it = M_param_map.find( unum );
    if ( it == M_param_map.end() )
    {
        // create new parameter
        boost::shared_ptr< FormationParam > p( new FormationParam );
        M_param_map.insert( std::make_pair( unum, p ) );
    }
}

/*-------------------------------------------------------------------*/
/*!
  \brief set mirroring info
  \param unum changed player's number
  \param target_unum reffered player's number
*/
void
Formation::setMirror( const int unum,
                      const int target_unum )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! invalid unum " << unum
                  << std::endl;
        return;
    }
    if ( target_unum < 1 || 11 < target_unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! invalid target_unum " << unum
                  << std::endl;
        return;
    }
    if ( target_unum == unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! Never mirror itself. unum = " << unum
                  << "  mirro = " << target_unum
                  << std::endl;
        return;
    }
    if ( M_mirror_number[target_unum - 1] > 0 )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! " << target_unum << " is already mirrored player. "
                  << std::endl;
        return;
    }
    if ( M_mirror_number[target_unum - 1] == 0 )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! " << target_unum << " is center type player. "
                  << std::endl;
        return;
    }


    // check mirrored player
    for ( int i = 0; i < 11; ++i )
    {
        if ( i + 1 == unum ) continue;
        if ( M_mirror_number[i] == unum )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Error! player " << unum
                      << " is mirrored. "
                      << std::endl;
            return;
        }
    }

    // erase current parameter
    M_param_map.erase( unum );

    M_mirror_number[unum - 1] = target_unum;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Formation::setSide( const int unum )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! invalid unum " << unum
                  << std::endl;
        return;
    }

    M_mirror_number[unum - 1] = -1;

    std::map< int, boost::shared_ptr< FormationParam > >::iterator
        it = M_param_map.find( unum );
    if ( it == M_param_map.end() )
    {
        // create new parameter
        boost::shared_ptr< FormationParam > p( new FormationParam );
        p->randomize();
        M_param_map.insert( std::make_pair( unum, p ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Formation::setParam( const int unum,
                     boost::shared_ptr< FormationParam > fparam )
{
    int param_number = unum;
    if ( M_mirror_number[unum - 1] > 0 )
    {
        param_number = M_mirror_number[unum - 1];
    }

    std::map< int, boost::shared_ptr< FormationParam > >::iterator
        it = M_param_map.find( param_number );
    if ( it != M_param_map.end() )
    {
        // found old param.
        // replaced by new param.
        it->second = fparam;
        return;
    }

    // create new parameter
    M_param_map.insert( std::make_pair( unum, fparam ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
boost::shared_ptr< FormationParam >
Formation::getParam( const int unum )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! invalid unum " << unum
                  << std::endl;
        return boost::shared_ptr< FormationParam >
            ( static_cast< FormationParam * >( 0 ) );
    }

    if ( M_mirror_number[unum - 1] > 0 )
    {
        return boost::shared_ptr< FormationParam >
            ( static_cast< FormationParam * >( 0 ) );
    }

    std::map< int, boost::shared_ptr< FormationParam > >::const_iterator
        it = M_param_map.find( unum );
    if ( it != M_param_map.end() )
    {
        return it->second;
    }

    std::cerr << __FILE__ << ":" << __LINE__
              << " Error! Parameter not found! unum = "
              << unum << std::endl;

    return boost::shared_ptr< FormationParam >
        ( static_cast< FormationParam * >( 0 ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
const
boost::shared_ptr< const FormationParam >
Formation::param( const int unum ) const
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error! invalid unum " << unum
                  << std::endl;
        return  boost::shared_ptr< const FormationParam >
            ( static_cast< FormationParam * >( 0 ) );
    }

    const int player_number = ( M_mirror_number[unum - 1] <= 0 // center or original
                                ? unum
                                : M_mirror_number[unum - 1] );

    std::map< int, boost::shared_ptr< FormationParam > >::const_iterator
        it = M_param_map.find( player_number );
    if ( it != M_param_map.end() )
    {
        return it->second;
    }

    std::cerr << __FILE__ << ":" << __LINE__
              << " Error! Parameter not found! unum = "
              << unum << std::endl;

    return boost::shared_ptr< const FormationParam >
        ( static_cast< FormationParam * >( 0 ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Formation::getPosition( const int unum,
                        const rcsc::Vector2D & ball_pos ) const
{
    const boost::shared_ptr< const FormationParam > ptr = param( unum );
    if ( ! ptr )
    {
        std::cerr << "Error. FormationParam not found. unum = " << unum
                  << std::endl;
        return rcsc::Vector2D( 0.0, 0.0 );
    }
    FormationParam::Type type = FormationParam::SIDE;
    if ( M_mirror_number[unum - 1] > 0 )  type = FormationParam::MIRROR;
    if ( M_mirror_number[unum - 1] == 0 ) type = FormationParam::CENTER;

    return ptr->getPosition( ball_pos, type );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Formation::read( std::istream & is )
{
    std::string line_buf;

    while ( std::getline( is, line_buf ) )
    {
        //std::cerr << "Check tag [" << line_buf << "]" << std::endl;

        std::string tag;
        std::istringstream istr( line_buf );
        istr >> tag;
        if ( tag.empty()
             || tag[0] == '#'
             || tag[0] == '/' )
        {
            continue;
        }

        if ( tag == "Formation" )
        {
            istr >> M_name;
            break;
        }
    }

    //---------------------------------------------------
    // read each player's parameter set
    for ( int i = 0; i < 11; ++i )
    {
        if ( ! is.good() )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Formation read error! at number "
                      << i + 1 << std::endl;;
            return false;
        }
        // get one line
        std::getline( is, line_buf );
        //std::cerr << line_buf << std::endl;
        // check id
        int unum = 0;
        int mirror = 0;
        int n_read = 0;
        if ( std::sscanf( line_buf.c_str(),
                          " player %d %d %n",
                          &unum, &mirror,
                          &n_read ) != 2
             || n_read == 0 )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Formation read error! at number "
                      << i + 1
                      << " [" << line_buf << "]"
                      << std::endl;
            return false;
        }
        if ( unum != i + 1 )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Formation read error! Invalid player number. "
                      << " Expected " << i + 1
                      << "  but read number = " << unum
                      << std::endl;
            return false;
        }
        if ( mirror == unum )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Formation read error! Invalid mirror number. at "
                      << i
                      << " mirroing player itself. unum = " << unum
                      << "  mirror = " << mirror
                      << std::endl;
            return false;
        }
        if ( 11 < mirror )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Formation read error! Invalid mirror number. at "
                      << i
                      << "  Mirror number is out of range. unum = " << unum
                      << "  mirror = " << mirror
                      << std::endl;
            return false;
        }

        M_mirror_number[i] = mirror;

        // this player is mirror type
        if ( mirror > 0 )
        {
            continue;
        }

        // read parameters
        boost::shared_ptr< FormationParam > param( new FormationParam );
        if ( ! param->read( is ) )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Formation read error! at number"
                      << i + 1
                      << " [" << line_buf << "]"
                      << std::endl;
            return false;
        }

        M_param_map.insert( std::make_pair( unum, param ) );
    }

    //---------------------------------------------------
    // check mirror number circuration reference
    for ( int i = 0; i < 11; ++i )
    {
        if ( M_mirror_number[i] <= 0 ) continue;
        int mirror_unum = M_mirror_number[i];
        if ( M_mirror_number[mirror_unum - 1] > 0 )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Formation read error! Bad mirroring. player "
                      << i + 1
                      << " mirro = " << mirror_unum
                      << " is already mirroring player"
                      << std::endl;
            return false;
        }
    }

    //---------------------------------------------------
    // read other data  if exist
    while ( std::getline( is, line_buf ) )
    {
        //std::cerr << " search end\n" <<  line_buf << std::endl;
        int n_read = 0;

        if ( line_buf.empty()
             || line_buf[0] == '#'
             || line_buf[0] == ';'
             || line_buf[0] == '/' )
        {
            continue;
        }

        char read_buf[256];
        if ( std::sscanf( line_buf.c_str(),
                          " %s %n",
                          read_buf,
                          &n_read ) != 1
             || n_read == 0 )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Formation read error! End tag is not found.["
                      << line_buf << "] read = " << n_read
                      << std::endl;;
            return false;
        }

        if ( ! std::strcmp( read_buf, "End" ) )
        {
            // end of formation
            return true;
        }
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!
  condition <playmode> <>
  player <unum> 0 <Role> <Param1> <Param2> ...
  player <unum> <mirro> <Role>
  ...
  important <x> <y>
  important <x> <y>
  ...
*/
std::ostream &
Formation::print( std::ostream & os ) const
{
    os << "Formation " << getName() << '\n';

    for ( int i = 0; i < 11; ++i )
    {
        const int unum = i + 1;
        os << "player " << unum << " " << M_mirror_number[i] << " ";
        if ( M_mirror_number[i] > 0 )
        {
            os << '\n';
            continue;
        }
        os << '\n';

        std::map< int, boost::shared_ptr< FormationParam > >::const_iterator
            it = M_param_map.find( unum );
        if ( it == M_param_map.end() )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Invalid player Id at number "  << i + 1
                      << ".  No formation param!"
                      << std::endl;
        }
        else
        {
            it->second->print( os );
        }
    }

    os << "End" << std::endl;
    return os;
}
