// -*-c++-*-

/*!
	\file formation.h
	\brief formation data classes Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCERWINDOW2_FORMATION_H
#define SOCCERWINDOW2_FORMATION_H

#include <cassert>
#include <iostream>
#include <vector>
#include <map>
#include <boost/shared_ptr.hpp>

#include <rcsc/geom/vector_2d.h>
#include <rcsc/bpn1.h>

///////////////////////////////////////////////////////////////////////

class FormationParam {
public:
    static const double PITCH_LENGTH;
    static const double PITCH_WIDTH;

    enum Type {
        SIDE,
        MIRROR,
        CENTER,
    };

    //typedef BPNetwork1< 2, 4, 2 > PosNet;  // bad
    //typedef BPNetwork1< 2, 5, 2 > PosNet;  // not bad
    //typedef BPNetwork1< 2, 6, 2 > PosNet;   // not bad
    //typedef BPNetwork1< 2, 7, 2 > PosNet;   // good?
    //typedef BPNetwork1< 2, 8, 2 > PosNet;   // good?
    typedef rcsc::BPNetwork1< 2, 10, 2 > PosNet; // good
    //typedef BPNetwork1< 2, 12, 2 > PosNet;   // good
private:

    std::string M_role_name;

    mutable PosNet M_param;

public:

    FormationParam();

    void randomize();

    const
    std::string & getRoleName() const
      {
          return M_role_name;
      }

    PosNet & getParam()
      {
          return M_param;
      }

    const
    PosNet & param() const
      {
          return M_param;
      }

    //! returns strategic position
    rcsc::Vector2D getPosition( const rcsc::Vector2D & ball_pos,
                                const Type type ) const;


    void setRoleName( const std::string & name )
      {
          M_role_name = name;
      }

    bool read( std::istream & is );
    std::ostream & print( std::ostream & os ) const;

private:
    bool readRoleName( std::istream & is );
    bool readParam( std::istream & is );
    std::ostream & printRoleName( std::ostream & os ) const;
    std::ostream & printParam( std::ostream & os ) const;

};

///////////////////////////////////////////////////////////////////////

class Formation {
private:
    //! name of this formation
    std::string M_name;

    /*!
      mirror number.holder
      >0 positive numver means that this player referes other player.
      ==0 Zero means center type player
      <0 negative number means that this player is original.
    */
    int M_mirror_number[11];

    //! key: unum. but size is not always 11 if mirror player exists.
    std::map< int, boost::shared_ptr< FormationParam > > M_param_map;

public:
    Formation();

    explicit
    Formation( const std::string & name );

    void clear();

    void setName( const std::string & name )
      {
          M_name = name;
      }

    void setCenter( const int unum );
    void setSide( const int unum );
    void setMirror( const int unum,
                    const int target_unum );

    void setParam( const int unum,
                   boost::shared_ptr< FormationParam > fparam );

    void checkRemove();

    //------------------------------------------------------
    const
    std::string & getName() const
      {
          return M_name;
      }

    bool isLeft( const int unum ) const
      {
          if ( unum < 1 || 11 < unum ) return false;
          return ( M_mirror_number[unum - 1] < 0 );
      }
    bool isCenter( const int unum ) const
      {
          if ( unum < 1 || 11 < unum ) return false;
          return ( M_mirror_number[unum - 1] == 0 );
      }
    bool isRight( const int unum ) const
      {
          return isUsingMirror( unum );
      }
    bool isUsingMirror( const int unum ) const
      {
          if ( unum < 1 || 11 < unum ) return false;
          return ( M_mirror_number[unum - 1] > 0 );
      }

    int getMirrorNumber( const int unum ) const
      {
          if ( unum < 1 || 11 < unum ) return 0;
          return M_mirror_number[unum - 1];
      }

    boost::shared_ptr< FormationParam > getParam( const int unum );

    const
    boost::shared_ptr< const FormationParam > param( const int unum ) const;


    rcsc::Vector2D getPosition( const int unum,
                                const rcsc::Vector2D & ball_pos ) const;

    //------------------------------------------------------
    bool read( std::istream & is );
    // add new line at last
    std::ostream & print( std::ostream & os ) const;
};


#endif
