/*
Copyright (c) 2000-2003, Jelle Kok, University of Amsterdam
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

3. Neither the name of the University of Amsterdam nor the names of its
contributors may be used to endorse or promote products derived from this
software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


/*****************************************************************************/
/********************** CLASS FORMATIONS *************************************/
/*****************************************************************************/


#include <cctype>    // needed for isdigit
#include <cmath>     // needed for pow
#include <cstring>   // needed for strlen
#include <fstream>

#include <rcsc/math_util.h>
#include <rcsc/param/server_param.h>

#include "formation_uva.h"


//                                                                           //
//           Parse is only used in this files                                //
//                                                                           //


namespace {

/*****************************************************************************/
/********************** CLASS PARSE ******************************************/
/*****************************************************************************/

/*! This method walks through the string starting at the character where strMsg
  points to and stops when or the end of the string is reached or a non space
  is reached. strMsg is updated to this new position.
  \param strMsg pointer to a character in a string array
  \return character that is at the first non space, '\0' when end of string
  is reached. */
char
gotoFirstNonSpace( char** strMsg )
{
    while(*strMsg && isspace(**strMsg) )
        (*strMsg)++;
    return  (*strMsg) ?  **strMsg : '\0';

}

/*! This method walks through the string starting at the character where strMsg
  points to and stops when<BR>
  - the end of the string is reached<BR>
  - a character different than ' ' and ')' is read<BR>
  strMsg is changed after this method is called.
  \param strMsg pointer to a character in a string array
  \return first character that is not equal to ' ' or ')', '\0' when string
  didn't contain such a character. */
char
gotoFirstSpaceOrClosingBracket( char** strMsg )
{
    while( *strMsg && **strMsg!=' ' && **strMsg!=')' )
        (*strMsg)++;
    return  (*strMsg) ?  **strMsg : '\0';
}

/*! This method walks through the string starting at the character where strMsg
  points to and stops when the character c is reached. strMsg is changed
  after this method is called..
  \param c character that is searched for in strMsg.
  \param strMsg pointer to a character in a string array
  \return number of character skipped to reach c, -1 when not found */
int
gotoFirstOccurenceOf( char c, char** strMsg )
{
    int i=0;
    while(**strMsg && **strMsg != c )
    {
        i++;
        (*strMsg)++;
    }
    if( ! **strMsg )
        return -1;
    return i;
}

/*! This method walks through the string starting at the character where strMsg
  points to and returns the first integer that can be read from this string.
  Any other characters in front of this integer are discarded. After this
  method is returned, strMsg points to the first character after the final
  value of the integer.
  \param strMsg pointer to a character in a string array
  \return first integer that can be read from this string. */
int
parseFirstInt( char** strMsg )
{
    int  iRes  = 0;
    bool bIsMin= false;
    char *str  = *strMsg;

    while( *str != '\0' && !isdigit(*str) && *str!='-')
        str++;                             // walk to first non digit or minus sign


    if( *str != '\0' && *str=='-')       // if it was a minus sign, remember
    {
        bIsMin=true;
        str++;
    }

    while( *str != '\0' && *str<='9' && *str>='0' )   // for all digits
    {
        iRes=iRes*10+(int)(*str-'0');                 // multiply old res with 10
        str++;                                        // and add new value
    }
    *strMsg = str;
    return (bIsMin) ? -iRes : iRes;
}

/*! This method walks through the string starting at the character where strMsg
  points to and returns the first double that can be read from this string.
  Any other characters in front of this integer are discarded. After this
  method is returned, strMsg points to the first character after the final
  value of the double. 4e-3, and NaN or nan are also recognized. When input
  contains NaN or nan, -1000.0 is returned.
  \param strMsg pointer to a character in a string array
  \return first double that can be read from this string. */
double
parseFirstDouble( char** strMsg )
{
    double dRes=0.0, dFrac=1.0;
    bool bIsMin=false, bInDecimal=false, bCont=true;
    char *str = *strMsg;

    // go to first part of double (digit, minus sign or '.')
    while( *str != '\0' && !isdigit(*str) && *str!='-' && *str!='.')
    {
        // when NaN or nan is double value, return -1000.0
        if( (*str=='N' && strlen(str)>3 && *(str+1)=='a' && *(str+2)=='N') ||
            (*str=='n' && strlen(str)>3 && *(str+1)=='a' && *(str+2)=='n') )
        {
            *strMsg = str+3;
            return -1000.0;
        }
        else
            str++;
    }

    if( *str != '\0' && *str=='-')               // if minus sign, remember that
    {
        bIsMin=true;
        str++;
    }

    while( *str != '\0' && bCont)                // process the number bit by bit
    {
        if( *str=='.' )                            // in decimal part after '.'
            bInDecimal = true;
        else if( bInDecimal && *str<='9' && *str>='0') // number and in decimal
        {
            dFrac=dFrac*10.0;                        // shift decimal part to right
            dRes += (double)(*str-'0')/dFrac;        // and add number
        }
        else if( *str<='9' && *str>='0' )          // if number and not decimal
            dRes=dRes*10+(double)(*str-'0');         // shift left and add number
        else if( *str=='e' ) // 10.6e-08           // if to power e
        {
            if( *(str+1) == '+' )                    // determine sign
                dRes *=  pow(10, parseFirstInt(&str)); // and multiply with power
            else if( *(str+1) == '-' )
            {
                str = str+2;                           // skip -
                dRes /=  pow(10, parseFirstInt(&str)); // and divide by power
            }
            bCont = false;                           // after 'e' stop
        }
        else
            bCont = false;                           // all other cases stop

        if( bCont == true )                        // update when not stopped yet
            str++;
    }
    *strMsg = str;
    return (bIsMin && dRes != 0.0) ? -dRes : dRes;
}

}

//end namespace for Parse

namespace uva {

double UnknownDoubleValue = 0.0;

PlayerTypeInfo::PlayerTypeInfo()
{
    setValues( PT_ILLEGAL, UnknownDoubleValue,UnknownDoubleValue,
               UnknownDoubleValue,UnknownDoubleValue, false );
}

/*! This Constructor receives the values for all the member variables as
  arguments and initializes the member variables using the method setValues.
  \param pt PlayerType corresponding to the player type of this class
  \param dAttrX x attraction to the ball
  \param dAttrY y attraction to the ball
  \param dMinX minimal x coordinate for this player type
  \param dMaxX maximal x coordinate for this player type
  \param bBehindBall boolean indicating whether this player type should
  always stay behind the ball. */

PlayerTypeInfo::PlayerTypeInfo( PlayerT pt, double dAttrX, double dAttrY,
                                double dMinX, double dMaxX, bool bBehindBall  )
{
    setValues( pt, dAttrX, dAttrY, dMinX, dMaxX, bBehindBall );
}

/*! This method receives the values for all the member variables as arguments
  and sets these member variables.
  \param pt PlayerType corresponding to the player type of this class
  \param ax x attraction to the ball
  \param ay y attraction to the ball
  \param minx minimal x coordinate for this player type
  \param maxx maximal x coordinate for this player type
  \param bb boolean indicating whether this player type should always
  stay behind the ball.
  \return bool indicating whether update was successful. */
bool
PlayerTypeInfo::setValues( PlayerT pt,   double ax,   double ay,
                           double minx,  double maxx, bool bb )
{
    playerType      = pt;
    dAttrX          = ax;
    dAttrY          = ay;
    dMinX           = minx;
    dMaxX           = maxx;
    bBehindBall     = bb;

    return true;
}

/*! This method print the different member values separated by comma's to the
  specified output stream.
  \param os output stream to which member values are printed */
void
PlayerTypeInfo::show( std::ostream & os )
{
    os << "(" << (int)playerType << ", "
       << dAttrX << ", " << dAttrY << ", "
       << dMinX  << ", " << dMaxX << ", "
       << bBehindBall
       <<  ")" << std::endl;
}


/*****************************************************************************/
/********************** CLASS FORMATIONTYPEINFO*******************************/
/*****************************************************************************/

/*! This method prints all the information about this formation to the
  specified output stream. The format is the following:
  - x coordinate of the home position for all the roles
  - y coordinate of the home position for all the roles
  - the player types for all the roles
  - the x attraction for all the player types
  - the y attraction for all the player types
  - indication whether to stay behind the ball for all the player types
  - minimal x coordinate for all the player types
  - maximal x coordinate for all the player types
  \param os output stream for output. */
void
FormationTypeInfo::show( std::ostream & os )
{
    char str[128];
    for( int i = 0; i < 11; i++ )
    {
        sprintf(str, "%3.2f ", posHome[i].x );
        os << str;
    }
    os << std::endl;
    for( int i = 0; i < 11; i++ )
    {
        sprintf( str, "%3.2f ", posHome[i].y );
        os << str;
    }
    os << std::endl;
    for( int i = 0; i < 11; i++ )
    {
        sprintf( str, "%5d ", (int)playerType[i] );
        os << str;
    }
    os << std::endl;
    for( int i = 0; i < MAX_PLAYER_TYPES; i++ )
    {
        sprintf( str, "%3.2f ", playerTypeInfo[i].getAttrX() );
        os << str;
    }
    os << std::endl;
    for( int i = 0; i < MAX_PLAYER_TYPES; i++ )
    {
        sprintf( str, "%3.2f ", playerTypeInfo[i].getAttrY() );
        os << str;
    }
    os << std::endl;
    for( int i = 0; i < MAX_PLAYER_TYPES; i++ )
    {
        sprintf( str, "%5d ", playerTypeInfo[i].getBehindBall() );
        os << str;
    }
    os << std::endl;
    for( int i = 0; i < MAX_PLAYER_TYPES; i++ )
    {
        sprintf( str, "%3.2f ", playerTypeInfo[i].getMinX() );
        os << str;
    }
    os << std::endl;
    for( int i = 0; i < MAX_PLAYER_TYPES; i++ )
    {
        sprintf( str, "%3.2f ", playerTypeInfo[i].getMaxX() );
        os << str;
    }
    os << std::endl;
}


/*! This is the constructor for the Formations class and needs as arguments
  a formation configuration file, the current formation and the number of the
  agent in this formation (normally at start-up this equals the player
  number).
  \param strFile string representation of the formation configuration file
  \param curFt current formation type (default FT_ILLEGAL)
  \param iNr number of the agent in this formation (default 1)*/
#if 0
Formations::Formations( const char *strFile, FormationT curFt, int iNr )
{
    if( strFile[0] == '\0' )
    {
        std::cerr << "(Formations::Formations) No Filename given" << std::endl;
        return;
    }

    if( read( strFile ) == false )
        std::cerr << "(Formations::Formations) Error reading file " << strFile << std::endl;
    curFormation = curFt;
    setPlayerInFormation( iNr );
}
#endif

void
Formations::show( std::ostream & os )
{
    for( int i = 0 ; i < MAX_FORMATION_TYPES; i ++ )
        formations[i].show( os );
    os << "Current formation: " << (int)curFormation << std::endl
       << "Player nr in formation: " << iPlayerInFormation ;
}

/*!
  Read Function which read conf-files, from UvA source code
*/
bool
Formations::read( const char *filepath )
{
    std::ifstream in( filepath );
    if( !in )
    {
        std::cerr << "(readValues::readValues) Could not open file '" <<
            filepath << "'" << std::endl;
        return false;
    }

    char strLine[256], *str;
    int            iLineNr          = 0, i;
    int            iForm            = 0; // current formation type that is parsed
    int            iLineInFormation = 0; // current offset of line in formation
    bool           bReturn          = true;
    PlayerTypeInfo *pt_info;

    // read all lines
    while( bReturn && in.getline( strLine, sizeof(strLine) ) )
    {
        str = &strLine[0];
        iLineNr++;
        // comment and empty lines should be skipped
        if( !(strLine[0] == '\n' || strLine[0] == '#' || strLine[0]=='\0' ||
              gotoFirstNonSpace( &str ) == '\0' ) )
        {
            // there are ten different lines in a formation (see comment above)
            // all values for each line are parsed in one iteration
            // after all 10 lines are parsed, the sequence it is resetted.
            switch( iLineInFormation )
            {
            case 0: // first line is the number of the formation
                iForm =  parseFirstInt( &str );
                break;
            case 1: // the x coordinate of the home pos for all the players
                for( i = 0 ; i <  11  ; i ++ )
                    formations[iForm].setXPosHome( parseFirstDouble(&str), i);
                break;
            case 2: // the y coordinate of the home pos for all the players
                for( i = 0 ; i <  11  ; i ++ )
                    formations[iForm].setYPosHome( parseFirstDouble(&str), i);
                break;
            case 3: // the player types for all the players
                for( i = 0 ; i <  11  ; i ++ )
                    formations[iForm].setPlayerType(
                                                    (PlayerT)  parseFirstInt(&str), i);
                break;
            case 4: // the x attraction for all the player types
                for( i = 0 ; i < MAX_PLAYER_TYPES ; i ++ )
                {
                    pt_info = formations[iForm].getPlayerTypeInfo( i );
                    pt_info->setAttrX(  parseFirstDouble( &str ) );
                }
                break;
            case 5: // the y attraction for all the player types
                for( i = 0 ; i < MAX_PLAYER_TYPES ; i ++ )
                {
                    pt_info = formations[iForm].getPlayerTypeInfo( i );
                    pt_info->setAttrY(  parseFirstDouble( &str ) );
                }
                break;
            case 6: // stay behind the ball for all the player types
                for( i = 0 ; i < MAX_PLAYER_TYPES ; i ++ )
                {
                    pt_info = formations[iForm].getPlayerTypeInfo( i );
                    if(  parseFirstInt( &str ) == 1 )
                        pt_info->setBehindBall( true );
                    else
                        pt_info->setBehindBall( false );          ;
                }
                break;
            case 7: // the minimal x coordinate for all the player types
                for( i = 0 ; i < MAX_PLAYER_TYPES ; i ++ )
                {
                    pt_info = formations[iForm].getPlayerTypeInfo( i );
                    pt_info->setMinX(  parseFirstDouble( &str ));
                }
                break;
            case 8:// the maximal x coordinate for all the player types
                for( i = 0 ; i < MAX_PLAYER_TYPES ; i ++ )
                {
                    pt_info = formations[iForm].getPlayerTypeInfo( i );
                    pt_info->setMaxX(  parseFirstDouble( &str ));
                }
                break;
            default:
                std::cerr << "(Formation::readFormations) error line " << iLineNr
                          <<std::endl;
                return false;
            }

            iLineInFormation++;          // go one line further
            if( iLineInFormation == 9 ) // each formation consists of ten lines
                iLineInFormation = 0;
        }
    }

    M_valid = true;
    return true;
}



bool
Formations::setFormation( FormationT formation )
{
    curFormation = formation;
    return true;
}


/*! This method returns the current formation.
  \return current formation  */
FormationT
Formations::getFormation( ) const
{
    return curFormation;
}


/*! This method sets the player number of the agent in the current formation to
  'iNumber'.
  \param iNumber new player number for this agent
  \return bool indicating whether the update was succesfull */
bool
Formations::setPlayerInFormation( int iNumber )
{
    iPlayerInFormation = iNumber;
    return true;
}

#if 0
/*! This method returns the role number of the agent in the current formation
  \return player number for this agent in the current formation */
int
Formations::getPlayerInFormation( ObjectT obj  ) const
{
    int i;
    if( obj == OBJECT_ILLEGAL )
        i = iPlayerInFormation;
    else
        i = SoccerTypes::getIndex( obj );

    return i;
}
#endif

#if 0
/*! This method returns the player type for the specified object
  \return player type for the agent in the current formation */
PlayerT
Formations::getPlayerType( ObjectT obj, FormationT ft ) const
{
    if( ft == FT_ILLEGAL )
        ft = curFormation;
    return formations[ ft ].getPlayerType( SoccerTypes::getIndex( obj ) );
}
#endif

/*! This method returns the player type for the agent in the current formation
  \return player type for the agent in the current formation */
PlayerT
Formations::getPlayerType( int iIndex, FormationT ft ) const
{
    if( ft == FT_ILLEGAL )
        ft = curFormation;
    if( iIndex == -1  )
        iIndex = iPlayerInFormation;
    return formations[ ft ].getPlayerType( iIndex );
}

/*!
  \param number unum - 1
*/
rcsc::Vector2D
Formations::getStrategicPosition( const int number,
                                  const rcsc::Vector2D & posBall,
                                  const double & dMaxXInPlayMode,
                                  bool bInBallPossession,
                                  const double & dMaxYPercentage,
                                  FormationT ft ) const
{
    if( ft == FT_ILLEGAL )
    {
        ft = curFormation;
    }
    rcsc::Vector2D     posHome;
    const PlayerTypeInfo * ptInfo = formations[ft].getPlayerTypeInfoOfPlayer( number );
    double x, y;

    // get the home position and calculate the associated strategic position
    posHome = formations[ft].getPosHome( number );
    y = posHome.y + posBall.y * ptInfo->getAttrY();
    x = posHome.x + posBall.x * ptInfo->getAttrX();

    // do not move too much to the side
    if ( std::fabs( y ) > 0.5 * dMaxYPercentage * rcsc::ServerParam::i().pitchWidth() )
    {
        y = rcsc::sign( y )* 0.5 * dMaxYPercentage * rcsc::ServerParam::i().pitchWidth();
    }

    // when behind ball is set, do not move to point in front of ball
    if ( ptInfo->getBehindBall() == true && x > posBall.x )
    {
        x = posBall.x;
    }

    // do not move past maximal x or before minimal x
    if( x > ptInfo->getMaxX() )
    {
        x = ptInfo->getMaxX();
    }
    else if( x < ptInfo->getMinX() )
    {
        x = ptInfo->getMinX();
    }

    // when x coordinate is in front of allowed x value, change it
    if( x > dMaxXInPlayMode )
    {
        x = dMaxXInPlayMode;
    }

    return rcsc::Vector2D( x, y );
}


Formations &
Formations::instance()
{
    static Formations s_instance;
    return s_instance;

}

}
