/*
Copyright (c) 2000-2003, Jelle Kok, University of Amsterdam
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice, this
list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
this list of conditions and the following disclaimer in the documentation
and/or other materials provided with the distribution.

3. Neither the name of the University of Amsterdam nor the names of its
contributors may be used to endorse or promote products derived from this
software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/


#ifndef AGENT2D_FORMATION_UVA_H
#define AGENT2D_FORMATION_UVA_H

#include <iostream>
#include <rcsc/geom/vector_2d.h>

namespace uva {

/*!The FormationT enumeration contains the different formation types that
  are defined. */
enum FormationT {
    FT_ILLEGAL,             /*!< illegal formation type                  */
    FT_INITIAL,             /*!< initial formation type (before kick off)*/
    FT_433_OFFENSIVE,       /*!< 433 offensive formation                 */
    FT_334_OFFENSIVE,       /*!< 434 offensive formation                 */
    FT_DEFENSIVE,           /*!< defensive formation type                */
    FT_OPEN_DEFENSIVE,      /*!< open defensive formation type           */
    FT_343_ATTACKING,       /*!< attacking formation type                */
    MAX_FORMATION_TYPES
} ;

/*!The PlayerT enumeration contains the different playertypes that are defined
  in a formation. This should not be confused with the later introducted
  player_types in soccerserver 7.xx that denotes the different possible
  heterogeneous players. A player type in the context PlayerT denotes the
  kind of player (midfielder, attacker) on the field. Its role on the pitch.*/
enum PlayerT {
    PT_ILLEGAL,             /*!< illegal player type */
    PT_GOALKEEPER,          /*!< goalkeeper          */
    PT_DEFENDER_CENTRAL,    /*!< central defender    */
    PT_DEFENDER_SWEEPER,    /*!< sweeper defender    */
    PT_DEFENDER_WING,       /*!< wing defender       */
    PT_MIDFIELDER_CENTER,   /*!< central midfielder  */
    PT_MIDFIELDER_WING,     /*!< wing midfielder     */
    PT_ATTACKER_WING,       /*!< wing attacker       */
    PT_ATTACKER,            /*!< central attacker    */
    MAX_PLAYER_TYPES
} ;


/*****************************************************************************/
/********************** CLASS PLAYERTYPEINFO *********************************/
/*****************************************************************************/

/*! This class contains information for one individual player_type, defined in
  SoccerTypes.h. A player_type should not be confused with the player_types
  introduced in soccerserver 7.xx. A playerType PlayerT is defined as the kind of
  a player. Different possibilities are PT_ATTACKER, PT_MIDFIELDER_WING, etc.
  This class contains different characteristics of one playertype. This
  information consists of the following values:
  - dAttrX - x attraction to the ball for this player type.
  - dAtttY - y attraction to the ball for this player type.
  - dMinX  - minimal x coordinate for this player
  - dMaxX  - maximal x coordinate for this player
  - bBehindBall - indicating whether this player type should always stay behind
  the ball or not.


  This class contains different get and set methods to change the values
  associated for this class, normally these are changed when the
  Formations class reads in the formation file. */
class PlayerTypeInfo
{
    PlayerT playerType; /*!< This class gives information about this PlayerType*/
    double  dAttrX;     /*!< x attraction to the ball                          */
    double  dAttrY;     /*!< y attraction to the ball                          */
    double  dMinX;      /*!< minimal x coordinate for this player type         */
    double  dMaxX;      /*!< maximal x coordinate for this player type         */
    bool    bBehindBall;/*!< should player always stay behind the ball         */

public:
    PlayerTypeInfo( );
    PlayerTypeInfo( PlayerT, double, double, double, double, bool );

    // method to set all the values at once and displaying them all
    bool    setValues( PlayerT, double, double, double, double, bool  );
    void    show     ( std::ostream &os = std::cout );

    // standard get and set methods to individually set all the values
    bool    setPlayerType    ( PlayerT type  )
      {
          playerType = type;
          return true;
      }
    PlayerT getPlayerType    (               ) const
      {
          return playerType;
      }
    bool    setAttrX         ( double  attrX )
      {
          dAttrX = attrX;
          return true;
      }
    double  getAttrX         (               ) const
      {
          return dAttrX;
      }
    bool    setAttrY         ( double  attrY )
      {
          dAttrY = attrY;
          return true;
      }
    double  getAttrY         (               ) const
      {
          return dAttrY;
      }
    bool    setMinX          ( double  minX  )
      {
          dMinX = minX;
          return dMinX;
      }
    double  getMinX          (               ) const
      {
          return dMinX;
      }
    bool    setMaxX          ( double  maxX  )
      {
          dMaxX = maxX;
          return true;
      }
    double  getMaxX          (               ) const
      {
          return dMaxX;
      }
    bool    setBehindBall    ( bool    b     )
      {
          bBehindBall = b;
          return true;
      }
    bool    getBehindBall    (               ) const
      {
          return bBehindBall;
      }

};


/*****************************************************************************/
/********************** CLASS FORMATIONTYPEINFO ******************************/
/*****************************************************************************/

/*!This class contains information about one specific formation. It
  contains the formation type (defined in SoccerTypes.h), the home
  position of all the roles (=specific player in a formation), the
  player types for all the roles and the information about the
  different player_types. Furthermore it contains methods to retrieve
  this information for a specific role. */
class FormationTypeInfo
{
private:
    FormationT formationType;    /*!< type of this formation   */
    rcsc::Vector2D posHome[ 11 ]; /*!< home position for roles  */
    PlayerT playerType[ 11 ]; /*!< player_types for roles   */
    PlayerTypeInfo playerTypeInfo[ MAX_PLAYER_TYPES ]; /*!< info for roles    */

public:
    FormationTypeInfo()
      { }

    void show( std::ostream & os = std::cout );

    // get and set methods to get information for a player in this formation
    bool setFormationType( FormationT type )
      {
          formationType = type;
          return true;
      }

    FormationT getFormationType() const
      {
          return formationType;
      }

    bool setPosHome( const rcsc::Vector2D & pos,
                     int atIndex )
      {
          posHome[ atIndex ] = pos;
          return true;
      }

    bool setXPosHome( const double & x,
                      const int atIndex )
      {
          posHome[ atIndex ].x = x;
          return true;
      }
    bool setYPosHome( const double & y,
                      const int atIndex )
      {
          posHome[ atIndex ].y =  y;
          return true;
      }
    rcsc::Vector2D getPosHome( const int atIndex ) const
      {
          return posHome[ atIndex ];
      }
    bool setPlayerType( PlayerT type,
                        const int atIndex )
      {
          playerType[ atIndex ] = type;
          return true;
      }
    PlayerT getPlayerType( const int atIndex ) const
      {
          return playerType[ atIndex ];
      }
    bool setPlayerTypeInfo( PlayerTypeInfo info,
                            const int atIndex )
      {
          playerTypeInfo[ atIndex ] = info;
          return true;
      }

    PlayerTypeInfo * getPlayerTypeInfo( const int atIndex )
      {
          return &playerTypeInfo[ atIndex ];
      }

    const
    PlayerTypeInfo * getPlayerTypeInfoOfPlayer( const int iPlayerInFormation ) const
      {
          return &playerTypeInfo[ playerType[ iPlayerInFormation ] ];
      }
};

/*****************************************************************************/
/********************** CLASS FORMATIONS *************************************/
/*****************************************************************************/

/*!This class is a container for all different Formation Types: it
  contains the information of all the formation types. Furthermore it
  contains two other values: the current formation type that is used
  by the agent and the role of the agent in the current
  formation. These two values fully specify the position of this
  player in the formation. */

class Formations {
private:
    bool M_valid;

    FormationTypeInfo formations[ MAX_FORMATION_TYPES ]; /*!< stored formations*/
    FormationT        curFormation;       /*!< type of the current formation   */
    int               iPlayerInFormation; /*!< role agent in current formation */


    Formations()
        : M_valid( false )
        , curFormation( FT_ILLEGAL )
        , iPlayerInFormation( 1 )
      { }

public:

    void show ( std::ostream &os = std::cout );

    // method to get the strategic position depending on different factors


    //dMaxXInPlaymode apply proper power consider from Offside
    rcsc::Vector2D getStrategicPosition( const int number,
                                         const rcsc::Vector2D & posBall,
                                         const double & dMaxXInPlayMode,
                                         bool bInBallPossession = false,
                                         const double & dMaxYPercentage = 0.75,
                                         FormationT ft = FT_ILLEGAL ) const;

    // method to read the formations from a formation configuration file.
    bool        read      ( const char *strFile );

    // standard get and set methods for the different member variables
    bool        setFormation        ( FormationT formation );
    FormationT  getFormation        () const;
    bool        setPlayerInFormation( int number );
    PlayerT     getPlayerType       ( int iIndex = -1,
                                      FormationT ft = FT_ILLEGAL ) const;

    bool valid() const
      {
          return M_valid;
      }

    static
    Formations & instance();
};

}

#endif
