// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>

#include <boost/shared_ptr.hpp>

#include <rcsc/player/logger.h>
#include <rcsc/player/audio_sensor.h>
#include <rcsc/param/server_param.h>
#include <rcsc/param/player_param.h>

#include "bhv_goalie_free_kick.h"
#include "bhv_penalty_kick.h"
#include "bhv_pre_process.h"
#include "bhv_set_play.h"
#include "bhv_set_play_kick_in.h"

#include "formation.h"
#include "formation_uva.h"
#include "soccer_role.h"

#include "sample_player.h"

/*-------------------------------------------------------------------*/
/*!

*/
SamplePlayer::SamplePlayer()
    : PlayerAgent()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
SamplePlayer::~SamplePlayer()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
bool
SamplePlayer::doInit( const int argc,
                      const char * const * argv )
{
    if ( ! PlayerAgent::doInit( argc, argv ) )
    {
        setServerAlive( false );
        return false;
    }

    if ( ! M_strategy.read( config().configDir() ) )
    {
        std::cerr << "***ERROR*** Failed to read team strategy." << std::endl;
        setServerAlive( false );
        return false;
    }

    ///// delete here, if you do not use SBSP formation ////////
    for ( int i = 0; i < argc; ++i )
    {
        if ( ! std::strcmp( argv[i], "--fconf" ) )
        {
            if ( argc == i+1 )
            {
                std::cerr << "***ERROR*** --fconf must have value."
                          << std::endl;
                return false;
            }
            if ( ! uva::Formations::instance().read( argv[i+1] ) )
            {
                std::cerr << "***ERROR*** Failed to read [" << argv[i+1]
                          << "]"
                          << std::endl;
                return false;
            }
        }
    }
    ////////////////////////////////////////////////////////////

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
SamplePlayer::initTeam()
{
    int number = config().playerNumber();
    if ( number == 0 )
    {
        number = world().self().unum();
        getConfig().setPlayerNumber( number );
    }
}

/*-------------------------------------------------------------------*/
/*!
  main decision
  virtual method in super class
*/
void
SamplePlayer::doAction()
{
    if ( audioSensor().trainerMessage().time_ == world().time() )
    {
        std::cerr << world().teamName() << ' '
                  << world().self().unum() << ": "
                  << world().time()
                  << " hear trainer message ["
                  << audioSensor().trainerMessage().message_
                  << "]"
                  << std::endl;
    }

    if ( audioSensor().freeformMessage().time_ == world().time() )
    {
        std::cerr << world().teamName() << ' '
                  << world().self().unum() << ": "
                  << world().time()
                  << " hear freeform message ["
                  << audioSensor().freeformMessage().message_
                  << "]"
                  << std::endl;
    }

    //////////////////////////////////////////////////////////////
    // check tackle expires
    // check self position accuracy
    // ball search
    // check queued intention
    // check simultaneous kick
    if ( Bhv_PreProcess( M_strategy ).execute( this ) )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: preprocess done"
                            ,__FILE__, __LINE__ );
        return;
    }

    //////////////////////////////////////////////////////////////
    // create current role
    boost::shared_ptr< SoccerRole >
        role_ptr = M_strategy.createRole( config().playerNumber(),
                                          world() );

    if ( ! role_ptr )
    {
        std::cerr << config().teamName() << ": "
                  << world().self().unum()
                  << " Error. Role is not registerd.\nExit ..."
                  << std::endl;
        setServerAlive( false );
        return;
    }
    if  ( ! role_ptr->hasFormation() )
    {
        std::cerr << config().teamName() << ": "
                  << world().self().unum()
                  << " Error. does not have formation.\nExit ..."
                  << std::endl;
        setServerAlive( false );
        return;
    }

    //////////////////////////////////////////////////////////////
    // play_on mode
    if ( world().getGameMode().type() == rcsc::GameMode::PlayOn  )
    {
        role_ptr->execute( this );
        return;
    }

    //////////////////////////////////////////////////////////////
    // kick_in or corner_kick
    if ( ( world().getGameMode().type() == rcsc::GameMode::KickIn_
           || world().getGameMode().type() == rcsc::GameMode::CornerKick_ )
         && world().getOurSide() == world().getGameMode().side() )
    {
        if ( world().self().goalie() )
        {
            Bhv_GoalieFreeKick().execute( this );
        }
        else
        {
            const rcsc::Vector2D base_pos = world().ball().pos();
            rcsc::Vector2D home_pos
                = role_ptr->formation().getPosition( config().playerNumber(),
                                                     base_pos );
            ///// delete here, if you do not use SBSP formation ////////
            if ( uva::Formations::instance().valid() )
            {
                uva::Formations::instance().setFormation( uva::FT_433_OFFENSIVE );
                home_pos = uva::Formations::instance()
                    .getStrategicPosition( config().playerNumber() - 1,
                                           base_pos,
                                           52.5 );
            }
            //////////////////////////////////////////////////////////////
            Bhv_SetPlayKickIn( home_pos ).execute( this );
        }
        return;
    }


    //////////////////////////////////////////////////////////////
    // penalty kick mode
    if ( world().getGameMode().isPenaltyKickMode() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "%s:%d: penalty kick"
                            ,__FILE__, __LINE__ );
        Bhv_PenaltyKick().execute( this );
        return;
    }

    //////////////////////////////////////////////////////////////
    // goalie free kick mode
    if ( world().self().goalie() )
    {
        Bhv_GoalieFreeKick().execute( this );
        return;
    }

    //////////////////////////////////////////////////////////////
    // other set play mode

    rcsc::Vector2D move_pos
        = M_strategy.getSetPlayPosition( config().playerNumber(),
                                         role_ptr->formation(),
                                         world() );

    ///// delete here, if you do not use SBSP formation ////////
    if ( uva::Formations::instance().valid() )
    {
        uva::Formations::instance().setFormation( uva::FT_433_OFFENSIVE );
        move_pos = uva::Formations::instance()
            .getStrategicPosition( config().playerNumber() - 1,
                                   world().ball().pos(),
                                   world().getOffsideLineX() );
    }
    //////////////////////////////////////////////////////////////

    Bhv_SetPlay( move_pos ).execute( this );
}

/*-------------------------------------------------------------------*/
/*!
  communication decision.
  virtual method in super clas
*/
void
SamplePlayer::doCommunication()
{
    if ( ! config().useCommunication()
         || world().getGameMode().type() == rcsc::GameMode::BeforeKickOff
         || world().getGameMode().type() == rcsc::GameMode::AfterGoal_
         || world().getGameMode().isPenaltyKickMode() )
    {
        return;
    }

    const rcsc::PlayerPtrCont & ball_near_mates = world().getTeammatesFromBall();


    std::string say_msg;
    say_msg.reserve( rcsc::ServerParam::i().playerSayMsgSize() );

    bool reserved = false;

    if ( ! effector().getSayMessage().empty() )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            "say reserved message" );
        debugClient().addMessage( "Say[%s]",
                                  effector().getSayMessage().c_str() );
        reserved = true;
    }

    // ball Info: seen at current
    if ( ! reserved
         && world().ball().posCount() == 0
         && world().ball().velCount() == 0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                      "say chance" );
        if ( world().self().isKickable()  // I am kickable
             || ( ball_near_mates.empty()
                  || ( ball_near_mates[0]->distFromBall() >
                       world().ball().distFromSelf() - 3.0) )// I am closest to ball
             || ( ball_near_mates.size() > 1
                  && ( ball_near_mates[0]->distFromBall()  // closest to ball teammate is
                       > rcsc::ServerParam::i().visibleDistance() - 0.5 ) // over vis dist
                  && ( ball_near_mates[1]->distFromBall()
                       > world().ball().distFromSelf() - 5.0 ) ) ) // I am second
        {
            // say
            audioSensor().encodeBall( effector().queuedNextBallPos(),
                                      effector().queuedNextBallVel(),
                                      say_msg );
            if ( ! say_msg.empty() )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                              "say ball status" );
                debugClient().addMessage( "SayBall[%s]", say_msg.c_str() );
                doSay( say_msg );
                reserved = true;
            }
            else
            {
                std::cerr << config().teamName() << ": "
                          << world().self().unum()
                          << " Failed to encode ball info"
                          << std::endl;
            }
        }
    }

    // goalie info: ball is in chance area
    if ( ! reserved
         && world().ball().pos().x > 34.0
         && world().ball().pos().absY() < 25.0 )
    {
        // goalie is seen
        const rcsc::PlayerObject * opp_goalie = world().getOpponentGoalie();
        if ( opp_goalie
             && opp_goalie->posCount() == 0
             && opp_goalie->unum() > 0
             && opp_goalie->distFromSelf() < 25.0 )
        {
            audioSensor().encodeGoalie( opp_goalie->unum(),
                                        opp_goalie->pos(),
                                        say_msg );
            if ( ! say_msg.empty() )
            {
                rcsc::dlog.addText( rcsc::Logger::TEAM,
                                    "say goalie info to %d (%.1f %.1f)",
                                    opp_goalie->unum(),
                                    opp_goalie->pos().x,
                                    opp_goalie->pos().y );
                debugClient().addMessage( "SayGoalie[%s]",
                                          say_msg.c_str() );
                doSay( say_msg );
                reserved = true;
            }
            else
            {
                std::cerr << config().teamName() << ": "
                          << world().self().unum()
                          << " Failed to encode goalie info"
                          << std::endl;
            }
        }
    }


    // attention

    if ( ball_near_mates.empty() )
    {
        if ( world().self().attentiontoUnum() > 0 )
        {
            doAttentiontoOff();
        }
    }
    else if ( ball_near_mates.front()->distFromBall() >
              world().ball().distFromSelf() ) // I am closest
    {
        // if I am closest to ball, set attention to self nearest teammate
        if ( ! world().getTeammatesFromSelf().empty() )
        {
            doAttentionto( world().getOurSide(),
                           world().getTeammatesFromSelf().front()->unum() );
        }
    }
    else
    {
        // set attention to ball nearest teammate
        doAttentionto( world().getOurSide(),
                       ball_near_mates.front()->unum() );
    }
}
