// -*-c++-*-

/*!
  \file strategy.cpp
  \brief team strategh Source File
*/

/*
 *Copyright:

  Copyright (C) Hidehisa AKIYAMA

  This code is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2, or (at your option)
  any later version.

  This code is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this code; see the file COPYING.  If not, write to
  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cstdio>
#include <iostream>
#include <fstream>
#include <set>

#include <rcsc/game_mode.h>
#include <rcsc/param/server_param.h>

#include <rcsc/player/intercept_table.h>
#include <rcsc/player/world_model.h>
#include <rcsc/player/logger.h>

#include "role_sample.h"

#include "role_goalie.h"
#include "role_side_back.h"
#include "role_center_back.h"
#include "role_defensive_half.h"
#include "role_offensive_half.h"
#include "role_side_forward.h"
#include "role_center_forward.h"

#include "soccer_role.h"
#include "formation.h"
#include "formation_uva.h"

#include "strategy.h"

const std::string Strategy::BEFORE_KICK_OFF_CONF = "before-kick-off.conf";
const std::string Strategy::GOAL_KICK_OPP_CONF = "goal-kick-opp.conf";
const std::string Strategy::GOAL_KICK_OUR_CONF = "goal-kick-our.conf";
const std::string Strategy::GOALIE_CATCH_OPP_CONF = "goalie-catch-opp.conf";
const std::string Strategy::GOALIE_CATCH_OUR_CONF = "goalie-catch-our.conf";
const std::string Strategy::KICKIN_OUR_FORMATION_CONF = "kickin-our-formation.conf";
const std::string Strategy::DEFENSE_FORMATION_CONF = "defense-formation.conf";
const std::string Strategy::OFFENSE_FORMATION_CONF = "offense-formation.conf";

/*-------------------------------------------------------------------*/
/*!

*/
Strategy::Strategy()
    : M_before_kick_off_pos( 11 )
    , M_goal_kick_opp_pos( 11 )
    , M_goal_kick_our_pos( 11 )
    , M_goalie_catch_opp_pos( 11 )
    , M_goalie_catch_our_pos( 11 )
{
    M_role_factory[RoleSample::name()] = &RoleSample::create;

    M_role_factory[RoleGoalie::name()] = &RoleGoalie::create;
    M_role_factory[RoleCenterBack::name()] = &RoleCenterBack::create;
    M_role_factory[RoleSideBack::name()] = &RoleSideBack::create;
    M_role_factory[RoleDefensiveHalf::name()] = &RoleDefensiveHalf::create;
    M_role_factory[RoleOffensiveHalf::name()] = &RoleOffensiveHalf::create;
    M_role_factory[RoleSideForward::name()] = &RoleSideForward::create;
    M_role_factory[RoleCenterForward::name()] = &RoleCenterForward::create;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::read( const std::string & config_dir )
{
    std::string configpath = config_dir;
    if ( ! configpath.empty()
         && configpath[ configpath.length() - 1 ] != '/' )
    {
        configpath += '/';
    }

    // before kick off
    if ( ! readStaticPositions( configpath + BEFORE_KICK_OFF_CONF,
                                M_before_kick_off_pos ) )
    {
        return false;
    }
    // opponent goal kick
    if ( ! readStaticPositions( configpath + GOAL_KICK_OPP_CONF,
                                M_goal_kick_opp_pos ) )
    {
        return false;
    }
    // our goal kick
    if ( ! readStaticPositions( configpath + GOAL_KICK_OUR_CONF,
                                M_goal_kick_our_pos ) )
    {
        return false;
    }
    // opponent goalie catch
    if ( ! readStaticPositions( configpath + GOALIE_CATCH_OPP_CONF,
                                M_goalie_catch_opp_pos) )
    {
        return false;
    }
    // our goalie catch
    if ( ! readStaticPositions( configpath + GOALIE_CATCH_OUR_CONF,
                                M_goalie_catch_our_pos ) )
    {
        return false;
    }

    ///////////////////////////////////////////////////////////
    if ( ! readOffenseFormation( configpath ) )
    {
        std::cerr << "Failed to read offense formation" << std::endl;
        return false;
    }
    if ( ! readDefenseFormation( configpath ) )
    {
        std::cerr << "Failed to read offense formation" << std::endl;
        return false;
    }
    if ( ! readKickInOurFormation( configpath ) )
    {
        std::cerr << "Failed to read kickin our formation" << std::endl;
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
boost::shared_ptr< Formation >
Strategy::readFormation( const std::string & filepath )
{
    std::ifstream fin( filepath.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** failed to open file [" << filepath << "]"
                  << std::endl;
        return boost::shared_ptr< Formation >( static_cast< Formation * >( 0 ) );
    }

    boost::shared_ptr< Formation > ptr( new Formation );
    if ( ! ptr->read( fin ) )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** failed to read formation [" << filepath << "]"
                  << std::endl;
        return boost::shared_ptr< Formation >( static_cast< Formation * >( 0 ) );
    }

    for ( int unum = 1; unum <= 11; ++unum )
    {
        boost::shared_ptr< const FormationParam > fparam = ptr->param( unum );
        if ( ! fparam )
        {
            std::cerr << __FILE__ << ": " << __LINE__
                      << " ***ERROR*** detected Null formation param in ["
                      << filepath << "]"
                      << std::endl;
            return boost::shared_ptr< Formation >( static_cast< Formation * >( 0 ) );
        }
        if ( M_role_factory.find( fparam->getRoleName() ) == M_role_factory.end() )
        {
            std::cerr << __FILE__ << ": " << __LINE__
                      << " ***ERROR*** Unsupported role name ["
                      << fparam->getRoleName() << "] is included in in ["
                      << filepath << "]" << std::endl;
            return boost::shared_ptr< Formation >( static_cast< Formation * >( 0 ) );
        }
    }

    return ptr;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::readStaticPositions( const std::string & filepath,
                               std::vector< rcsc::Vector2D > & positions )
{
    std::ifstream fin( filepath.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** Failed to open configuration file ["
                  << filepath << "]"
                  << std::endl;
        return false;
    }

    positions.resize( 11 );

    std::string line_buf;
    int n_line = 0;
    std::set< int > read_number;
    while ( std::getline( fin, line_buf ) )
    {
        ++n_line;
        if ( line_buf.empty()
             || line_buf[0] == '#'
             || ( line_buf.length() > 1
                  && line_buf[0] == '/'
                  && line_buf[1] == '/' )
             )
        {
            continue;
        }

        int number;
        double x, y;
        if ( std::sscanf( line_buf.c_str(),
                          " %d %lf %lf ",
                          &number, &x, &y ) != 3 )
        {
            std::cerr << __FILE__ << ": " << __LINE__
                      << " ***ERROR*** Invalid line in ["
                      << filepath << "]:" << n_line
                      << std::endl;
            fin.close();
            return false;
        }

        if ( number < 0 || 11 < number )
        {
            std::cerr << __FILE__ << ": " << __LINE__
                      << " ***ERROR*** Invalid player number in ["
                      << filepath << ":" << n_line << "]"
                      << std::endl;
            fin.close();
            return false;
        }

        read_number.insert( number );
        positions[number - 1].assign( x, y );
    }

    fin.close();

    if ( read_number.size() != 11 )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** Invalid number of configuration in ["
                  << filepath << ":" << n_line << "]"
                  << std::endl;
        return false;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::readBeforeKickOffPos( const std::string & configpath )
{
    std::vector< rcsc::Vector2D > positions;
    if ( ! readStaticPositions( configpath + BEFORE_KICK_OFF_CONF,
                                positions )  )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** faled to read BeforeKickOffPos(1)" << std::endl;
        return false;
    }
    if ( positions.size() != 11 )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** faled to read BeforeKickOffPos(2)" << std::endl;
        return false;
    }
    for ( int i = 0; i < 11; ++i )
    {
        M_before_kick_off_pos[i] = positions[i];
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::readGoalKickOppPos( const std::string & configpath )
{
    std::vector< rcsc::Vector2D > positions;
    if ( ! readStaticPositions( configpath + GOAL_KICK_OPP_CONF,
                                positions )  )
    {
        std::cerr << "***ERROR*** faled to read GoalKickOppPos(1)" << std::endl;
        return false;
    }
    if ( positions.size() != 11 )
    {
        std::cerr << "***ERROR*** faled to read GoalKickOppPos(2)" << std::endl;
        return false;
    }
    for ( int i = 0; i < 11; ++i )
    {
        M_goal_kick_opp_pos[i] = positions[i];
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::readGoalieCatchOurPos( const std::string & configpath )
{
    std::vector< rcsc::Vector2D > positions;
    if ( ! readStaticPositions( configpath + GOALIE_CATCH_OUR_CONF,
                                positions )  )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** faled to read GoalieCatchOppPos(1)" << std::endl;
        return false;
    }
    if ( positions.size() != 11 )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** faled to read GoalieCatchppPos(2)" << std::endl;
        return false;
    }
    for ( int i = 0; i < 11; ++i )
    {
        //std::cerr << i + 1 << " " << positions[i] << std::endl;
        M_goalie_catch_our_pos[i] = positions[i];
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::readGoalieCatchOppPos( const std::string & configpath )
{
    std::vector< rcsc::Vector2D > positions;
    if ( ! readStaticPositions( configpath + GOALIE_CATCH_OPP_CONF,
                                positions )  )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** faled to read GoalieCatchOppPos(1)" << std::endl;
        return false;
    }
    if ( positions.size() != 11 )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** faled to read GoalieCatchppPos(2)" << std::endl;
        return false;
    }
    for ( int i = 0; i < 11; ++i )
    {
        M_goalie_catch_opp_pos[i] = positions[i];
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!
  create role instance
*/
boost::shared_ptr< SoccerRole >
Strategy::createRole( const int number,
                      const rcsc::WorldModel & world ) const
{
    if ( number < 1 || 11 < number )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** Invalid player number " << number
                  << std::endl;
        return boost::shared_ptr< SoccerRole >( static_cast< SoccerRole * >( 0 ) );
    }

    boost:: shared_ptr< const Formation > formation = getFormation( world );
    if ( ! formation )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** faled to create role. Null formation" << std::endl;
        return boost::shared_ptr< SoccerRole >( static_cast< SoccerRole * >( 0 ) );
    }

    const boost::shared_ptr< const FormationParam > fparam
        = formation->param( number );
    if ( ! fparam )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** faled to create role. Null formation param"
                  << std::endl;
        return boost::shared_ptr< SoccerRole >( static_cast< SoccerRole * >( 0 ) );
    }


    ///// delete here, if you do not use SBSP formation ////////
    if ( uva::Formations::instance().valid() )
    {
        uva::PlayerT type = uva::Formations::instance()
            .getPlayerType( number - 1, uva::FT_433_OFFENSIVE );
        switch ( type ) {
        case uva::PT_GOALKEEPER:
            {
                boost::shared_ptr< SoccerRole > role_ptr( RoleGoalie::create( formation ) );
                return role_ptr;
            }
            break;
        case uva::PT_DEFENDER_CENTRAL:
            {
                boost::shared_ptr< SoccerRole > role_ptr( RoleCenterBack::create( formation ) );
                return role_ptr;
            }
            break;
        case uva::PT_DEFENDER_SWEEPER:
            {
                boost::shared_ptr< SoccerRole > role_ptr( RoleCenterBack::create( formation ) );
                return role_ptr;
            }
            break;
        case uva::PT_DEFENDER_WING:
            {
                boost::shared_ptr< SoccerRole > role_ptr( RoleSideBack::create( formation ) );
                return role_ptr;
            }
            break;
        case uva::PT_MIDFIELDER_CENTER:
            {
                boost::shared_ptr< SoccerRole > role_ptr( RoleDefensiveHalf::create( formation ) );
                return role_ptr;
            }
            break;
        case uva::PT_MIDFIELDER_WING:
            {
                boost::shared_ptr< SoccerRole > role_ptr( RoleOffensiveHalf::create( formation ) );
                return role_ptr;
            }
            break;
        case uva::PT_ATTACKER_WING:
            {
                boost::shared_ptr< SoccerRole > role_ptr( RoleSideForward::create( formation ) );
                return role_ptr;
            }
            break;
        case uva::PT_ATTACKER:
            {
                boost::shared_ptr< SoccerRole > role_ptr( RoleCenterForward::create( formation ) );
                return role_ptr;
            }
            break;
        default:
            return boost::shared_ptr< SoccerRole >( static_cast< SoccerRole * >( 0 ) );
            break;
        }

    }
    /////////////////////////////////////////////////////////


    const std::string & rolename = fparam->getRoleName();

    // get role factory of this role name
    std::map< std::string, RoleCreator >::const_iterator
        factory = M_role_factory.find( rolename );
    if ( factory == M_role_factory.end() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** No such a role name ["
                  << rolename << "]"
                  << std::endl;
        return boost::shared_ptr< SoccerRole >( static_cast< SoccerRole * >( 0 ) );
    }

    boost::shared_ptr< SoccerRole > role_ptr( factory->second( formation ) );
    return role_ptr;
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Strategy::getBeforeKickOffPos( const int number ) const
{
    if ( number < 1 || 11 < number )
    {

        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** Invalid player number " << number
                  << std::endl;
        return rcsc::Vector2D( -50.0, -30.0 );
    }

    return M_before_kick_off_pos[number - 1];
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Strategy::getPosWhenGoalKickOpp( const int number ) const
{
    if ( number < 1 || 11 < number )
    {

        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** Invalid player number " << number
                  << std::endl;
        return rcsc::Vector2D( -50.0, -30.0 );
    }

    return M_goal_kick_opp_pos[number - 1];
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Strategy::getPosWhenGoalKickOur( const int number ) const
{
    if ( number < 1 || 11 < number )
    {

        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** Invalid player number " << number
                  << std::endl;
        return rcsc::Vector2D( -50.0, -30.0 );
    }
    return M_goal_kick_our_pos[number - 1];
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Strategy::getPosWhenGoalieCatchOpp( const int number ) const
{
    if ( number < 1 || 11 < number )
    {

        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** Invalid player number " << number
                  << std::endl;
        return rcsc::Vector2D( -50.0, -30.0 );
    }
    return M_goalie_catch_opp_pos[number - 1];
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Strategy::getPosWhenGoalieCatchOur( const int number ) const
{
    if ( number < 1 || 11 < number )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " ***ERROR*** Invalid player number " << number
                  << std::endl;
        return rcsc::Vector2D( -50.0, -30.0 );
    }
    return M_goalie_catch_our_pos[number - 1];
}

/*-------------------------------------------------------------------*/
/*!

*/
rcsc::Vector2D
Strategy::getSetPlayPosition( const int number,
                              const Formation & formation,
                              const rcsc::WorldModel & world ) const
{
    if ( world.getGameMode().type() == rcsc::GameMode::GoalKick_ )
    {
        if ( world.getGameMode().side() == world.getOurSide() )
        {
            return getPosWhenGoalKickOur( number );
        }
        if ( world.getGameMode().side() != world.getOurSide() )
        {
            return getPosWhenGoalKickOpp( number );
        }
    }
    if ( world.getGameMode().type() == rcsc::GameMode::GoalieCatch_ )
    {
        if ( world.getGameMode().side() == world.getOurSide() )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                "%s: Strategy. get our galie catch mode position"
                                ,__FILE__ );
            return getPosWhenGoalieCatchOur( number );
        }
        if ( world.getGameMode().side() != world.getOurSide() )
        {
            return getPosWhenGoalieCatchOpp( number );
        }
    }

    rcsc::Vector2D base_pos( 0.0, 0.0 );
    if ( ! world.ball().posValid() )
    {
        return formation.getPosition( number, base_pos );
    }


    base_pos = world.ball().pos();


    bool our_setplay = false;
    switch ( world.getGameMode().type() ) {
    case rcsc::GameMode::KickOff_:
    case rcsc::GameMode::KickIn_:
    case rcsc::GameMode::CornerKick_:
    case rcsc::GameMode::GoalKick_:
    case rcsc::GameMode::FreeKick_:
    case rcsc::GameMode::GoalieCatch_:
    case rcsc::GameMode::IndFreeKick_:
        if ( world.getGameMode().side() == world.getOurSide() )
        {
            our_setplay = true;
        }
        break;
    case rcsc::GameMode::OffSide_:
    case rcsc::GameMode::FreeKickFault_:
    case rcsc::GameMode::BackPass_:
    case rcsc::GameMode::CatchFault_:
        if ( world.getGameMode().side() != world.getOurSide() )
        {
            our_setplay = true;
        }
        break;
    default:
        break;
    }

    if ( world.getGameMode().type() == rcsc::GameMode::GoalieCatch_
         && world.getGameMode().side() == world.getOurSide() )
    {
        base_pos.x = -40.0;
        base_pos.y = 0.0;
    }
    else if ( world.getGameMode().type() == rcsc::GameMode::GoalKick_
              && world.getGameMode().side() == world.getOurSide() )
    {
        base_pos.x = -40.0;
        base_pos.y = 0.0;
    }
    else if ( ! our_setplay )
    {
        base_pos.x = std::max( -41.0, base_pos.x - 15.0 );
        base_pos.y = std::min( rcsc::ServerParam::i().pitchHalfLength(),
                               base_pos.y * 1.1 );

    }

    rcsc::Vector2D home_pos = formation.getPosition( number, base_pos );

    // to onside
    if ( rcsc::ServerParam::i().useOffside() )
    {
        if ( world.getGameMode().type() != rcsc::GameMode::KickIn_
             || world.getGameMode().side() != world.getOurSide() )
        {
            home_pos.x = std::min( home_pos.x, world.getOffsideLineX() - 0.5 );
        }
    }

    return home_pos;
}

/*-------------------------------------------------------------------*/
/*!

*/
boost:: shared_ptr< const Formation >
Strategy::getFormation( const rcsc::WorldModel & world ) const
{
    if ( world.getGameMode().type() == rcsc::GameMode::KickIn_ )
    {
        if ( world.getOurSide() == world.getGameMode().side() )
        {
            return M_kickin_our_formation;
        }
    }

    if ( world.ball().pos().x > 0.0 )
    {
        return M_offense_formation;
    }
    else
    {
        return M_defense_formation;
    }

#if 0
    if ( world.getGameMode().type() == GameMode::KickIn_ )
    {
        if ( world.getOurSide() == world.getGameMode().side() )
        {
            return M_kickin_our;
        }
        else
        {
            return M_kickin_opponent;
        }
    }
#endif


#if 0
    if ( world.ball().pos().x > 20.0 )
    {
        return M_attack_area_offense;
    }
    else if ( world.ball().pos().x > -20.0 )
    {
        return M_middle_area_offense;
    }
    else
    {
        return M_block_area_defense;
    }
#endif
#if 0
    const InterceptTable * table = world.getInterceptTable();
    //const int self_min = table->getSelfReachCycle();
    const int mate_min = table->getTeammateReachCycle();
    const int opp_min = table->getOpponentReachCycle();

    // other playmode including playon
    if ( world.ball().pos().x > 25.0 )
    {
        if ( world.existKickableTeammate()
             || mate_min < opp_min )
        {
            return M_attack_area_offense;
        }

        return M_attack_area_defense;
    }

    if ( world.ball().pos().x > -20.0 )
    {
        if ( ( world.existKickableTeammate()
               && ! world.existKickableOpponent() )
             || mate_min < opp_min - 3 )
        {
            return M_middle_area_offense;
        }

        return M_middle_area_defense;
    }

    // block area

    {
        if ( ( world.existKickableTeammate()
               && ! world.existKickableOpponent() )
             || mate_min < opp_min - 5 )
        {
            return M_block_area_offense;
        }

    }
    return M_block_area_defense;
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::readOffenseFormation( const std::string & configpath )
{
    std::string filepath = configpath + OFFENSE_FORMATION_CONF;

    M_offense_formation = readFormation( filepath );

    return ( M_offense_formation );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::readDefenseFormation( const std::string & configpath )
{
    std::string filepath = configpath + DEFENSE_FORMATION_CONF;

    M_defense_formation = readFormation( filepath );

    return ( M_defense_formation );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Strategy::readKickInOurFormation( const std::string & configpath )
{
    std::string filepath = configpath + KICKIN_OUR_FORMATION_CONF;

    M_kickin_our_formation = readFormation( filepath );

    return ( M_kickin_our_formation );
}

/*-------------------------------------------------------------------*/
/*!

*/
Strategy::BallArea
Strategy::get_ball_area( const rcsc::Vector2D & ball_pos )
{
    //if (ball_pos.x > 41.5)
    if ( ball_pos.x > 39.0 )
    {
        if ( ball_pos.absY() > 17.0 )
        {
            return BA_Cross;
        }
        else
        {
            return BA_ShootChance;
        }
    }
    else if ( ball_pos.x > 34.0 )
    {
        if ( ball_pos.absY() > 20.0 )
        {
            return BA_DribbleAttack;
        }
        else
        {
            return  BA_ShootChance;
        }
    }
    else if ( ball_pos.x > -1.0 )
    {
        if ( ball_pos.absY() > 18.0 )
        {
            return BA_DribbleAttack;
        }
        else
        {
            return BA_OffMidField;
        }
    }
    else if ( ball_pos.x > -25.0 )
    {
        if ( ball_pos.absY() > 18.0 )
        {
            return BA_DribbleBlock;
        }
        else
        {
            return BA_DefMidField;
        }
    }
    else if ( ball_pos.x > -34.0 )
    {
        if ( ball_pos.absY() > 18.0 )
        {
            return BA_DribbleBlock;
        }
        else
        {
            return BA_Stopper;
        }
    }
    else if ( ball_pos.x > -36.5 )
    {
        if ( ball_pos.absY() > 20.0 )
        {
            return BA_CrossBlock;
        }
        else
        {
            return BA_Stopper;
        }
    }
    else
    {
        if ( ball_pos.absY() > 20.0 )
        {
            return BA_CrossBlock;
        }
        else
        {
            return BA_Danger;
        }
    }

    return BA_None;
}
