// -*-c++-*-

/*!
  \file body_shoot.cpp
  \brief advanced shoot planning and behavior.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <rcsc/math_util.h>
#include <rcsc/soccer_math.h>
#include <rcsc/geom/rect_2d.h>
#include <rcsc/param/server_param.h>

#include <rcsc/player/logger.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>

#include <rcsc/player/interception.h>

#include "body_hold_ball.h"
#include "body_kick_one_step.h"
#include "body_kick_two_step.h"
#include "body_kick_multi_step.h"
#include "body_stop_ball.h"
#include "intention_kick.h"

#include "body_shoot.h"

namespace rcsc {

GameTime Body_Shoot::S_last_calc_time( 0, 0 );
std::vector< Body_Shoot::ShootRoute > Body_Shoot::S_cached_shoot_route;

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Body_Shoot::execute( PlayerAgent* agent )
{
    dlog.addText( Logger::TEAM,
                  "%s:%d: Body_Shoot"
                  ,__FILE__, __LINE__ );

    if ( ! agent->world().self().isKickable() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " not ball kickable!"
                  << std::endl;
        dlog.addText( Logger::ACTION,
                      "%s:%d:  not kickable"
                      ,__FILE__, __LINE__ );
        return false;
    }

    // update
    search( agent );

    if ( S_cached_shoot_route.empty() )
    {
        dlog.addText( Logger::TEAM,
                      "%s:%d: not found shoot route"
                      ,__FILE__, __LINE__ );
        return false;
    }


    // it is necessary to evaluate shoot courses

    Vector2D target_point = S_cached_shoot_route.front().target_point;
    double first_speed = S_cached_shoot_route.front().first_speed;

    agent->debugClient().addMessage( "Shoot" );
    agent->debugClient().setTarget( target_point );

    Vector2D one_step_vel
        = Body_KickOneStep::get_max_possible_vel( ( target_point - agent->world().ball().pos() ).th(),
                                                  agent->world().self().kickRate(),
                                                  agent->world().ball().vel() );

    dlog.addText( Logger::TEAM,
                  "%s:%d: shoot to (%.2f, %.2f)  one_kick_max_speed= %.2f"
                  ,__FILE__, __LINE__,
                  target_point.x, target_point.y,
                  one_step_vel.r() );
    /*
    if ( one_step_vel.r() > first_speed * 0.9 )
    {
        Body_KickOneStep( target_point,
                          ServerParam::i().ballSpeedMax()
                          ).execute( agent );
        agent->debugClient().addMessage( "Enforce1Step" );
        return true;
    }
    else 
    */
    if ( agent->world().self().pos().x > 45.0 )
    {
        Body_KickTwoStep( target_point,
                          ServerParam::i().ballSpeedMax(),
                          true // enforce
                          ).execute( agent );
        agent->debugClient().addMessage( "Enforce2Step" );
        return true;
    }

    Body_KickMultiStep( target_point, first_speed ).execute( agent );

    bool enforce_kick = false;
    if ( agent->world().self().pos().x > 45.0
         && ( agent->world().self().pos().absY()
              < ServerParam::i().goalHalfWidth() + 3.0 )
         )
    {
        enforce_kick = true;
    }

    agent->debugClient().addLine( agent->world().ball().pos(), target_point );

    agent->setIntention
        ( new IntentionKick( target_point,
                             first_speed,
                             2,
                             enforce_kick,
                             agent->world().time() ) );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Body_Shoot::search( const PlayerAgent * agent )
{
    /////////////////////////////////////////////////////////////////////
    static const Rect2D
        shootable_area( ServerParam::i().theirPenaltyAreaLine() - 5.0, // left
                        -ServerParam::i().penaltyAreaHalfWidth(), // top
                        ServerParam::i().penaltyAreaLength() + 5.0, // length
                        ServerParam::i().penaltyAreaWidth() ); // width
    static const Vector2D left_goal_post( ServerParam::i().pitchHalfLength(),
                                          -ServerParam::i().goalHalfWidth() + 1.0 );
    static const Vector2D right_goal_post( ServerParam::i().pitchHalfLength(),
                                           ServerParam::i().goalHalfWidth() - 1.0 );
    static const Line2D goal_line( left_goal_post, right_goal_post );

    /////////////////////////////////////////////////////////////////////

    if ( S_last_calc_time == agent->world().time() )
    {
        return;
    }

    // reset
    S_last_calc_time = agent->world().time();
    S_cached_shoot_route.clear();

    /////////////////////////////////////////////////////////////////////

    if ( ! agent->world().self().isKickable()
         || ! shootable_area.contains( agent->world().ball().pos() ) )
    {
        return;
    }


    /////////////////////////////////////////////////////////////////////
    // check last executing shoot route


    /////////////////////////////////////////////////////////////////////


    const AngleDeg left_angle = ( left_goal_post - agent->world().ball().pos() ).th();
    const AngleDeg right_angle = ( right_goal_post - agent->world().ball().pos() ).th();

    const double scan_angle_range = (right_angle - left_angle).abs();

    AngleDeg tmp_angle;

    int scan_loop;
    double scan_inc;
    if ( scan_angle_range < 2.0 )
    {
        scan_inc = 0.0;
        scan_loop = 1;
        tmp_angle = AngleDeg::bisect( left_angle, right_angle );
    }
    else if ( scan_angle_range < 16.0 )
    {
        scan_inc = 2.0;
        scan_loop = static_cast< int >( scan_angle_range / scan_inc );
        tmp_angle
            = left_angle
            + ( scan_angle_range - std::floor( scan_angle_range ) ) * 0.5;
    }
    else
    {
        scan_inc = scan_angle_range / 8.0;
        scan_loop = 8;
        tmp_angle = left_angle;
    }

    const PlayerPtrCont & opps = agent->world().opponentsFromSelf();
    const PlayerPtrCont::const_iterator opps_end = opps.end();

    int goalie_conf = 1000;
    const PlayerObject * opp_goalie = agent->world().getOpponentGoalie();
    if ( opp_goalie )
    {
        goalie_conf = opp_goalie->posCount();
    }

    // search loop for each angle
    for ( int i = 0; i < scan_loop; i++ )
    {
        if ( goalie_conf > 5
             && agent->world().getDirCount( tmp_angle ) > 3 )
        {
            tmp_angle += scan_inc;
            continue;
        }

        Line2D ball_line( agent->world().ball().pos(), tmp_angle );
        Vector2D reach_point( 52.5, 0.0 );
        if ( ! ball_line.intersection( goal_line, &reach_point ) )
        {
            std::cerr << agent->world().time()
                      << "Why shoot cource does not intersect with goal line??"
                      << std::endl;
            tmp_angle += scan_inc;
            continue;
        }
#ifdef DEBUG
        dlog.addText( Logger::SHOOT,
                      "__Shoot to(%.1f %.1f) angle=%.1f",
                      reach_point.x, reach_point.y,  tmp_angle.degree() );
#endif
        double dist2goal = agent->world().ball().pos().dist( reach_point );

        Vector2D max_one_step_vel
            = Body_KickOneStep::get_max_possible_vel
            ( tmp_angle,
              agent->world().self().kickRate(),
              agent->world().ball().vel() );

        double shoot_first_speed
            = ( dist2goal + 5.0 ) * ( 1.0 - ServerParam::i().ballDecay() );
        shoot_first_speed = std::max( max_one_step_vel.r(), shoot_first_speed );
        shoot_first_speed = std::max( 1.5, shoot_first_speed );

        bool over_max = false;
        while ( ! over_max )
        {
            if ( shoot_first_speed > ServerParam::i().ballSpeedMax() )
            {
                over_max = true;
                shoot_first_speed = ServerParam::i().ballSpeedMax();
            }

            int min_cycle = get_opponent_cycle( agent,
                                                tmp_angle, shoot_first_speed,
                                                ball_line, dist2goal );

            if ( min_cycle > 0 )
            {
#ifdef DEBUG
                dlog.addText( Logger::SHOOT,
                              "____ Success! to(%.1f %.1f).speed=%.2f  opp_min_cycle= %d",
                              reach_point.x, reach_point.y,
                              shoot_first_speed, min_cycle );
#endif
                S_cached_shoot_route.push_back( ShootRoute( reach_point,
                                                            shoot_first_speed,
                                                            tmp_angle,
                                                            min_cycle ) );
            }

            shoot_first_speed += 0.5; // Magic Number: speed inc
        }

        tmp_angle += scan_inc;
    }

#ifdef DEBUG
    dlog.addText( Logger::SHOOT,
                  "__total shoot course is %d",
                  S_cached_shoot_route.size() );

#endif

    // sort by first speed
    std::sort( S_cached_shoot_route.begin(),
               S_cached_shoot_route.end(),
               ShootRouteSpeedComp() );

    // merge sort by opponent reach cycle
    //std::stable_sort( S_cached_shoot_route.begin(),
    //                  S_cached_shoot_route.end(),
    //                  ShootRouteCycleComp() );

    const PlayerObject * goalie = agent->world().getOpponentGoalie();
    if ( goalie )
    {
        AngleDeg angle = ( agent->world().ball().pos()
                           - goalie->pos() ).th();
        std::stable_sort( S_cached_shoot_route.begin(),
                          S_cached_shoot_route.end(),
                          ShootRouteAngleCmp( angle ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
int
Body_Shoot::get_opponent_cycle( const PlayerAgent * agent,
                                const AngleDeg & shoot_angle,
                                const double & first_speed,
                                const Line2D & shoot_line,
                                const double & dist_to_goal )
{
    static const double opp_x_thr = ServerParam::i().theirPenaltyAreaLine() - 5.0;
    static const double opp_y_thr = ServerParam::i().penaltyAreaHalfWidth();
    static const double goalie_max_speed
        = ServerParam::i().defaultRealSpeedMax() * 0.9;
    static const double player_max_speed
        = ServerParam::i().defaultPlayerSpeedMax() * 0.8;
    static const double player_control_area
        = ServerParam::i().defaultKickableArea() - 0.3;

    // estimate required ball travel step
    const double ball_reach_step
        = calc_length_geom_series( first_speed,
                                   dist_to_goal,
                                   ServerParam::i().ballDecay() )
        - 1.0;

    if ( ball_reach_step < 1.0 )
    {
        dlog.addText( Logger::SHOOT,
                      "____ Success! not need to check opponent."
                      " speed = %.2f  ball reach step = %.1f",
                      first_speed, ball_reach_step );
        return 1000;
    }

#ifdef DEBUG
    dlog.addText( Logger::SHOOT,
                  "____ speed = %.2f  ball reach step = %.1f",
                  first_speed, ball_reach_step );
#endif
    // estimate opponent interception
    const Interception util( agent->world().ball().pos()
                             + Vector2D::polar2vector( first_speed, shoot_angle ),
                             first_speed * ServerParam::i().ballDecay(),
                             shoot_angle );

    int min_cycle = 1000;
    const PlayerPtrCont::const_iterator end = agent->world().opponentsFromSelf().end();
    for ( PlayerPtrCont::const_iterator it = agent->world().opponentsFromSelf().begin();
          it != end;
          ++it )
    {
        // outside of penalty
        if ( (*it)->pos().x < opp_x_thr ) continue;
        if ( (*it)->pos().absY() > opp_y_thr ) continue;
        // behind of shoot course
        if ( ( shoot_angle - (*it)->angleFromSelf() ).abs() > 90.0 )
        {
            continue;
        }

        int cycle = 1000;
        int conf_decay = 0;
        if ( (*it)->goalie() )
        {
            // goalie already exist on shoot line
            /*
            if ( (*it)->distFromSelf() < dist_to_goal + 1.0
                 && ( shoot_angle - (*it)->angleFromSelf() ).abs() < 90.0
                 && ( shoot_line.dist( (*it)->pos() )
                      < ServerParam::i().catchAreaLength() )
                 )
            {
                return -1;
            }
            */
            conf_decay = std::min( 5, (*it)->posCount() );

            cycle = static_cast< int >
                ( std::ceil( util.getReachCycle( (*it)->pos(),
                                                 &((*it)->vel()),
                                                 NULL,
                                                 (*it)->posCount(),
                                                 ServerParam::i().catchAreaLength(),
                                                 goalie_max_speed ) ) );
#ifdef DEBUG
            dlog.addText( Logger::SHOOT,
                          "______  goalie_cycle = %d.",
                          cycle );
#endif
        }
        else
        {
            cycle = static_cast< int >
                ( std::ceil( util.getReachCycle( (*it)->pos(),
                                                 &((*it)->vel()),
                                                 NULL,
                                                 0,
                                                 player_control_area,
                                                 player_max_speed ) ) );
            cycle += 2;
        }

        if ( cycle < min_cycle )
        {
            min_cycle = cycle;
        }

        // opponent can reach the ball 
        if ( cycle - conf_decay < ball_reach_step )
        {
#ifdef DEBUG
            dlog.addText( Logger::SHOOT,
                          "______ opponent can reach. cycle = %d",
                          cycle );
#endif
            return -1;
        }
    }

    return min_cycle;
}

}
