// -*-c++-*-

/*!
  \file body_shoot.h
  \brief advanced shoot planning and behavior.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_ACTION_BODY_SHOOT_H
#define RCSC_ACTION_BODY_SHOOT_H

#include <functional>
#include <vector>

#include <rcsc/game_time.h>
#include <rcsc/geom/vector_2d.h>
#include <rcsc/geom/line_2d.h>
#include <rcsc/player/soccer_action.h>

namespace rcsc {

/*!
  \class Body_Shoot
  \brief advanced shoot planning and behavior.
 */
class Body_Shoot
    : public BodyAction {
public:
    struct ShootRoute {
        Vector2D target_point;
        double first_speed;
        AngleDeg angle_;
        int opp_reach_min_cycle;
        ShootRoute( const Vector2D & point,
                    const double & speed,
                    const AngleDeg & angle,
                    const int min_cycle )
            : target_point( point )
            , first_speed( speed )
            , angle_( angle )
            , opp_reach_min_cycle( min_cycle )
          { }
    };

    ////////////////////////////////////////////////////////////
    /*!
      \class ShootRouteCycleComp
      \brief function object to compare the cycles that the ball reach the
      goal.
    */
    class ShootRouteCycleComp
        : public std::binary_function< ShootRoute,
                                       ShootRoute,
                                       bool > {
    public:
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              // !!! CAUTION: operator is greater<>.
              return lhs.opp_reach_min_cycle > rhs.opp_reach_min_cycle;
          }
    };

    ////////////////////////////////////////////////////////////
    /*!
      \class ShootRouteSpeedComp
      \brief function object to compare the shoot first speed.
    */
    class ShootRouteSpeedComp
        : public std::binary_function< ShootRoute,
                                       ShootRoute,
                                       bool > {
    public:
        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return lhs.first_speed < rhs.first_speed;
          }
    };

    ////////////////////////////////////////////////////////////
    class ShootRouteAngleCmp
        : public std::binary_function< ShootRoute,
                                       ShootRoute,
                                       bool > {
    private:
        const AngleDeg M_goalie_angle;
    public:
        ShootRouteAngleCmp( const AngleDeg & goalie_angle )
            : M_goalie_angle( goalie_angle )
          { }

        result_type operator()( const first_argument_type & lhs,
                                const second_argument_type & rhs ) const
          {
              return ( lhs.angle_ - M_goalie_angle ).abs()
                  <  ( rhs.angle_ - M_goalie_angle ).abs();
          }
    };

private:
    //! last calculated time
    static GameTime S_last_calc_time;
    //! cached calculated shoot routes
    static std::vector< ShootRoute > S_cached_shoot_route;

public:
    /*!
      \brief accessible from global.
     */
    Body_Shoot()
      { }

    /*!
      \brief execute action
      \param agent pointer to the agent itself
      \return true if action is performed
     */
    bool execute( PlayerAgent * agent );


    static
    const
    std::vector< ShootRoute > &
    get_cached_shoot_route( const PlayerAgent * agent )
      {
          search( agent );
          return S_cached_shoot_route;
      }

private:
    /*!
      \brief search shoot route patterns. goal mouth is divided to several
      segment and several ball speed for each segment are calculated.
      \param agent const pointer to the agent itself
     */
    static
    void search( const PlayerAgent * agent );

    /*!
      \brief calculate ball reach step for all opponent.
      \param agent const  pointer to the agent itself
      \param shoot_angle shoot route angle from current ball position
      \param first_speed shoot first speed
      \param shoot_line shoot route trace line
      \param dist_to_goal distance from the ball position to goal line
      \return minimal reach cycle value. if opponent *can* get the ball,
      negative value is returned.
     */
    static
    int get_opponent_cycle( const PlayerAgent * agent,
                            const AngleDeg & shoot_angle,
                            const double & first_speed,
                            const Line2D & shoot_line,
                            const double & dist_to_goal );

};

}

#endif
