// -*-c++-*-

/*!
  \file neck_scan_field.cpp
  \brief scan field with neck evenly
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <numeric>

#include <rcsc/geom/rect_2d.h>
#include <rcsc/param/server_param.h>

#include <rcsc/player/logger.h>
#include <rcsc/player/player_agent.h>

#include "basic_actions.h"

#include "neck_scan_field.h"

namespace rcsc {

GameTime Neck_ScanField::S_last_calc_time( 0, 0 );
ViewWidth Neck_ScanField::S_last_calc_view_width = ViewWidth::NORMAL;
AngleDeg Neck_ScanField::S_cached_target_angle( 0.0 );

/*-------------------------------------------------------------------*/
/*!
  scan field by turn_neck
*/
bool
Neck_ScanField::execute( PlayerAgent * agent )
{

    const AngleDeg body_next = agent->effector().queuedNextMyBody();

    if ( S_last_calc_time == agent->world().time()
         && S_last_calc_view_width == agent->effector().queuedNextViewWidth() )
    {
        return agent->doTurnNeck( S_cached_target_angle
                                  - body_next
                                  - agent->world().self().neck() );
    }

    S_last_calc_time = agent->world().time();
    S_last_calc_view_width = agent->effector().queuedNextViewWidth();

    dlog.addText( Logger::ACTION,
                  "%s:%d: Neck_ScanField"
                  ,__FILE__, __LINE__ );

    ViewWidth vwidth = agent->effector().queuedNextViewWidth();
    double next_view_width = vwidth.getWidth();

    const Vector2D my_next = agent->effector().queuedNextMyPos();

    const AngleDeg next_limit_min
        = body_next
        + ( ServerParam::i().minNeckAngle() - (next_view_width * 0.5 + 1.0) );
    const double next_neck_range
        = ( ( ServerParam::i().maxNeckAngle() - ServerParam::i().minNeckAngle() )
            + next_view_width - 2.0 );

    dlog.addText( Logger::ACTION,
                  "%s:%d: next_limit_min= %f, next_neck_range= %f"
                  ,__FILE__, __LINE__,
                  next_limit_min.degree(), next_neck_range );

    S_cached_target_angle = calcAngle( agent,
                                       next_limit_min,
                                       next_neck_range,
                                       my_next,
                                       next_view_width );
    dlog.addText( Logger::ACTION,
                  "%s:%d: neck to = %f"
                  ,__FILE__, __LINE__,
                  S_cached_target_angle.degree() );

    return agent->doTurnNeck( S_cached_target_angle
                              - body_next
                              - agent->world().self().neck() );
}

/*-------------------------------------------------------------------*/
/*!
  static
*/
AngleDeg
Neck_ScanField::calcAngle( const PlayerAgent * agent,
                           const AngleDeg & left_start,
                           const double & scan_range,
                           const Vector2D & base_point,
                           const double & next_view_width )
{
    static const Rect2D pitch_rect( -ServerParam::i().pitchHalfLength(),
                                    -ServerParam::i().pitchHalfWidth(),
                                    ServerParam::i().pitchLength(),
                                    ServerParam::i().pitchWidth() );
    static const Rect2D goalie_rect( ServerParam::i().pitchHalfLength() - 3.0,
                                     -15.0,
                                     10.0,
                                     30.0 );
    dlog.addText( Logger::ACTION,
                  "%s:%d: get_lowest_conf_angle"
                  ,__FILE__, __LINE__ );

    AngleDeg max_count_angle = left_start + scan_range * 0.5;

    if ( scan_range < next_view_width )
    {
        return max_count_angle;
    }


    AngleDeg tmp_angle = left_start;

    const std::deque< int >::size_type vwidth_include_spans
        = static_cast< std::deque< int >::size_type >
        ( rint( next_view_width / WorldModel::DIR_STEP ) );

    std::deque< int > dir_counts( vwidth_include_spans );

    // generate first visible cone socre list
    const std::deque< int >::iterator it_end = dir_counts.end();
    for ( std::deque< int >::iterator it = dir_counts.begin();
          it != it_end;
          ++it )
    {
        *it = agent->world().getDirCount( tmp_angle );
        tmp_angle += WorldModel::DIR_STEP;
    }



    int max_count = 0;
    double add_dir = next_view_width;

    do
    {
        int dir_count_sum = std::accumulate( dir_counts.begin(), dir_counts.end(), 0 );

        if ( dir_count_sum > max_count )
        {
            AngleDeg candidate = tmp_angle - next_view_width * 0.5;
            Vector2D face_point
                = agent->world().self().pos()
                + Vector2D::polar2vector( 20.0, candidate );
            if ( pitch_rect.contains( face_point )
                 || goalie_rect.contains( face_point ) )
            {
                max_count_angle = candidate;
                max_count = dir_count_sum;
            }
        }

        dir_counts.pop_front();
        add_dir += WorldModel::DIR_STEP;
        tmp_angle += WorldModel::DIR_STEP;
        dir_counts.push_back( agent->world().getDirCount( tmp_angle ) );
    }
    while ( add_dir <= scan_range );


    return max_count_angle;
}

}
