// -*-c++-*-

/*!
  \file segment_2d.h
  \brief 2D segment line Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_GEOM_SEGMENT2D_H
#define RCSC_GEOM_SEGMENT2D_H

#include <cmath>

#include <rcsc/geom/vector_2d.h>
#include <rcsc/geom/line_2d.h>

namespace rcsc {

/*!
  \class Segment2D
  \brief 2d segment line class
*/
class Segment2D {
private:

    Vector2D M_a; //!< first point
    Vector2D M_b; //!< second point

    //! not used
    Segment2D();

public:
    /*!
      \brief construct from 2 points
      \param a 1st point of segment edge
      \param b 2nd point of segment edge
     */
    Segment2D( const Vector2D & a,
               const Vector2D & b )
        : M_a( a )
        , M_b( b )
      { }

    /*!
      \brief construct directly using raw coordinate values
      \param ax 1st point x value of segment edge
      \param ay 1st point x value of segment edge
      \param bx 1st point y value of segment edge
      \param by 1st point y value of segment edge
     */
    Segment2D( const double & ax,
               const double & ay,
               const double & bx,
               const double & by )
        : M_a( ax, ay )
        , M_b( bx, by )
      { }

    /*!
      \brief construct from 2 points
      \param a first point
      \param b second point
      \return const reference to itself
    */
    const
    Segment2D & assign( const Vector2D & a,
                        const Vector2D & b )
      {
          M_a = a;
          M_b = b;
          return *this;
      }

    /*!
      \brief construct directly using raw coordinate values
      \param ax 1st point x value of segment edge
      \param ay 1st point x value of segment edge
      \param bx 1st point y value of segment edge
      \param by 1st point y value of segment edge
    */
    const
    Segment2D & assign( const double & ax,
                        const double & ay,
                        const double & bx,
                        const double & by )
      {
          M_a.assign( ax, ay );
          M_b.assign( bx, by );
          return *this;
      }

    /*!
      \brief swap segment edge point
      \return const reference to itself
    */
    const
    Segment2D & swap()
      {
          // std::swap( M_a, M_b );
          rcsc::Vector2D tmp = M_a;
          M_a = M_b;
          M_b = tmp;
          return *this;
      }

    /*!
      \brief get 1st point of segment edge
      \return vector object
    */
    const
    Vector2D & a() const
      {
          return M_a;
      }

    /*!
      \brief get 2nd point of segment edge
      \return vector object
    */
    const
    Vector2D & b() const
      {
          return M_b;
      }

    /*!
      \brief get line generated from segment
      \return new line object
    */
    Line2D line() const
      {
          return Line2D( a(), b() );
      }

    /*!
      \brief get the length of this segment
      \return distance value
     */
    double length() const
      {
          return a().dist( b() );
      }

    /*!
      \brief make perpendicular bisector line from segment points
      \return line object
     */
    Line2D perpendicularBisector() const
      {
          return Line2D::perpendicular_bisector( a(), b() );
      }

    /*!
      \brief check if the point is within the rectangle defined by this
      segment as a diagonal line.
      \return true if rectangle contains p
     */
    bool contains( const Vector2D & p ) const
      {
          return ( ( p.x - a().x ) * ( p.x - b().x ) <= 0.0
                   && ( p.y - a().y ) * ( p.y - b().y ) <= 0.0 );
      }

    /*!
      \brief check & get the intersection point with other line segment
      \param other considered line segment
      \param sol pointer to the solution variable.
      \retval true exist intersection with 'segment'
      \retval false no intersection(has same slope or out of segment range)
    */
    bool intersection( const Segment2D & other,
                        Vector2D * sol ) const;

    /*!
      \brief check & get the intersection point with other line
      \param other considered line
      \param sol pointer to the solution variable.
      \retval true exist intersection with 'segment'
      \retval false no intersection(has same slope or out of segment range)
    */
    bool intersection( const Line2D & other,
                        Vector2D * sol ) const;

};

}

#endif
