// -*-c++-*-

/*!
  \file udp_socket.cpp
  \brief UDP connection socket class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include <cstdio> // std::perror()
#include <cstring> // memset(), memcpy()
#include <cerrno> // errno

#include <netdb.h> // gethostbyname(), getaddrinfo(), freeaddrinfo()
#include <unistd.h> // close()
#include <fcntl.h> // fcntl()
#include <sys/types.h> // socket(), getaddrinfo(), freeaddrinfo()
#include <sys/socket.h> // socket(), getaddrinfo(), freeaddrinfo()
#include <netinet/in.h> // struct sockaddr_in, struct in_addr, htons

#include <iostream>

#include "udp_socket.h"

namespace rcsc {

/*!
  \struct AddrImpl
  \brief Pimpl ideom. addres implementation class
*/
struct AddrImpl {
    typedef struct sockaddr_in AddrType;

    AddrType addr_;
};


/*-------------------------------------------------------------------*/
/*!

*/
UDPSocket::UDPSocket( const int port )
    : M_fd( -1 )
    , M_dest( new AddrImpl )
{
    std::memset( reinterpret_cast< char * >( &(M_dest->addr_) ),
                 0,
                 sizeof( AddrImpl::AddrType ) );
    if ( open()
         && bind( port )
         && setNonBlocking() != -1 )
    {
        return;
    }
    this->close();
}

/*-------------------------------------------------------------------*/
/*!

*/
UDPSocket::UDPSocket( const char * hostname,
                      const int port )
    : M_fd( -1 )
    , M_dest( new AddrImpl )
{
    std::memset( reinterpret_cast< char * >( &(M_dest->addr_) ),
                 0,
                 sizeof( AddrImpl::AddrType ) );
    if ( open()
         && bind()
         && setAddr( hostname, port )
         && setNonBlocking() != -1 )
    {
        return;
    }
    this->close();
}

/*-------------------------------------------------------------------*/
/*!

*/
UDPSocket::~UDPSocket()
{
    this->close();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
UDPSocket::open()
{
    // create socket
    M_fd = ::socket( AF_INET, SOCK_DGRAM, 0 );

    if ( fd() == -1 )
    {
        std::cerr << "UDPSocket::connect() failed to open a socket."
                  << std::endl;
        return false;
    }

    ::fcntl( fd(), F_SETFD, FD_CLOEXEC ); // close on exec
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
UDPSocket::bind( const int port )
{
    if ( fd() == -1 )
    {
        return false;
    }

    AddrImpl::AddrType my_addr;
    std::memset( reinterpret_cast< char * >( &my_addr ),
                 0,
                 sizeof( AddrImpl::AddrType ) );
    my_addr.sin_family = AF_INET; // internet connection
    my_addr.sin_addr.s_addr = htonl( INADDR_ANY );
    my_addr.sin_port = htons( port );

    if ( ::bind( fd(),
                 reinterpret_cast< struct sockaddr * >( &my_addr ),
                 sizeof( AddrImpl::AddrType ) ) < 0 )
    {
        std::cerr << "UDPSocket::connect(): failed to bind" << std::endl;
        this->close();
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
UDPSocket::setAddr( const char * hostname,
                    const int port )
{
#ifdef HAVE_GETADDRINFO
    struct addrinfo hints;
    std::memset( &hints, 0, sizeof( hints ) );
    hints.ai_family = AF_INET;
    hints.ai_socktype = SOCK_DGRAM;
    hints.ai_protocol = 0;

    struct addrinfo * res;
    if ( ::getaddrinfo( hostname, NULL, &hints, &res ) == 0 )
    {
        //std::cerr << "UDPSocket::connect(): created by getaddrinfo()"
        //          << std::endl;
        M_dest->addr_.sin_addr.s_addr
            = (reinterpret_cast< struct sockaddr_in * >(res->ai_addr))->sin_addr.s_addr;
        M_dest->addr_.sin_family = AF_INET;
        M_dest->addr_.sin_port = htons( port );

        freeaddrinfo( res );
        return true;
    }
#endif

#ifdef HAVE_GETHOSTBYNAME
    struct hostent * host_entry = ::gethostbyname( hostname );
    if ( host_entry )
    {
        //std::cerr << "UDPSocket::connect(): created by gethostbyname()"
        //          << std::endl;
        std::memcpy( &(M_dest->addr_.sin_addr.s_addr),
                     host_entry->h_addr_list[0],
                     host_entry->h_length );
        M_dest->addr_.sin_family = AF_INET;
        M_dest->addr_.sin_port = htons( port );

        return true;
    }
#endif

    std::cerr << "UDPSocket::connect() failed to find host [" << hostname
              << "]" << std::endl;
    this->close();
    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
UDPSocket::setNonBlocking()
{
    int flags = ::fcntl( fd(), F_GETFL, 0 );
    if ( flags == -1 )
    {
        return flags;
    }

    return ::fcntl( fd(), F_SETFL, O_NONBLOCK | flags );
}

/*-------------------------------------------------------------------*/
/*!

*/
int
UDPSocket::close()
{
    if ( fd() != -1 )
    {
        int ret = ::close( fd() );
        M_fd = -1;
        std::memset( reinterpret_cast< char * >( &(M_dest->addr_) ),
                     0,
                     sizeof( AddrImpl::AddrType ) );
        return ret;
    }

    return 0;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
UDPSocket::send( const char * msg,
                 const int len )
{
    if ( ::sendto( fd(), msg, len, 0,
                   reinterpret_cast< struct sockaddr * >( &(M_dest->addr_) ),
                   sizeof( AddrImpl::AddrType ) ) != len )
    {
        return -1;
    }

    return len;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
UDPSocket::receive( char * buf,
                    std::size_t len )
{
    AddrImpl::AddrType from_addr;
    socklen_t from_size = sizeof( AddrImpl::AddrType );
    int n = ::recvfrom( fd(), buf, len, 0,
                        reinterpret_cast< struct sockaddr * >( &from_addr ),
                        &from_size );
    //std::cerr << "receive: " << n << " bytes" << std::endl;
    if ( n == -1 )
    {
        if ( errno == EWOULDBLOCK )
        {
            return 0;
        }

        std::perror( "receive" );
        return -1;
    }

    if ( from_addr.sin_port != 0 )
    {
        //std::cerr << "dest port = " << from.sin_port << std::endl;
        M_dest->addr_.sin_port = from_addr.sin_port;
    }

    return n;
}

} // end namespace
