// -*-c++-*-

/*!
  \file cmd_line_parser.cpp
  \brief command line argument parser Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cstdio>
#include <iterator>

#include "param_map.h"
#include "cmd_line_parser.h"

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
CmdLineParser::CmdLineParser( const int argc,
                              const char * const * argv )
{
    std::copy( argv + 1, argv + argc, std::back_inserter( M_args ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
CmdLineParser::parse( ParamMap & param_map )
{
    for ( std::vector< std::string >::size_type i = 0; i < M_args.size(); ++i )
    {
        // get parameter name stiring & parameter string
        std::string name_str;
        bool is_long_name = false;

        if ( ! M_args[i].compare( 0, 2, "--" ) )
        {
            is_long_name = true;
            name_str = M_args[i].substr( 2 );
        }
        else if ( M_args[i].length() > 1
                  && M_args[i][0] == '-' )
        {
            name_str = M_args[i].substr( 1 );
        }
        else
        {
            M_positional_args.push_back( M_args[i] );
            continue;
        }

        if ( name_str.empty() )
        {
            std::cerr << "***ERROR*** CmdLineParser. Empty parameter name"
                      << std::endl;
            //return false;
            continue;
        }

        int next_i = i;
        if ( is_long_name )
        {
            next_i = parseLongOption( param_map, name_str, i );
        }
        else
        {
            next_i = parseShortOption( param_map, name_str, i );
        }

        if ( next_i != 0 )
        {
            i = next_i;
        }
        // else return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::size_t
CmdLineParser::parseLongOption( ParamMap & param_map,
                                const std::string & long_name,
                                const std::size_t cur_idx )
{
    ParamPtr param_ptr = param_map.getParamPtr( long_name );

    if ( ! param_ptr )
    {
        //std::cerr << "***ERROR*** CmdLineParser. Unknown option name ["
        //          << long_name << "]" << std::endl;
        if ( M_accepted_options.find( long_name ) == M_accepted_options.end() )
        {
            M_invalid_options.insert( long_name );
        }
        return 0;
    }

    // if parameter is switch type, always set true.
    if ( param_ptr->isSwitch() )
    {
        param_ptr->analyze( "" );
        return cur_idx;
    }

    // No value argument
    if ( cur_idx >= M_args.size() - 1 )
    {
        std::cerr << "***ERROR*** CmdLineParser. Value must be specified for [--"
                  << long_name << "]" << std::endl;
        //M_invalid_options.insert( long_name );
        return 0;
    }

    // analyze value string
    if ( ! param_ptr->analyze( M_args[cur_idx + 1] ) )
    {
        std::cerr << "***ERROR*** CmdLineParser. Invalid option. name=["
                  << long_name << "] value=[" << M_args[cur_idx + 1] << "]"
                  << std::endl;
        //M_invalid_options.insert( long_name );
        return 0;
    }

    M_accepted_options.insert( long_name );
    M_invalid_options.erase( long_name );

    return cur_idx + 1;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::size_t
CmdLineParser::parseShortOption( ParamMap & param_map,
                                 const std::string & short_name,
                                 const std::size_t cur_idx )
{
    std::size_t new_idx = cur_idx;

    for ( std::string::const_iterator c = short_name.begin();
          c != short_name.end();
          ++c )
    {
        ParamPtr param_ptr = param_map.getParamPtr( *c );

        if ( ! param_ptr )
        {
            //std::cerr << "***ERROR*** CmdLineParser. Unknown short option name ["
            //          << *c << "]" << std::endl;
            if ( M_accepted_options.find( std::string( 1, *c ) ) == M_accepted_options.end() )
            {
                M_invalid_options.insert( std::string( 1, *c ) );
            }
            return 0;
        }

        // if parameter is switch type, always set true.
        if ( param_ptr->isSwitch() )
        {;
            param_ptr->analyze( "" );
            M_accepted_options.insert( std::string( 1, *c ) );
            M_invalid_options.erase( std::string( 1, *c ) );
            continue;
        }

        // No value argument
        if ( new_idx >= M_args.size() - 1 )
        {
            std::cerr << "***ERROR*** CmdLineParser. Value must be specified for [-"
                      << *c << "]" << std::endl;
            //M_invalid_options.insert( std::string( 1, *c ) );
            return 0;
        }

        // analyze value string
        if ( ! param_ptr->analyze( M_args[new_idx + 1] ) )
        {
            std::cerr << "***ERROR*** CmdLineParser. Invalid option. name=["
                      << *c << "] value=[" << M_args[new_idx + 1] << "]"
                      << std::endl;
            //M_invalid_options.insert( std::string( 1, *c ) );
            return 0;
        }

        M_accepted_options.insert( std::string( 1, *c ) );
        M_invalid_options.erase( std::string( 1, *c ) );

        new_idx += 1;
    }

    return new_idx;
}

}
