// -*-c++-*-

/*!
  \file body_intercept.cpp
  \brief ball chasing action including smart planning.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "body_intercept.h"

#include "basic_actions.h"
#include "body_go_to_point.h"

#include <rcsc/player/logger.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/param/server_param.h>
#include <rcsc/soccer_math.h>
#include <rcsc/math_util.h>

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_Intercept::execute( PlayerAgent * agent )
{
    dlog.addText( Logger::TEAM,
                  "%s: Body_Intercept"
                  ,__FILE__ );

    const WorldModel & wm = agent->world();

    /////////////////////////////////////////////
    if ( doKickableOpponentCheck( agent ) )
    {
        return true;;
    }

    const InterceptTable * table = wm.interceptTable();
    Vector2D reach_point;

    /////////////////////////////////////////////
    if ( table->selfReachCycle() > 100 )
    {
        reach_point = wm.ball().inertiaFinalPoint();
        agent->debugClient().setTarget( reach_point );

        dlog.addText( Logger::INTERCEPT,
                      "%s: no solution... Just go to ball end point (%.2f %.2f)"
                      ,__FILE__,
                      reach_point.x, reach_point.y );
        agent->debugClient().addMessage( "InterceptNoSolution" );
        Body_GoToPoint( reach_point,
                        2.0,
                        ServerParam::i().maxPower()
                        ).execute( agent );
        return true;
    }

    /////////////////////////////////////////////
    InterceptInfo best_intercept = getBestPoint( wm, table );

    dlog.addText( Logger::INTERCEPT,
                  "%s: solution size= %d. selected best cycle is %d"
                  " (turn:%d + dash:%d)"
                  ,__FILE__,
                  table->selfCache().size(),
                  best_intercept.reachCycle(),
                  best_intercept.turnCycle(), best_intercept.dashCycle() );

    reach_point = wm.ball().inertiaPoint( best_intercept.reachCycle() );
    agent->debugClient().setTarget( reach_point );

    if ( best_intercept.dashCycle() == 0 )
    {
        dlog.addText( Logger::INTERCEPT,
                      "%s: can get the ball only by inertia move. Turn!"
                      ,__FILE__ );
        Vector2D face_point( 50.5, wm.self().pos().y * 0.75 );
        Body_TurnToPoint( face_point ).execute( agent );
        agent->debugClient().addMessage( "InterceptTurnOnly" );
        return true;
    }

    /////////////////////////////////////////////
    if ( best_intercept.turnCycle() > 0 )
    {
        Vector2D my_inertia
            = wm.self().playerType().inertiaPoint( wm.self().pos(),
                                                   wm.self().vel(),
                                                   best_intercept.reachCycle() );
        AngleDeg target_angle = ( reach_point - my_inertia ).th();
        if ( best_intercept.dashPower() < 0.0 )
        {
            // back dash
            target_angle -= 180.0;
        }

        dlog.addText( Logger::INTERCEPT,
                      "%s:%d: turn.first.%s target_body_angle = %.1f"
                      ,__FILE__, __LINE__,
                      ( best_intercept.dashPower() < 0.0 ? "BackMode" : "" ),
                      target_angle.degree() );
        agent->debugClient().addMessage( "InterceptTurn%d(%d/%d)",
                                         best_intercept.reachCycle(),
                                         best_intercept.turnCycle(),
                                         best_intercept.dashCycle() );

        return agent->doTurn( target_angle - wm.self().body() );
    }

    /////////////////////////////////////////////
    dlog.addText( Logger::INTERCEPT,
                  "%s:%d: try dash. power=%.1f  pos=(%.2f, %.2f)"
                  ,__FILE__, __LINE__,
                  best_intercept.dashPower(),
                  reach_point.x, reach_point.y );
    if ( doWaitTurn( agent, best_intercept ) )
    {
        return true;
    }

    if ( M_save_recovery
         && ( wm.self().stamina()
              - ( best_intercept.dashPower() * ( best_intercept.dashPower() > 0.0
                                                 ? 1.0
                                                 : -2.0 )
                  )
              < ServerParam::i().recoverDecThrValue() + 1.0 )
         )
    {
        dlog.addText( Logger::INTERCEPT,
                      "%s:%d: insufficient stamina"
                      ,__FILE__, __LINE__ );
        agent->debugClient().addMessage( "InterceptRecover" );
        agent->doTurn( 0.0 );
        return false;
    }

    //return agent->doDash( best_intercept.dashPower() );
    return doInertiaDash( agent,
                          reach_point,
                          best_intercept.dashPower(),
                          best_intercept.reachCycle() );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_Intercept::doKickableOpponentCheck( PlayerAgent * agent )
{
    const WorldModel & wm = agent->world();
    if ( wm.ball().distFromSelf() < 2.0
         && wm.existKickableOpponent() )
    {
        dlog.addText( Logger::INTERCEPT,
                      "%s:%d: attack to opponent"
                      ,__FILE__, __LINE__ );
        Vector2D attack_pos
            = wm.opponentsFromBall().front()->pos();
        //attack_pos.x -= 1.0;
        Body_GoToPoint( attack_pos,
                        0.2,
                        ServerParam::i().maxPower(),
                        1, // cycle
                        false, // no back
                        true, // save recovery
                        15.0  // dir thr
                        ).execute( agent );
        return true;
    }
    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
InterceptInfo
Body_Intercept::getBestPoint( const WorldModel & wm,
                              const InterceptTable * table ) const
{

    const std::vector< InterceptInfo > & cache = table->selfCache();

    const int opp_cycle = table->opponentReachCycle();

    std::size_t best_idx = 1000;
    double max_x = -1000.0;
    double min_dist2 = 40000.0;
    double turn0_min_dist2 = 40000.0;

    const std::size_t max_idx = cache.size();
    for ( std::size_t i = 0; i < max_idx; ++i )
    {
        if ( M_save_recovery
             && cache[i].mode() != InterceptInfo::NORMAL )
        {
            continue;
        }

        int cycle = cache[i].reachCycle();
        const Vector2D ball_pos = wm.ball().inertiaPoint( cycle );

        if ( ball_pos.absX() > ServerParam::i().pitchHalfLength() - 1.0
             || ball_pos.absY() > ServerParam::i().pitchHalfWidth() - 1.0 )
        {
            continue;
        }

        const Vector2D ball_vel = wm.ball().vel() * std::pow( ServerParam::i().ballDecay(),
                                                              cycle );

        bool attacker = false;
        if ( ball_vel.x > 0.5
             && ball_pos.x > wm.offsideLineX() - 5.0 )
        {
            attacker = true;
        }

        // can get the ball without turn
        if ( cache[i].turnCycle() == 0
             && cycle <= opp_cycle )
        {
            if ( best_idx > 100 && opp_cycle > 3 )
            {

                Vector2D ball_pos = wm.ball().inertiaPoint( cycle );
                double d2 = wm.self().pos().dist2( ball_pos );
                if ( d2 < turn0_min_dist2 )
                {
                    best_idx = i;
                    turn0_min_dist2 = d2;
                }
                continue;
            }
        }

        // opponent check

        int opp_buf = 3;

        if ( attacker )
        {
            opp_buf = -1;
        }

        if ( cycle >= opp_cycle - opp_buf )
        {
            continue;
        }

        if ( attacker )
        {
            if ( best_idx < 100 )
            {
                if ( ball_vel.r() < 0.7
                     || ball_pos.x > 47.0
                     || std::fabs( ball_pos.y - wm.self().pos().y ) > 10.0
                     )
                {
                    continue;
                }
            }

            if ( ball_pos.x > max_x )
            {
                max_x = ball_pos.x;
                best_idx = i;
            }

            continue;
        }

        // can get the ball only by inertia move
        if ( cache[i].dashCycle() == 0 )
        {
            best_idx = i;
            break;
        }

        // no turn is recommended
        if ( cache[i].turnCycle() == 0 )
        {
            best_idx = i;
            break;
        }

        if ( turn0_min_dist2 < 30000.0 )
        {
            // turn 0 has priority
            continue;
        }

        // select the nearest pointa
        double d2 = wm.self().pos().dist2( ball_pos );
        if ( d2 < min_dist2 )
        {
            min_dist2 = d2;
            best_idx = i;
        }
    }

    // found
    if ( best_idx < cache.size() )
    {
        return cache[best_idx];
    }

    // select the first one
    for ( std::size_t i = 0; i < max_idx; ++i )
    {
        if ( M_save_recovery
             && cache[i].mode() != InterceptInfo::NORMAL )
        {
            continue;
        }

        return cache[i];
    }

    if ( cache.empty() )
    {
        return InterceptInfo( InterceptInfo::NORMAL,
                              100, 100, 100 );
    }

    return cache[0];
}


/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_Intercept::doWaitTurn( PlayerAgent * agent,
                            const InterceptInfo & info )
{
    const int reach_cycle = info.reachCycle();
    const WorldModel & wm = agent->world();

    Vector2D ball_pos = wm.ball().inertiaPoint( reach_cycle );

    const int max_wait_cycle = ( ball_pos.x > 10.0
                                 ? 8
                                 : 3 );
    if ( reach_cycle == 1
         || max_wait_cycle <= reach_cycle )
    {
        return false;
    }

    {
        const PlayerObject * opp = wm.getOpponentNearestToSelf( 5 );
        if ( opp && opp->distFromSelf() < 3.0 )
        {
            dlog.addText( Logger::INTERCEPT,
                          "%s: doWaitTurn. exist near opponent  cancel"
                          ,__FILE__ );
            return false;
        }
    }

    Vector2D my_inertia = wm.self().playerType().inertiaPoint( wm.self().pos(),
                                                               wm.self().vel(),
                                                               reach_cycle );
    const double inertia_dist = my_inertia.dist( ball_pos );
    /*
    if ( inertia_dist < ( wm.self().playerType().playerSize()
                          + ServerParam::i().ballSize()
                          + 0.15 ) )
    {
        dlog.addText( Logger::INTERCEPT,
                      "%s: doWaitTurn. ball distance %.3f is too small. avoid collision "
                      ,__FILE__,
                      inertia_dist );
        return false;
    }
    */

    if ( inertia_dist < wm.self().kickableArea() - 0.3 + (0.1*reach_cycle) )
    {
        Vector2D face_point( 50.5, wm.self().pos().y * 0.9 );

        Body_TurnToPoint( face_point ).execute( agent );
        dlog.addText( Logger::INTERCEPT,
                      "%s: doWaitTurn. done"
                      ,__FILE__ );
        agent->debugClient().addMessage( "WaitTurn" );
        return true;
    }

    Vector2D target_rel = ball_pos - my_inertia;
    AngleDeg accel_angle = wm.self().body();

    if ( info.dashPower() < 0.0 )
    {
        accel_angle += 180.0;
    }

    target_rel.rotate( - accel_angle );
    if ( reach_cycle >= 3 )
    {
        if ( target_rel.x > 0.0 )
        {
            target_rel.x -= 0.3;
            if ( target_rel.x < 0.3 ) target_rel.x = 0.3;
        }
        else
        {
            target_rel.x += 0.3;
            if ( target_rel.x > -0.3 ) target_rel.x = -0.3;
        }

    }

    double first_speed
        = calc_first_term_geom_series
        ( target_rel.x,
          wm.self().playerType().playerDecay(),
          reach_cycle );
    first_speed
        = min_max( - wm.self().playerType().playerSpeedMax(),
                   first_speed,
                   wm.self().playerType().playerSpeedMax() );
    Vector2D rel_vel = wm.self().vel().rotatedVector( - accel_angle );
    double required_accel = first_speed - rel_vel.x;
    double used_power = required_accel / wm.self().dashRate();
    if ( info.dashPower() < 0.0 ) used_power = -used_power;

    used_power = ServerParam::i().normalizePower( used_power );

    used_power = wm.self().getSafetyDashPower( used_power );
    agent->debugClient().addMessage( "WaitTurnInertiaDash%.0f",
                                     used_power );

    dlog.addText( Logger::INTERCEPT,
                  "%s: doWaitTurn. aball_dist after wait %.3f  do dash "
                  " accel_abs= %.3f power = %.1f"
                  ,__FILE__,
                  inertia_dist,
                  required_accel, used_power );
    //std::cerr << wm.time() << wm.self().unum()
    //          << " InterceptInertiaDist= "
    //          << my_inertia.dist( ball_pos )
    //          << " InertiaDash"
    //          << " origPow="  << info.dashPower()
    //          << " newPow=" << used_power
    //          << std::endl;
    agent->doDash( used_power );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_Intercept::doInertiaDash( PlayerAgent * agent,
                               const Vector2D & target_point,
                               const double & dash_power,
                               const int cycle )
{
    const WorldModel & wm = agent->world();
    Vector2D my_inertia = wm.self().playerType().inertiaPoint( wm.self().pos(),
                                                               wm.self().vel(),
                                                               cycle );
    Vector2D target_rel = target_point - my_inertia;
    AngleDeg accel_angle = wm.self().body();
    if ( dash_power < 0.0 )
    {
        accel_angle += 180.0;
    }

    target_rel.rotate( - accel_angle );

    double first_speed
        = calc_first_term_geom_series
        ( target_rel.x,
          wm.self().playerType().playerDecay(),
          cycle );

    first_speed
        = min_max( - wm.self().playerType().playerSpeedMax(),
                   first_speed,
                   wm.self().playerType().playerSpeedMax() );
    Vector2D rel_vel = wm.self().vel().rotatedVector( - accel_angle );
    double required_accel = first_speed - rel_vel.x;
    double used_power = required_accel / wm.self().dashRate();
    if ( dash_power < 0.0 ) used_power = -used_power;

    used_power = ServerParam::i().normalizePower( used_power );
    used_power = wm.self().getSafetyDashPower( used_power );

    agent->debugClient().addMessage( "InterceptInertiaDash%.0f",
                                     used_power );

    dlog.addText( Logger::INTERCEPT,
                  "%s: doInertiaDash. x_diff= %.2f first_speed = %.2f"
                  " accel= %.2f power = %.1f"
                  ,__FILE__,
                  target_rel.x, first_speed, required_accel, used_power );

    if ( cycle >= 4
         && ( target_rel.absX() < 0.5
              || std::fabs( used_power ) < 5.0 )
         )
    {
        agent->debugClient().addMessage( "LookBall" );

        Vector2D face_point( 50.5, wm.self().pos().y * 0.75 );
        AngleDeg face_angle = ( face_point - my_inertia ).th();

        Vector2D ball_next = wm.ball().pos() + wm.ball().vel();
        AngleDeg ball_angle = ( ball_next - my_inertia ).th();

        if ( ( ball_angle - face_angle ).abs()
             > ( ServerParam::i().maxNeckAngle()
                 + ServerParam::i().visibleAngle() * 0.5
                 - 10.0 )
             )
        {
            face_point.x = my_inertia.x;
            if ( ball_next.y > my_inertia.y + 1.0 ) face_point.y = 50.0;
            else if ( ball_next.y < my_inertia.y - 1.0 ) face_point.y = -50.0;
            else  face_point = ball_next;
            dlog.addText( Logger::INTERCEPT,
                          "%s: doInertiaDash. check ball  with turn. face to (%.1f %.1f)"
                          ,__FILE__,
                          face_point.x, face_point.y );
        }
        else
        {
            dlog.addText( Logger::INTERCEPT,
                          "%s: doInertiaDash. can check ball without turn face to (%.1f %.1f)"
                          ,__FILE__,
                          face_point.x, face_point.y );
        }
        Body_TurnToPoint( face_point ).execute( agent );
        return true;
    }

    agent->doDash( used_power );
    return true;
}

}
