// -*-c++-*-

/*!
  \file body_kick_two_step.cpp
  \brief two step kick behavior to accelerate the ball to the desired
  speed.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "body_kick_two_step.h"

#include "body_kick_one_step.h"
#include "body_stop_ball.h"
#include "body_kick_to_relative.h"

#include <rcsc/player/logger.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/param/server_param.h>
#include <rcsc/geom/circle_2d.h>
#include <rcsc/geom/ray_2d.h>
#include <rcsc/soccer_math.h>
#include <rcsc/math_util.h>

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_KickTwoStep::execute( PlayerAgent* agent )
{
    dlog.addText( Logger::KICK,
                  "%s:%d: Body_KickTwoStep"
                  ,__FILE__, __LINE__ );

    if ( ! agent->world().self().isKickable() )
    {
        std::cerr << __FILE__ << ": " << __LINE__
                  << " not ball kickable!"
                  << std::endl;
        dlog.addText( Logger::ACTION,
                      "%s:%d:  not kickable"
                      ,__FILE__, __LINE__ );
        return false;
    }

    Vector2D required_acc;
    double kick_pow;
    AngleDeg kick_dir;

    //---------------------------------------------------

    Vector2D target_rpos = M_target_point - agent->world().self().pos();
    M_first_speed = std::min( M_first_speed, ServerParam::i().ballSpeedMax() );

    Vector2D achieved_vel;

    if ( simulate_one_kick( &achieved_vel,
                            target_rpos,
                            M_first_speed,
                            Vector2D( 0.0, 0.0 ), // my current pos
                            agent->world().self().vel(),
                            agent->world().self().body(),
                            agent->world().ball().rpos(),
                            agent->world().ball().vel(),
                            agent,
                            false ) ) // not enforce
    {
        required_acc = achieved_vel - agent->world().ball().vel();
        kick_pow = required_acc.r() / agent->world().self().kickRate();
        kick_dir = required_acc.th() - agent->world().self().body();

        dlog.addText( Logger::KICK,
                      "%s:%d: only one step. result=(%.3f, %.3f) r=%.3f"
                      " acc=(%.3f, %.3f) power=%.1f dir=%.1f"
                      ,__FILE__, __LINE__,
                      achieved_vel.x, achieved_vel.y, achieved_vel.r(),
                      required_acc.x, required_acc.y,
                      kick_pow, kick_dir.degree() );
        return agent->doKick( kick_pow, kick_dir );
    }


    Vector2D next_vel;
    if ( simulate_two_kick( &achieved_vel,
                            &next_vel,
                            target_rpos,
                            M_first_speed,
                            Vector2D( 0.0, 0.0 ), // my current pos
                            agent->world().self().vel(),
                            agent->world().self().body(),
                            agent->world().ball().rpos(),
                            agent->world().ball().vel(),
                            agent,
                            M_enforce_kick ) ) // enforced
    {
        required_acc = next_vel - agent->world().ball().vel();
        kick_pow = required_acc.r() / agent->world().self().kickRate();
        kick_dir = required_acc.th() - agent->world().self().body();

        dlog.addText( Logger::KICK,
                      "%s:%d: two step. result=(%.3f, %.3f)r=%.3f, next_vel=(%.3f, %.3f) r=%.3f "
                      "  acc=(%.3f, %.3f) power=%.1f dir=%.1f"
                      ,__FILE__, __LINE__,
                      achieved_vel.x, achieved_vel.y, achieved_vel.r(),
                      next_vel.x, next_vel.y, next_vel.r(),
                      required_acc.x, required_acc.y,
                      kick_pow, kick_dir.degree() );
        return agent->doKick( kick_pow, kick_dir );
    }

    if ( M_enforce_kick )
    {
        dlog.addText( Logger::KICK,
                      "%s:%d: enforce one kick"
                      ,__FILE__, __LINE__ );
        return Body_KickOneStep( M_target_point,
                                 M_first_speed
                                 ).execute( agent );
    }

    dlog.addText( Logger::KICK,
                  "%s:%d: why reach here?? stop ballk"
                  ,__FILE__, __LINE__ );

    double keep_dist = agent->world().self().playerType().playerSize()
        + agent->world().self().playerType().kickableMargin() * 0.6;
    AngleDeg keep_angle = agent->world().ball().angleFromSelf()
        - agent->world().self().body();
    return Body_KickToRelative( keep_dist,
                                keep_angle,
                                false
                                ).execute( agent );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_KickTwoStep::is_opp_kickable( const PlayerAgent * agent,
                                   const Vector2D & rel_pos )
{
    // use default value
    static const double KICKABLE2
        = square( ServerParam::i().defaultKickableArea() + 0.17 );
    const PlayerPtrCont & opps = agent->world().opponentsFromSelf();

    PlayerPtrCont::const_iterator it = opps.begin();
    const PlayerPtrCont::const_iterator it_end = opps.end();
    for ( PlayerPtrCont::const_iterator it = opps.begin();
          it != it_end;
          ++it )
    {
        if ( (*it)->distFromSelf() > 6.0 )
        {
            break;
        }

        if ( (*it)->posCount() > 1 )
        {
            continue;
        }

        if ( rel_pos.dist2( (*it)->rpos() + (*it)->vel() ) < KICKABLE2 )
        {
#ifdef DEBUG
            dlog.addText( Logger::KICK,
                          "%s:%d: detect kickable oppfor rpos(%.2f %.2f)"
                          ,__FILE__, __LINE__,
                          rel_pos.x, rel_pos.y );
#endif
            return true;
        }
        // check opponent goalie
        else if ( (*it)->goalie()
                  && (*it)->pos().x > ServerParam::i().theirPenaltyAreaLine() + 1.0
                  && ( (*it)->pos().absY()
                       < ServerParam::i().penaltyAreaHalfWidth() - 1.0 )
                  && ( rel_pos.dist( (*it)->rpos() + (*it)->vel() )
                       < ServerParam::i().catchAreaLength() )
                  )
        {
            return true;
        }
    }

#ifdef DEBUG
    dlog.addText( Logger::KICK,
                  "%s:%d: No kickable opp. for rpos(%.2f %.2f)"
                  ,__FILE__, __LINE__,
                  rel_pos.x, rel_pos.y );
#endif
    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_KickTwoStep::simulate_one_kick( Vector2D * achieved_vel,
                                     const Vector2D & target_rpos,
                                     const double & first_speed,
                                     const Vector2D & my_rpos,
                                     const Vector2D & my_vel,
                                     const AngleDeg & my_body,
                                     const Vector2D & ball_rpos,
                                     const Vector2D & ball_vel,
                                     const PlayerAgent * agent,
                                     const bool enforce )
{
    //dlog.addText( Logger::KICK,
    //"Kick Search: one step @~@~@~@~@~@~@~@~@~@~@~@~@~@" );

    Vector2D ball_rel_at_here = ball_rpos - my_rpos;
    const double krate
        = kick_rate( ball_rel_at_here.r(),
                     ( ball_rel_at_here.th() - my_body ).degree(),
                     ServerParam::i().kickPowerRate(),
                     ServerParam::i().ballSize(),
                     agent->world().self().playerType().playerSize(),
                     agent->world().self().playerType().kickableMargin() );
    const double enable_acc
        = std::min( ServerParam::i().maxPower() * krate,
                    ServerParam::i().ballAccelMax() );

    Vector2D required_vel
        = Vector2D::polar2vector( first_speed, ( target_rpos - ball_rpos ).th() );

    // can NOT reach the target vel
    if ( ( required_vel - ball_vel ).r() > enable_acc ) // * 0.95 )
    {
        if ( ! enforce )
        {
            //required_vel
            //    = Body_KickOneStep::get_max_possible_vel( ( target_rpos - ball_rpos ).th(),
            //                                              krate,
            //                                              ball_vel );
            //dlog.addText( Logger::KICK,
            //" failed. enable max speed=%f",
            //required_vel.r() );
            return false;
        }
        required_vel
            = Body_KickOneStep::get_max_possible_vel( (target_rpos - ball_rpos).th(),
                                                      krate,
                                                      ball_vel );
        dlog.addText( Logger::KICK,
                      "%s:%d: simulate_one_kick. never reach. try enforce kick"
                      ,__FILE__, __LINE__ );
    }

    // check collision & opponents

    Vector2D next_ball_rpos = ball_rpos + required_vel;
    if ( next_ball_rpos.dist2( my_rpos + my_vel )
         < square( agent->world().self().playerType().playerSize()
                   + ServerParam::i().ballSize()
                   + 0.1 )
         || is_opp_kickable( agent, next_ball_rpos ) )
    {
        //TRACE(dlog.addText( Logger::KICK,
        //" sim_one_kick collide or close opp" ));
        return false;
    }


    if ( achieved_vel )
    {
        *achieved_vel = required_vel;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
Body_KickTwoStep::simulate_two_kick( Vector2D * achieved_vel,
                                     Vector2D * next_vel,
                                     const Vector2D & target_rpos,
                                     const double & first_speed,
                                     const Vector2D & my_rpos,
                                     const Vector2D & my_vel,
                                     const AngleDeg & my_body,
                                     const Vector2D & ball_rpos,
                                     const Vector2D & ball_vel,
                                     const PlayerAgent * agent,
                                     const bool enforce )
{
    //dlog.addText( Logger::KICK,
    //" Kick Search: two step !_!_!_!_!_!_!_!_!_!_!_!_!" );

    const double noise_robust_rate = 0.9;

    Vector2D ball_rel_at_here = ball_rpos - my_rpos;
    const double krate
        = kick_rate( ball_rel_at_here.r(),
                     ( ball_rel_at_here.th() - my_body ).degree(),
                     ServerParam::i().kickPowerRate(),
                     ServerParam::i().ballSize(),
                     agent->world().self().playerType().playerSize(),
                     agent->world().self().playerType().kickableMargin() );
    const double enable_acc
        = std::min( ServerParam::i().maxPower() * krate,
                    ServerParam::i().ballAccelMax() );

    const double my_kickable
        = agent->world().self().playerType().playerSize()
        + agent->world().self().playerType().kickableMargin()
        + ServerParam::i().ballSize();

    // next ball position & vel
    std::vector< std::pair< Vector2D, Vector2D > > subtargets;

    //////////////////////////////////////////////////////////////////
    // first element is next kickable edge
    {
        const Ray2D desired_ray( ball_rpos, (target_rpos - ball_rpos).th() );
        const Circle2D next_kickable_circle( my_rpos + my_vel,
                                             my_kickable * 0.9 );

        // get intersection next kickable circle & desired ray
        // solutions are next ball pos, relative to current my pos
        Vector2D sol1, sol2;
        int num = next_kickable_circle.intersection( desired_ray, &sol1, &sol2 );
        //Vector2D require_vel( Vector2D::Invalid() );
        Vector2D require_vel( Vector2D::INVALIDATED );
        if ( num == 1 )
        {
            // if ball point is not within next kicable area.
            // it is very dangerous to kick to the kickable edge.
            if ( next_kickable_circle.contains( ball_rpos ) )
            {
                require_vel = sol1 - ball_rpos;
            }
        }
        else if ( num == 2 )
        {
            // current ball point is NOT within next kicable area.
            Vector2D v1 = sol1 - ball_rpos;
            Vector2D v2 = sol2 - ball_rpos;
            // set bigger one
            require_vel = ( v1.r2() > v2.r2() ? v1 : v2 );
        }

        if ( require_vel.valid()
             && ! is_opp_kickable( agent, ball_rpos + require_vel )
             && ( require_vel - ball_vel ).r() < enable_acc )
        {
            double d = require_vel.r();
            if ( d > ServerParam::i().ballSpeedMax() )
            {
                require_vel *= ServerParam::i().ballSpeedMax() / d;
            }
            require_vel *= noise_robust_rate; // for noise robustness
            // add element
            subtargets.push_back( std::make_pair
                                  ( ball_rpos + require_vel,
                                    require_vel * ServerParam::i().ballDecay()) );
        }
    }


    const Vector2D my_next = my_rpos + my_vel;

    //////////////////////////////////////////////////////////////////
    // generate other subtargets
    {
        const double subtarget_dist = my_kickable * 0.8;
        const double default_dir_inc =  30.0;

        const AngleDeg angle_self_to_target = ( target_rpos - my_next ).th();
        const double ball_target_dir_diff
            = ( angle_self_to_target - (ball_rpos - my_next).th() ).abs();

        // sub-targets are forward than ball
        double inc = ball_target_dir_diff / 5.0;
        inc = std::max( inc, default_dir_inc );
        inc = std::min( inc, ball_target_dir_diff );
        for ( double d = -ball_target_dir_diff;
              d <= ball_target_dir_diff + 1.0;
              d += inc )
        {

            Vector2D sub // rel to current my pos
                = my_next
                + Vector2D::polar2vector( subtarget_dist,
                                          angle_self_to_target + d );

            Vector2D require_vel = sub - ball_rpos;
            if ( ! is_opp_kickable( agent, sub ) // check opponents
                 && (require_vel - ball_vel).r() < enable_acc )
            {
                subtargets.push_back
                    ( std::make_pair( sub,
                                      require_vel * ServerParam::i().ballDecay() ) );
            }
        }
    }


    //////////////////////////////////////////////////////////////////
    // subtargets are next ball position

    Vector2D max_achieved_vel(0.0, 0.0);
    Vector2D sol_next_ball_vel(100.0, 100.0);
    bool found = false;
    {
        const Vector2D my_next_vel
            = my_vel * agent->world().self().playerType().playerDecay();

        Vector2D sol_vel;
        const std::vector< std::pair<Vector2D, Vector2D> >::iterator
            it_end = subtargets.end();
        for ( std::vector< std::pair< Vector2D, Vector2D > >::iterator
                  it = subtargets.begin();
              it != it_end;
              ++it )
        {
            //Vector2D tmp_rel = it->first - my_next;
            //dlog.addText( Logger::KICK,
            //              " check subtarget(%f, %f) r=%f th=%f",
            //              tmp_rel.x, tmp_rel.y,
            //              tmp_rel.r(), tmp_rel.th().degree() );

            if ( simulate_one_kick( &sol_vel,
                                    target_rpos,
                                    first_speed,
                                    my_next,
                                    my_next_vel,
                                    my_body,
                                    it->first, // next ball
                                    it->second, // next ball vel
                                    agent,
                                    enforce )
                 )
            {
                //dlog.addText( Logger::KICK,
                //"  --> success. final_vel=(%f, %f) r=%f",
                //sol_vel.x, sol_vel.y, sol_vel.r() );

                Vector2D bvel = it->second / ServerParam::i().ballDecay();
                if ( ( enforce && sol_vel.r2() > max_achieved_vel.r2() )
                     || ( ! enforce
                          && sol_vel.r2() >= max_achieved_vel.r2() - 0.0001
                          && sol_next_ball_vel.r2() > bvel.r2() ) ) // lower speed makes lower noise
                {
                    found = true;
                    max_achieved_vel = sol_vel;
                    sol_next_ball_vel = bvel;
                    //dlog.addText( Logger::KICK,
                    //"      --> updated" );
                }
            }
            //else
            //{
            //    dlog.addText( Logger::KICK,
            //                  "  --> failed " );
            //}
        }


        if ( enforce )
        {
            // test one step kick & compare the result of two kick
            if ( simulate_one_kick( &sol_vel,
                                    target_rpos, // relative to current
                                    first_speed,
                                    my_rpos, // relative to current
                                    my_vel,
                                    my_body,
                                    ball_rpos, // relative to current
                                    ball_vel,
                                    agent,
                                    true ) // enforce
                 )
            {
                if ( sol_vel.r2() > max_achieved_vel.r2() )
                {
                    found = true;
                    //max_achieved_vel = sol_vel;
                    sol_next_ball_vel = sol_vel;
                    //dlog.addText( Logger::KICK,
                    //" ONE kick enable me to get higher speed\n"
                    //"      --> updated. vel=(%f, %f) r=%f ",
                    //sol_vel.x, sol_vel.y, sol_vel.r() );
                }
            }

        }

    }


    if ( ! found )
    {
        //dlog.addText( Logger::KICK,
        //" two step not found" );
        return false;
    }

    if ( achieved_vel )
    {
        *achieved_vel = max_achieved_vel;
    }
    if ( next_vel )
    {
        *next_vel = sol_next_ball_vel;
    }

    //TRACE(dlog.addText( Logger::KICK,
    //" two step found" ));
    return true;
}

}
