// -*-c++-*-

/*!
  \file basic_client.cpp
  \brief abstract soccer client class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#include "basic_client.h"

#include <rcsc/net/udp_socket.h>
#include <rcsc/param/cmd_line_parser.h>

#include <iostream>
#include <cstring>

#include <unistd.h> // select()
#include <sys/select.h> // select()
#include <sys/time.h> // select()
#include <sys/types.h> // select()

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

*/
BasicClient::BasicClient()
    : M_server_alive( false )
    , M_interval_msec( 10 )
    , M_waited_msec( 0 )
    , M_timeout_count( 0 )
    , M_last_recv_time()
{
    std::memset( M_recv_buf, 0, MAX_MESG );
}

/*-------------------------------------------------------------------*/
/*!

*/
BasicClient::~BasicClient()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
 bool
 BasicClient::init( const int argc,
                    const char * const * argv )
{
    CmdLineParser cmd_parser( argc, argv );
    return doInit( cmd_parser );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
BasicClient::run()
{
    doConnect();

    if ( ! M_socket
         || M_socket->fd() == -1 )
    {
        std::cerr << "BasicClinet::run(). No connection!" << std::endl;
        handleExit();
        return;
    }

    // send init command to server
    sendInitCommand();

    // init timer
    M_waited_msec = 0;
    M_last_recv_time.setCurrent();

    // set interval timeout
    struct timeval interval;

    fd_set read_fds;
    fd_set read_fds_back;

    FD_ZERO( &read_fds );
    FD_SET( M_socket->fd(), &read_fds );
    read_fds_back = read_fds;

    while ( isServerAlive() )
    {
        read_fds = read_fds_back;
        interval.tv_sec = M_interval_msec / 1000;
        interval.tv_usec = ( M_interval_msec % 1000 ) * 1000;

        int ret = ::select( M_socket->fd() + 1, &read_fds,
                            static_cast< fd_set * >( 0 ),
                            static_cast< fd_set * >( 0 ),
                            &interval );
        if ( ret < 0 )
        {
            perror( "select" );
            break;
        }
        else if ( ret == 0 )
        {
            // no meesage. timeout.
            M_waited_msec += M_interval_msec;
            ++M_timeout_count;
            handleTimeout();
        }
        else
        {
            // received message, reset wait time
            M_waited_msec = 0;
            M_timeout_count = 0;
            M_last_recv_time.setCurrent();
            handleMessage();
        }
    }

    handleExit();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
BasicClient::connectTo( const char * hostname,
                        const int port,
                        const long & interval_msec )
{
    M_socket
        = boost::shared_ptr< UDPSocket >
        ( new UDPSocket( hostname, port ) );

    if ( ! M_socket
         || M_socket->fd() == -1 )
    {
        setServerAlive( false );
        return false;
    }

    setServerAlive( true );

    if ( interval_msec < 10 )
    {
        std::cerr << "***WARNING*** interval msec should be more than or equal 10."
                  << std::endl;
        setIntervalMSec( 10 );
    }
    else
    {
        setIntervalMSec( interval_msec );
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
BasicClient::sendMessage( const char * msg )
{
    if ( ! M_socket )
    {
        return 0;
    }

#if 0
    if ( M_compression_level > 0 && M_compress )
    {
        M_compress->compress( msg );
        int len = M_compress->getBuf().length();
        if ( len > 0 )
        {
            return M_socket->send( M_compress->getBuf().data(), len );
        }
        return 0;
    }
#endif

    //std::cerr << "send [" << msg << "0]" << endl;
    // if the length of message is the result of only strlen,
    // server will reply "(warning message_not_null_terminated)"
    return M_socket->send( msg, std::strlen( msg ) + 1 );
}

/*-------------------------------------------------------------------*/
/*!

*/
int
BasicClient::recvMessage()
{
    if ( ! M_socket )
    {
        return 0;
    }

    int n = M_socket->receive( M_recv_buf, MAX_MESG );
    if ( n > 0 )
    {
#if 0
        if ( M_compresstion_level > 0 && M_decompress )
        {
            M_decompress->decompress( M_recv_buf, n );
            std::strncpy( M_recv_buf, M_decompress->getBuf().c_str(), MAX_MESG );
        }
        else
#endif
        {
            M_recv_buf[n] = '\0';
        }
    }

    return n;
}

}
