// -*-c++-*-

/*!
  \file basic_client.h
  \brief abstract soccer agent class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_BASIC_CLIENT_H
#define RCSC_BASIC_CLIENT_H

#include <rcsc/timer.h>

#include <boost/shared_ptr.hpp>

namespace rcsc {

class UDPSocket;
class CmdLineParser;

/*!
  \class BasicClient
  \brief abstract soccer agent class

  This class supplies:
  - interface to accept command line option
  - UDP connection interfece to the rcssserver.
  - interface to handle server message
  - interface to handle timeout event
  - interface to handle exit event
 */
class BasicClient {
public:
    enum {
        MAX_MESG = 8192, //!< max length of receive buffer.
    };

private:
    //! udp connection
    boost::shared_ptr< UDPSocket > M_socket;

    //! flag to check server status
    bool M_server_alive;

    //! timeout interval for select()
    long M_interval_msec;

    //! max time to wait the server message
    long M_waited_msec;

    //! the number of timeout without server response
    int M_timeout_count;

    //! time last server message is received
    TimeStamp M_last_recv_time;

    //! buffer to receive server message
    char M_recv_buf[MAX_MESG];

#if 0
    //! gzip compression level
    int M_compression_level;
    //! gzip compressor
    boost::shared_ptr< Compressor > M_compress;
    //! gzip decompressor
    boost::shared_ptr< Decompressor > M_decompress;
#endif

    // nocopyable
    BasicClient( const BasicClient & );
    BasicClient & operator=( const BasicClient & );

public:
    /*!
      \brief init member variables
     */
    BasicClient();

    /*!
      \brief virtual destructor.
     */
    virtual
    ~BasicClient();

    /*!
      \brief initialize with command line options.
      \param argc number of options
      \param argv array of option string
      \return initialization result of the derived class.

      Connection must be created after this method.
      You should specify the server host name, port number
      and wait interval msec in doInit() virtual method in
      the derived class.
      (init) commad is sent in run() method. So,do not call it yourself!
    */
    bool init( const int argc,
               const char * const * argv );

    /*!
      \brief program mainloop

      Thie method keep infinite loop while client can estimate server is alive.
      To handle server message, select() is used.
      Timeout interval of select() is specified by M_interval_msec member variable.
      When server message is received, handleMessage() is called.
      When timeout occurs, handleTimeout() is called.
      When server is not alive, loop is end and handleExit() is called.
     */
    void run();

protected:

    /*!
      \brief connect to the specified server with timeout value for select()
      \param hostname server host name
      \param port server port number
      \param interval_msec timeout interval for select() by milli second
      \return true if connection is created.
     */
    bool connectTo( const char * hostname,
                    const int port,
                    const long & interval_msec );

    /*!
      \brief set new interval time for select()
      \param interval_msec new interval by milli second
     */
    void setIntervalMSec( const long & interval_msec )
      {
          if ( interval_msec <= 0 ) return;
          M_interval_msec = interval_msec;
      }

    /*!
      \brief set server status
      \param alive server status flag. if server is dead, this value becomes false.
     */
    void setServerAlive( const bool alive )
      {
          M_server_alive = alive;
      }

    /*!
      \brief init interval status using command line options
      \param cmd_parser command line parser object
      \return derived class should return the status of initialization result.

      This method is called from init(argc,argv);
      This method must be overrided in the derived class.
    */
    virtual
    bool doInit( CmdLineParser & cmd_parser ) = 0;

    /*!
      \brief create connection to the server

      This method must be overrided in the derived class.
      This method is called on the top of run() method.
      Do *NOT* call this method by yourself.
      You should use connectTo() in the definition of this method.
     */
    virtual
    void doConnect() = 0;

    /*!
      \brief (pure virtual) send init command to the server

      This method must be overrided in the derived class.
      This method should send init command to the server.
      BUT this method is called at the top of run() method.
      Do *NOT* call this method by yourself.
     */
    virtual
    void sendInitCommand() = 0;

    /*!
      \brief (pure virtual) send bye command to the server

      This method must be overrided in the derived class.
      This method also should call "setServerAlive( false );"
     */
    virtual
    void sendByeCommand() = 0;

    /*!
      \brief (pure virtual) handle server messege

      This method is called when server message is arrived.
     */
    virtual
    void handleMessage() = 0;

    /*!
      \brief (pure virtual) handle server messege

      This method is called when select() timeout occurs
     */
    virtual
    void handleTimeout() = 0;

    /*!
      \brief (pure virtual) handle exit event

      This method is called when client estimates server is not alive.
     */
    virtual
    void handleExit() = 0;


    /*!
      \brief send raw string to the server
      \param msg message to be sent
      \return result of ::sendto()
     */
    int sendMessage( const char * msg );

    /*!
      \brief receive server message in the socket queue
      \return length of received message
     */
    int recvMessage();

    /*!
      \brief check server alive status
      \return true if client can estimate server is alive
     */
    bool isServerAlive() const
      {
          return M_server_alive;
      }

    /*!
      \brief get recieved message buffer
      \return const pointer to the message buffer
     */
    const
    char * recvBuf() const
      {
          return M_recv_buf;
      }

    /*!
      \brief get current waiting time
      \return milli second value of waiting time
     */
    const
    long & waitedMSec() const
      {
          return M_waited_msec;
      }

    /*!
      \brief get the number of timeout without server message
      \return number of timeout count
     */
    int timeoutCount() const
      {
          return M_timeout_count;
      }

    /*!
      \brief get last server message receive time
      \return const reference to the time stamp object
     */
    const
    rcsc::TimeStamp & lastRecvTime() const
      {
          return M_last_recv_time;
      }

};

}

#endif
