// -*-c++-*-

/*!
  \file coach_agent.h
  \brief basic coach agent Header File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef RCSC_COACH_COACH_AGENT_H
#define RCSC_COACH_COACH_AGENT_H

#include <rcsc/coach/global_world_model.h>
#include <rcsc/coach/coach_config.h>
#include <rcsc/client/basic_client.h>

#include <boost/scoped_ptr.hpp>

#include <string>

namespace rcsc {

struct CoachAgentImpl;
class CoachCommand;
class GlobalVisualSensor;

class CoachAgent
    : public BasicClient {
private:

    //! internal implementation object
    boost::scoped_ptr< CoachAgentImpl > M_impl;

    //! configuration parameter set
    CoachConfig M_config;

    //! internal memory of field status
    GlobalWorldModel M_worldmodel;

public:
    /*!
      \brief init member variables
     */
    CoachAgent();
    /*!
      \brief delete dynamic allocated memory
     */
    virtual
    ~CoachAgent();

    /*!
      \brief get configuration set
      \return const reference to the configuration class object
     */
    const
    CoachConfig & config() const
      {
          return M_config;
      }

    /*!
      \brief get field status
      \return const reference to the worldmodel instance
     */
    const
    GlobalWorldModel & world() const
      {
          return M_worldmodel;
      }

    /*!
      \brief get visual sensor.
      \return const reference to the visual sensor instance.
     */
    const
    GlobalVisualSensor & visualSensor() const;

    /*!
      \brief finalize program process
     */
    void finalize()
      {
          handleExit();
      }

protected:
    /*!
      \brief analyze command line options
      \param cmd_parser command lien parser object
      \return only if "help" option is given, false is returned.

      This method is called from BasicClient::init( argc, argv )
      BasicClient::init(argc,argv) should be called in main().
     */
    virtual
    bool doInit( CmdLineParser & cmd_parser );

    /*!
      \brief create connection to the server

      This method is called on the top of BasicClient::run() method.
      Do NOT call this method by yourself!
     */
    virtual
    void doConnect();

    /*!
      \brief send init or reconnect command to server

      init commad is sent in BasicClient's run() method
      Do not call this method yourself!
     */
    virtual
    void sendInitCommand();

    /*!
      \brief send disconnection command message to server
      and set the server status to end.
     */
    virtual
    void sendByeCommand();

    /*!
      \brief handle server message event
     */
    virtual
    void handleMessage();

    /*!
      \brief handle select timeout
     */
    virtual
    void handleTimeout();

    /*!
      \brief handle exit event
     */
    virtual
    void handleExit();

private:
    /*!
      \brief launch each parsing method depending on the message type
      \param msg raw server message
     */
    void parse( const char * msg );

    /*!
      \brief analyze init replay message
      \paramm msg raw server message
    */
    void analyzeInit( const char * msg );

    /*!
      \brief analyze cycle info in server message
      \param msg raw server message
      \param by_see_global if message type is see_global, this value becomes true
      \return parsing result status
     */
    bool analyzeCycle( const char * msg,
                       const bool by_see_global );

    /*!
      \brief analyze see_global message
      \paramm msg raw server message
     */
    void analyzeSeeGlobal( const char * msg );

    /*!
      \brief analyze hear message
      \paramm msg raw server message
     */
    void analyzeHear( const char * msg );

    /*!
      \brief analyze referee message
      \paramm msg raw server message
     */
    void analyzeHearReferee( const char * msg );

    /*!
      \brief analyze audio message from player
      \paramm msg raw server message
    */
    void analyzeHearPlayer( const char * msg );

    /*!
      \brief analyze change_player_type message
      \paramm msg raw server message
    */
    void analyzeChangePlayerType( const char * msg );

    /*!
      \brief analyze player_type parameter message
      \paramm msg raw server message
    */
    void analyzePlayerType( const char * msg );

    /*!
      \brief analyze player_param parameter message
      \paramm msg raw server message
    */
    void analyzePlayerParam( const char * msg );

    /*!
      \brief analyze server_param parameter message
      \paramm msg raw server message
    */
    void analyzeServerParam( const char * msg );

    /*!
      \brief analyze clang version message
      \paramm msg raw server message
    */
    void analyzeCLangVer( const char * msg );

    /*!
      \brief analyze ok message
      \paramm msg raw server message
    */
    void analyzeOK( const char * msg );

    /*!
      \brief analyze ok teamnames message
      \param msg raw server message
     */
    void analyzeTeamNames( const char * msg );

    /*!
      \brief analyze error message
      \paramm msg raw server message
    */
    void analyzeError( const char * msg );

    /*!
      \brief analyze warning message
      \paramm msg raw server message
    */
    void analyzeWarning( const char * msg );

    /*!
      \brief main action decision. this method is called from handleMessage()
    */
    void action();

    /*!
      \brief send command string to the rcssserver
      \param com coach command object
      \return true if command is sent
     */
    bool sendCommand( const CoachCommand & com );

public:
    /*!
      \brief send check_ball command
      \return true if command is generated and sent
    */
    bool doCheckBall();

    /*!
      \brief send look command
      \return true if command is generated and sent
    */
    bool doLook();

    /*!
      \brief send team_name command
      \return true if command is generated and sent
    */
    bool doTeamNames();

    //bool doTeamGraphic();

    /*!
      \brief send eye command
      \brief on if true, send (eye on), else (eye off)
      \return true if command is generated and sent
    */
    bool doEye( bool on );

    /*!
      \brief send change_player_type command
      \param unum target player's uniform number
      \param type new player type Id
      \return true if command is generated and sent
    */
    bool doChangePlayerType( const int unum,
                             const HeteroID type );

    /*!
      \brief send freeform message by say command
      \return true if command is generated and sent
    */
    bool doSayFreeform( const std::string & msg );


    //bool doSendCLang( const CLang & lang );

protected:

    /*!
      \brief pure virtual method. register decision.

      This method is used to set coach's action.
      This method is called from action().
      So, do *NOT* call this method by yourself.
    */
    virtual
    void doAction() = 0;
};

}

#endif
