// -*-c++-*-

/*!
  \file audio_sensor.cpp
  \brief audio message analyzer Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "audio_sensor.h"
#include "logger.h"

#include <rcsc/math_util.h>

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

 */
AudioSensor::AudioSensor()
    : M_time( 0, 0 )
    , M_ball_pos( 0.0, 0.0 )
    , M_ball_vel( 0.0, 0.0 )
    , M_receiver_number( 0 )
    , M_receive_pos( 50.0, 0.0 )
    , M_goalie_number( 0 )
    , M_goalie_pos( 50.0, 0.0 )
    , M_offside_line_x( 100000.0 )
    , M_defense_line_x( -100000.0 )
    , M_intercept_time( 0, 0 )
    , M_intercept_number( 0 )
    , M_intercept_cycle( 1000 )
    , M_hey_pass_time( 0, 0 )
    , M_hey_pass_number( 0 )
    , M_hey_pass_pos( 0.0, 0.0 )
{

}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parsePlayer( const char * msg,
                          const GameTime & current )
{
    /*
      players' communication audio format

      // from self
      v7-: (hear <TIME> self <MSG>)
      v7+: (hear <TIME> self "<MSG>")

      // from other
      v7-: (hear <TIME> <DIR> <MSG>)
      v7:  (hear <TIME> <DIR> "<MSG>")
      v8+: (hear <TIME> <DIR> our <UNUM> "<MSG>")
      (hear <TIME> <DIR> opp "<MSG>")
      (hear <TIME> our)
      (hear <TIME> opp)
    */

    M_time = current;

    long cycle;
    double dir;
    int unum;
    int n_read = 0;

    // v8+ complete message
    if ( std::sscanf( msg, " (hear %ld %lf our %d %n ",
                      &cycle, &dir, &unum, &n_read ) != 3 )
    {
        std::cerr << "***ERROR*** AudioSensor::parse."
                  << " got partial or opponent audio?? ["
                  << msg << "]"
                  << std::endl;
        return;
    }

    msg += n_read;


    while ( *msg != '\0' && *msg == ' ' ) ++msg;
    if ( *msg == '\"' ) ++msg;

    std::size_t len = 0;
    while ( *(msg + len) != '\0'
            && *(msg + len) != '\"' )
    {
        ++len;
    }

    if ( len == 0 )
    {
        std::cerr << "***ERROR*** AudioSensor::parsePlayer."
                  << " Empty message "
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Empty message" );
        return;
    }

    M_player_message.message_.assign( msg, len );

    M_player_message.time_ = current;
    M_player_message.side_ = NEUTRAL;
    M_player_message.unum_ = unum;
    M_player_message.type_ = PlayerMessage::ILLEGAL;
    M_player_message.dir_ = dir;

    parseTeammateAudioV8();
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parseCoach( const char * msg,
                         const GameTime & current )
{
    // (hear <time> online_coach_{left,right} <msg>) : v7-
    // (hear <time> online_coach_{left,right} (freeform "<msg>")) : v7+
    // (hear <time> online_coach_{left,right} <clang>)   : v7+

    M_time = current;

    // skip "(hear "
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    // skip "<time> "
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    // skip sender
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    if ( *msg != '(' )
    {
        // not a clang message
        M_freeform_message.message_.assign( msg );
        if ( ! M_freeform_message.message_.empty()
             && *M_freeform_message.message_.rbegin() == ')' )
        {
            M_freeform_message.message_.erase( M_freeform_message.message_.length() - 1 );
        }

        return;
    }

    // clang message

    int n_read = 0;
    char msg_type[32];

    if ( std::sscanf( msg, "(%[^ ] %n ",
                      msg_type, &n_read ) != 1 )
    {
        std::cerr << "***ERROR*** failed to parse clang message type. ["
                  << msg
                  << std::endl;
        return;
    }

    msg += n_read;

    if ( std::strcmp( msg_type, "freeform" ) != 0 )
    {
        // not a freeform message
        std::cerr << current << ": "
                  << "recv not a freeform message. "
                  << msg_type
                  << std::endl;
        return;
    }

    bool quoted = false;
    if ( *msg != '\0'
         && *msg == '\"' )
    {
        ++msg;
        quoted = true;
    }

    M_freeform_message.time_ = current;
    M_freeform_message.message_.assign( msg );

    // remove quotation
    if ( quoted )
    {
        std::string::size_type qpos = M_freeform_message.message_.find_last_of( '\"' );
        if ( qpos != std::string::npos )
        {
            M_freeform_message.message_.erase( qpos );
        }

        std::cerr << current << ": "
                  << "recv freeform message. ["
                  << M_freeform_message.message_ << ']'
                  << std::endl;
        return;
    }

    // remove last two paren
    for ( int i = 0; i < 2; ++i )
    {
        std::string::size_type ppos = M_freeform_message.message_.find_last_of( ')' );
        if ( ppos != std::string::npos )
        {
            M_freeform_message.message_.erase( ppos );
        }
    }
}


/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parseTrainer( const char * msg,
                           const GameTime & current )
{

    // (hear <time> referee <msg>) : v7-
    // (hear <time> coach "<msg>")   : v7+
    // (hear <time> coach <clang>)   : v7+

    M_time = current;
    M_trainer_message.time_ = current;
    M_trainer_message.message_.erase();

    int n_read = 0;
    long cycle;
    char sender[32];

    if ( std::sscanf( msg, "(hear %ld %s %n ",
                      &cycle, sender, &n_read ) != 2 )
    {
        std::cerr << "***ERRORR*** failed to parse trainer message. ["
                  << msg << ']'
                  << std::endl;
        return;
    }

    msg += n_read;

    bool quoted = false;
    if ( *msg != '\0'
         && *msg == '\"' )
    {
        ++msg;
        quoted = true;
    }

    M_trainer_message.message_.assign( msg );

    // remove quotation
    if ( quoted )
    {
        std::string::size_type qpos = M_trainer_message.message_.find_last_of( '\"' );
        if ( qpos != std::string::npos )
        {
            M_trainer_message.message_.erase( qpos );
        }
        return;
    }

    // remove last paren
    std::string::size_type ppos = M_trainer_message.message_.find_last_of( ')' );
    if ( ppos != std::string::npos )
    {
        M_trainer_message.message_.erase( ppos );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::parseTeammateAudioV8()
{
    if ( M_player_message.message_.empty() )
    {
        return false;
    }

    const char * msg = M_player_message.message_.c_str();

    // ball info
    if ( ! std::strncmp( msg, "b ", 2 ) )
    {
        if ( M_player_message.message_.length() != 8 )
        {
            std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                      << " Illega ball info [" << msg << "]"
                      << std::endl;
            dlog.addText( Logger::SENSOR,
                          "AudioSensor: Illegal ball info [%s]",
                          msg );
            return false;
        }

        std::string ball_str = M_player_message.message_.substr( 2, 6 );
        if ( ! M_codec.decodeL6( ball_str, &M_ball_pos, &M_ball_vel ) )
        {
            std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                      << " Failed to decode ball [" << msg
                      << "]  str=["  << ball_str << "]"
                      << std::endl;
            dlog.addText( Logger::SENSOR,
                          "AudioSensor: Failed to decode Ball Info [%s]",
                          msg );

            return false;
        }

        M_player_message.type_ = PlayerMessage::BALL_INFO;
        return true;
    }
    ////////////////////////////////////////////////////////
    // pass info
    if ( ! std::strncmp( msg, "p ", 2 ) )
    {
        if ( parsePassInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::PASS_INFO;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode pass [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Pass Info [%s]",
                      msg );
        return false;
    }

    ////////////////////////////////////////////////////////
    // goalie info
    if ( ! std::strncmp( msg, "g ", 2 ) )
    {
        if ( parseGoalieInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::OPPONENT_GOALIE_INFO;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode goalie [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Goalie Info [%s]",
                      msg );
        return false;
    }

    ////////////////////////////////////////////////////////
    // offside line
    if ( ! std::strncmp( msg, "o ", 2 ) )
    {
        if ( parseOffsideLineInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::OFFSIDE_LINE_INFO;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode offside line [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Offside Line Info [%s]",
                      msg );
        return false;
    }
    ////////////////////////////////////////////////////////
    // defense line
    if ( ! std::strncmp( msg, "d ", 2 ) )
    {
        if ( parseDefenseLineInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::DEFENSE_LINE_INFO;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode defense line [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Defense Line Info [%s]",
                      msg );
        return false;
    }

    ////////////////////////////////////////////////////////
    // wait request
    if ( ! std::strncmp( msg, "w", 1 ) )
    {
        M_player_message.type_ = PlayerMessage::WAIT_REQUEST;
        return true;
    }

    ////////////////////////////////////////////////////////
    // attack request
    if ( ! std::strncmp( msg, "a", 1 ) )
    {
        M_player_message.type_ = PlayerMessage::ATTACK_REQUEST;
        return true;
    }

    ////////////////////////////////////////////////////////
    // intercept info
    if ( ! std::strncmp( msg, "i ", 2 ) )
    {
        if ( parseInterceptInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::INTERCEPT;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode intercept info [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode intercept info [%s]",
                      msg );
        return false;
    }

    ////////////////////////////////////////////////////////
    // hey pass info
    if ( ! std::strncmp( msg, "h ", 2 ) )
    {
        M_player_message.type_ = PlayerMessage::HEY_PASS;
        return true;
    }


    //std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
    //          << " Failed to decode ["
    //          << raw_message << "]" << std::endl;
    dlog.addText( Logger::SENSOR,
                  "AudioSensor: unsupported message [%s]",
                  M_player_message.message_.c_str() );

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::parsePassInfo( const char * msg )
{
    if ( *msg != 'p' )
    {
        return false;
    }

    // skip space
    msg += 2;

    ///////////////////////////////////////
    // read recever number
    switch ( *msg ) {
    case '1': M_receiver_number = 1; break;
    case '2': M_receiver_number = 2; break;
    case '3': M_receiver_number = 3; break;
    case '4': M_receiver_number = 4; break;
    case '5': M_receiver_number = 5; break;
    case '6': M_receiver_number = 6; break;
    case '7': M_receiver_number = 7; break;
    case '8': M_receiver_number = 8; break;
    case '9': M_receiver_number = 9; break;
    case 'A': M_receiver_number = 10; break;
    case 'B': M_receiver_number = 11; break;
    default:
        std::cerr << "AudioSensor:: Unexpected receiver number " << msg
                  << std::endl;
        return false;
    }

    // skip space
    msg += 2;

    ///////////////////////////////////////
    // read x coordinate
    int x = 0, y = 0;

    x += 100 * (*msg - '0'); ++msg;
    x += 10 * (*msg - '0'); ++msg;
    x += (*msg - '0'); ++msg;

    y += 100 * (*msg - '0'); ++msg;
    y += 10 * (*msg - '0'); ++msg;
    y += (*msg - '0'); ++msg;

    M_receive_pos.x = static_cast< double >( x ) * 0.5 - 52.5;
    M_receive_pos.y = static_cast< double >( y ) * 0.5 - 34.0;


    dlog.addText( Logger::SENSOR,
                  "AudioSensor: success!  receiver= %d  receive_pos(%f %f)",
                  M_receiver_number,
                  M_receive_pos.x, M_receive_pos.y );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeBall( const Vector2D & ball_pos,
                         const Vector2D & ball_vel,
                         std::string & say_buf ) const
{
    say_buf = "b ";
    std::string msg = M_codec.encodeL6( ball_pos, ball_vel );
    if ( msg.empty() )
    {
        say_buf.erase();
    }
    else
    {
        say_buf += msg;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodePass( const int receiver,
                         const Vector2D & receive_pos,
                         std::string & say_buf ) const
{
    say_buf = "p ";

    switch ( receiver ) {
    case 1: say_buf += "1 "; break;
    case 2: say_buf += "2 "; break;
    case 3: say_buf += "3 "; break;
    case 4: say_buf += "4 "; break;
    case 5: say_buf += "5 "; break;
    case 6: say_buf += "6 "; break;
    case 7: say_buf += "7 "; break;
    case 8: say_buf += "8 "; break;
    case 9: say_buf += "9 "; break;
    case 10: say_buf += "A "; break;
    case 11: say_buf += "B "; break;
    default:
        std::cerr << "AudioSensor::encode Unexpected receiver number "
                  << receiver << std::endl;
        say_buf.erase();
        return;
    }

    double tmp = min_max( -52.5, receive_pos.x, 52.5 );
    tmp += 52.5;
    tmp *= 2.0;

    int ix = static_cast< int >( rint( tmp ) );

    tmp = min_max( -34.0, receive_pos.y, 34.0 );
    tmp += 34.0;
    tmp *= 2.0;

    int iy = static_cast< int >( rint( tmp ) );


    char xymsg[8];
    std::snprintf( xymsg, 8, "%03d%03d", ix, iy );
    if ( std::strlen( xymsg ) != 6 )
    {
        std::cerr << "AudioSensor::encode XY error "
                  << receive_pos << " -> " << xymsg << std::endl;
        say_buf.erase();
        return;
    }

    say_buf += xymsg;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeGoalie( const int goalie_number,
                           const Vector2D & goalie_pos,
                           std::string & say_buf ) const
{
    say_buf = "g ";

    char hex = AudioCodec::unum2hex( goalie_number );
    if ( hex == '\0' )
    {
        std::cerr << "AudioSensor::encodeGoalieInfo. "
                  << "Unexpected goalie number "
                  << goalie_number << std::endl;
        say_buf.erase();
        return;
    }

    say_buf += hex;

    std::string pos_str = codec().encodePosToStr7( goalie_pos );
    if ( pos_str.length() != 7 )
    {
        std::cerr << "AudioSensor::encodeGoalieInfo. "
                  << "Failed to encode goalie info "
                  << "( " << goalie_pos.x << goalie_pos.y << ")"
                  << std::endl;
        say_buf.erase();
    }

    say_buf += pos_str;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::parseGoalieInfo( const char * msg )
{
    if ( std::strlen( msg ) != 10
         || msg[0] != 'g'
         || msg[1] != ' ' )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected goalie message ["
                  << msg << "] len = " << std::strlen( msg )
                  << std::endl;
        return false;
    }
    msg += 2;

    int unum = AudioCodec::hex2unum( *msg );
    if ( unum == 0 )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected goalie number " << msg
                  << std::endl;
        return false;
    }

    ++msg;

    M_goalie_pos = codec().decodeStr7ToPos( msg );

    if ( ! M_goalie_pos )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << " Failed to decode goalie potiiton"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode goalie potiiton" );
        return false;
    }

    dlog.addText( Logger::SENSOR,
                  "AudioSensor: success!  goalie pos = (%.2f %.2f)",
                  M_goalie_pos.x, M_goalie_pos.y );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeOffsideLineX( const double & offside_line_x,
                                 std::string & say_buf ) const
{
    char buf[10];
    std::snprintf( buf, 10, "o %.1f", offside_line_x );
    say_buf = buf;
}

/*-------------------------------------------------------------------*/
/*!
  "o <x>"
 */
bool
AudioSensor::parseOffsideLineInfo( const char * msg )
{
    if ( msg[0] != 'o'
         || msg[1] != ' ' )
    {
        std::cerr << "AudioSensor::parseOffsideLineInfo. "
                  << "Unexpected value message ["
                  << msg << "]"
                  << std::endl;
        return false;
    }
    msg += 2;


    double val = std::strtod( msg, NULL );
    if ( val == -HUGE_VAL
         || val == HUGE_VAL )
    {
        std::cerr << "AudioSensor::parseOffsideLineInfo. "
                  << " Failed to read offside line"
                  << std::endl;
        return false;
    }


    M_offside_line_x = val;

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeDefenseLineX( const double & defense_line_x,
                                 std::string & say_buf ) const
{
    char buf[10];
    std::snprintf( buf, 10, "d %.1f", defense_line_x );
    say_buf = buf;
}

/*-------------------------------------------------------------------*/
/*!
  "d <x>"
 */
bool
AudioSensor::parseDefenseLineInfo( const char * msg )
{
    if ( msg[0] != 'd'
         || msg[1] != ' ' )
    {
        std::cerr << "AudioSensor::parseDefenseLineInfo. "
                  << "Unexpected value message ["
                  << msg << "]"
                  << std::endl;
        return false;
    }
    msg += 2;


    double val = std::strtod( msg, NULL );
    if ( val == -HUGE_VAL
         || val == HUGE_VAL )
    {
        std::cerr << "AudioSensor::parseDefenseLineInfo. "
                  << " Failed to read defense line"
                  << std::endl;
        return false;
    }


    M_defense_line_x = val;

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeInterceptInfo( const int unum,
                                  const int cycle,
                                  std::string & say_buf ) const
{
    char buf[10];
    std::snprintf( buf, 10, "i %d %d", unum, cycle );
    say_buf = buf;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::parseInterceptInfo( const char * msg )
{
    if ( msg[0] != 'i'
         || msg[1] != ' ' )
    {
        std::cerr << "AudioSensor::parseInterceptInfo. "
                  << "Unexpected message Id ["
                  << msg << "]"
                  << std::endl;
        return false;
    }

    int unum = 0;
    int cycle = 0;

    if ( std::sscanf( msg, "i %d %d", &unum , &cycle ) != 2 )
    {
        std::cerr << "AudioSensor::parseInterceptInfo. "
                  << "Unexpected value message(2) ["
                  << msg << "]"
                  << std::endl;
        return false;
    }

    M_intercept_time = M_time;
    M_intercept_number = unum;
    M_intercept_cycle = cycle;

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeHeyPassInfo( const int unum,
                                const Vector2D & pos,
                                std::string & say_buf ) const
{
    char hex = AudioCodec::unum2hex( unum );
    if ( hex == '\0' )
    {
        return;
    }

    say_buf = "i ";
    say_buf += hex;

    std::string pos_str = codec().encodePosToStr7( pos );
    if ( pos_str.length() != 7 )
    {
        std::cerr << "AudioSensor::encodeHeyPassInfo. "
                  << "Failed to encode hey passe info. unum="
                  << unum
                  << " ( " << pos.x << pos.y << ")"
                  << std::endl;
        say_buf.erase();
    }

    say_buf += pos_str;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::parseHeyPassInfo( const char * msg )
{
    if ( std::strlen( msg ) != 10
         || msg[0] != 'h'
         || msg[1] != ' ' )
    {
        std::cerr << "AudioSensor::parseHeyPassInfo. "
                  << "Unexpected hey pass message ["
                  << msg << "] len = " << std::strlen( msg )
                  << std::endl;
        return false;
    }
    msg += 2;

    M_hey_pass_number = AudioCodec::hex2unum( *msg );
    if ( M_hey_pass_number == 0 )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected goalie number " << msg
                  << std::endl;
        return false;
    }

    ++msg;

    M_hey_pass_pos = codec().decodeStr7ToPos( msg );

    if ( ! M_hey_pass_pos )
    {
        std::cerr << "AudioSensor::parseHeyPassInfo. "
                  << " Failed to decode hey pass potiiton"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode hey pass potiiton" );
        return false;
    }

    dlog.addText( Logger::SENSOR,
                  "AudioSensor: success!  hey pass pos = (%.2f %.2f)",
                  M_hey_pass_pos.x, M_hey_pass_pos.y );

    M_hey_pass_time = M_time;
    return true;
}

}
