// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "sample_coach.h"

#include <rcsc/coach/coach_command.h>
#include <rcsc/coach/coach_config.h>
#include <rcsc/coach/global_visual_sensor.h>
#include <rcsc/common/basic_client.h>
#include <rcsc/common/player_param.h>
#include <rcsc/common/server_param.h>
#include <rcsc/common/player_type.h>
#include <rcsc/param/param_map.h>
#include <rcsc/param/cmd_line_parser.h>

#include <functional>
#include <algorithm>
#include <vector>
#include <sstream>
#include <iostream>
#include <cstdio>

/////////////////////////////////////////////////////////

struct IsIgnoredStaminaIncMax
    : public std::unary_function< const rcsc::PlayerType *, bool > {
    const double M_ignore_thr;

    IsIgnoredStaminaIncMax( const double & thr )
        : M_ignore_thr( thr )
      { }

    result_type operator()( argument_type arg ) const
      {
          return arg->staminaIncMax() < M_ignore_thr;
      }
};


/////////////////////////////////////////////////////////

struct IsIgnoredSpeedMax
    : public std::unary_function< const rcsc::PlayerType *, bool > {
    const double M_ignore_thr;

    IsIgnoredSpeedMax( const double & thr )
        : M_ignore_thr( thr )
      { }

    result_type operator()( argument_type arg ) const
      {
          return arg->realSpeedMax() < M_ignore_thr;
      }
};


/////////////////////////////////////////////////////////

struct TypeStaminaIncComp
    : public std::binary_function< const rcsc::PlayerType *,
                                   const rcsc::PlayerType *,
                                   bool > {

    result_type operator()( first_argument_type lhs,
                            second_argument_type rhs ) const
      {
          return lhs->staminaIncMax() > rhs->staminaIncMax();
      }

};

/////////////////////////////////////////////////////////

struct RealSpeedMaxCmp
    : public std::binary_function< const rcsc::PlayerType *,
                                   const rcsc::PlayerType *,
                                   bool > {

    result_type operator()( first_argument_type lhs,
                            second_argument_type rhs ) const
      {
          return lhs->realSpeedMax() < rhs->realSpeedMax();
      }

};

/////////////////////////////////////////////////////////

struct MaxSpeedReachStepCmp
    : public std::binary_function< const rcsc::PlayerType *,
                                   const rcsc::PlayerType *,
                                   bool > {

    result_type operator()( first_argument_type lhs,
                            second_argument_type rhs ) const
      {
          return lhs->cyclesToReachMaxSpeed() > rhs->cyclesToReachMaxSpeed();
      }

};

/*-------------------------------------------------------------------*/
/*!

 */
SampleCoach::SampleCoach()
    : CoachAgent()
{
    for ( int i = 0; i < 11; ++i )
    {
        // init map values
        M_player_type_id[i] = rcsc::Hetero_Default;
    }

    M_center_defender_unum.push_back( 2 );
    M_center_defender_unum.push_back( 3 );
    M_center_defender_unum.push_back( 6 );

    M_side_defender_unum.push_back( 4 );
    M_side_defender_unum.push_back( 5 );

    M_midfielder_unum.push_back( 7 );
    M_midfielder_unum.push_back( 8 );

    M_forward_unum.push_back( 9 );
    M_forward_unum.push_back( 10 );
    M_forward_unum.push_back( 11 );
}

/*-------------------------------------------------------------------*/
/*!

*/
SampleCoach::~SampleCoach()
{

}

/*-------------------------------------------------------------------*/
/*!

 */
bool
SampleCoach::initImpl( rcsc::CmdLineParser & cmd_parser )
{
    rcsc::ParamMap my_params;

#if 0
    std::string formation_conf;
    my_map.add()
        ( &conf_path, "fconf" )
        ;
#endif

    if ( ! rcsc::CoachAgent::initImpl( cmd_parser ) )
    {
        my_params.printHelp( std::cout );
        M_client->setServerAlive( false );
        return false;
    }

    if ( cmd_parser.failed() )
    {
        std::cerr << "coach: ***WARNING*** detected invalid options: ";
        cmd_parser.print( std::cerr );
        std::cerr << std::endl;
    }

    //////////////////////////////////////////////////////////////////
    // Add your code here.
    //////////////////////////////////////////////////////////////////


    return true;
}


/*-------------------------------------------------------------------*/
/*!

 */
void
SampleCoach::actionImpl()
{
    doSubstitute();

    //visualSensor().print( std::cout );
    //world().print( std::cout );
#if 0
    if ( world().canSendFreeform() )
    {
        char msg[128];
        std::sprintf( msg, "freeform message %d/%d",
                      world().freeformSendCount(),
                      world().freeformAllowedCount() );
        doSayFreeform( msg );
    }
#endif
#if 1
    if ( world().time().cycle() > 0 )
    {
        M_client->setServerAlive( false );
    }
#endif
}

/*-------------------------------------------------------------------*/
/*!

 */
void
SampleCoach::doSubstitute()
{
    static bool S_first_substituted = false;

    if ( ! S_first_substituted
         && world().time().cycle() == 0
         && world().time().stopped() > 10 )
    {
        doFirstSubstitute();
        S_first_substituted = true;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
SampleCoach::doFirstSubstitute()
{
    const rcsc::PlayerTypeSet::PlayerTypeMap & type_map
        = rcsc::PlayerTypeSet::i().playerTypeMap();
    std::vector< const rcsc::PlayerType * > candidates;

    std::fprintf( stderr,
                  "id speed step inc  power  stam decay moment dprate   karea  krand effmax effmin\n" );
    for ( rcsc::PlayerTypeSet::PlayerTypeMap::const_iterator it = type_map.begin();
          it != type_map.end();
          ++it )
    {
        candidates.push_back( &(it->second) );
        std::fprintf( stderr,
                      "%d  %.3f  %2d  %.1f %5.1f %5.1f %.3f  %4.1f  %.5f  %.3f  %.2f  %.3f  %.3f\n",
                      it->second.id(),
                      it->second.realSpeedMax(),
                      it->second.cyclesToReachMaxSpeed(),
                      it->second.staminaIncMax(),
                      it->second.getDashPowerToKeepMaxSpeed( rcsc::ServerParam::i() ),
                      it->second.getOneStepStaminaComsumption( rcsc::ServerParam::i() ),
                      it->second.playerDecay(),
                      it->second.inertiaMoment(),
                      it->second.dashPowerRate(),
                      it->second.kickableArea(),
                      it->second.kickRand(),
                      it->second.effortMax(),
                      it->second.effortMin() );
    }

    // change forward
    rcsc::HeteroID type_id = getFastestType( candidates );
    substituteTo( M_forward_unum, type_id );

    type_id = getFastestType( candidates );
    substituteTo( M_center_defender_unum, type_id );

    type_id = getFastestType( candidates );
    substituteTo( M_side_defender_unum, type_id );

    type_id = getFastestType( candidates );
    substituteTo( M_midfielder_unum, type_id );

}

/*-------------------------------------------------------------------*/
/*!

 */
void
SampleCoach::substituteTo( const std::vector< int > & unum_set,
                           const rcsc::HeteroID id )
{
    if ( id == rcsc::Hetero_Default )
    {
        //std::cout << "player not changed " <<  std::endl;
        return;
    }

    if ( (int)unum_set.size() > rcsc::PlayerParam::i().ptMax() )
    {
        std::cerr << config().teamName() << " coach: "
                  << " ***WARNING** too many player "
                  << unum_set.size()
                  << " to substitute to " << id
                  << std::endl;
    }

    if ( M_used_id.find( id ) != M_used_id.end() )
    {
        std::cerr << config().teamName() << " coach: "
                  << " ***ERROR*** player type "  << id << "  is already used."
                  << std::endl;
        return;
    }

    for ( std::vector< int >::const_iterator it = unum_set.begin();
          it != unum_set.end();
          ++it )
    {
        std::cout << config().teamName() << " coach: "
                  << "change player " << *it
                  << " to type " << id
                  << std::endl;
        M_player_type_id.insert( std::make_pair( *it, id ) );

        doChangePlayerType( *it, id );
    }

    // register used id
    M_used_id.insert( id );
}

/*-------------------------------------------------------------------*/
/*!

 */
rcsc::HeteroID
SampleCoach::getFastestType( const PlayerTypePtrCont & candidates )
{
    if ( candidates.empty() )
    {
        return rcsc::Hetero_Default;
    }

    double default_speed_max = 1.0;
    // prepare type list
    std::vector< const rcsc::PlayerType * > tmplist;
    for ( PlayerTypePtrCont::const_iterator it = candidates.begin();
          it != candidates.end();
          ++it )
    {
        if ( M_used_id.find( (*it)->id() ) != M_used_id.end() )
        {
            // already used
            continue;
        }
        if ( (*it)->id() == 0 )
        {
            default_speed_max = (*it)->realSpeedMax();
        }
        tmplist.push_back( *it );
    }

    // sort by max speed
    std::sort( tmplist.begin(), tmplist.end(), std::not2( RealSpeedMaxCmp() ) );

    const rcsc::PlayerType * besttype = tmplist.front();
    double max_speed = besttype->realSpeedMax();
    int min_cycle = 100;
    for ( PlayerTypePtrCont::iterator it = tmplist.begin();
          it != tmplist.end();
          ++it )
    {
        if ( (*it)->realSpeedMax() < default_speed_max )
        {
            // slower than default type
            break;
        }
        if ( (*it)->realSpeedMax() < max_speed - 0.01 )
        {
            break;
        }
        if ( (*it)->cyclesToReachMaxSpeed() < min_cycle )
        {
            besttype = (*it);
            max_speed = besttype->realSpeedMax();
            min_cycle = besttype->cyclesToReachMaxSpeed();
            continue;
        }
        if ( (*it)->cyclesToReachMaxSpeed() == min_cycle )
        {
            if ( (*it)->getOneStepStaminaComsumption( rcsc::ServerParam::i() )
                 < besttype->getOneStepStaminaComsumption( rcsc::ServerParam::i()) )
            {
                besttype = (*it);
                max_speed = besttype->realSpeedMax();
            }
        }
    }

    return besttype->id();
}
