// -*-c++-*-

/*!
  \file serializer.cpp
  \brief rcg serializer Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif

#include "serializer.h"

#include "util.h"

#include <algorithm>
#include <cstring>

#ifndef M_PI
#define M_PI 3.14159265358979323846
#endif

namespace rcsc {
namespace rcg {

/*-------------------------------------------------------------------*/
/*!

*/
Serializer::Serializer()
    : M_playmode( static_cast< char >( 0 ) )
{
    for ( int i = 0; i < 2; ++i )
    {
        std::memset( M_teams[i].name, 0, 16 );
        M_teams[i].score = 0;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const int version )
{
    if ( version == REC_OLD_VERSION )
    {
        // v1 protocl does not have header.
        return os;
    }

    char buf[5];

    buf[0] = 'U';
    buf[1] = 'L';
    buf[2] = 'G';
    buf[3] = static_cast< char >( version );

    os.write( buf, 4 );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const server_params_t & param )
{
    short mode = htons( PARAM_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    os.write( reinterpret_cast< const char * >( &param ),
              sizeof( server_params_t ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const player_params_t & pparam )
{
    short mode = htons( PPARAM_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    os.write( reinterpret_cast< const char * >( &pparam ),
              sizeof( player_params_t ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const player_type_t & type )
{
    short mode = htons( PT_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    os.write( reinterpret_cast< const char * >( &type ),
              sizeof( player_type_t ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const team_t & team_l,
                           const team_t & team_r )
{
    short mode = htons( TEAM_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    os.write( reinterpret_cast< const char * >( &team_l ),
              sizeof( team_t ) );

    os.write( reinterpret_cast< const char * >( &team_r ),
              sizeof( team_t ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const char pmode )
{
    short mode = htons( PM_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    os.write( reinterpret_cast< const char * >( &pmode ),
              sizeof( char ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const PlayMode pmode )
{
    short mode = htons( PM_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    char pm = static_cast< char >( pmode );

    os.write( reinterpret_cast< char * >( &pm ),
              sizeof( char ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const dispinfo_t & disp )
{
    os.write( reinterpret_cast< const char * >( &disp ),
              sizeof( dispinfo_t ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const showinfo_t & show )
{
    short mode = htons( SHOW_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    os.write( reinterpret_cast< const char * >( &show ),
              sizeof( showinfo_t ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const showinfo_t2 & show2 )
{
    if ( M_playmode != show2.pmode )
    {
        M_playmode = show2.pmode;

        serializeImpl( os, show2.pmode );
    }

    if ( std::strlen( M_teams[0].name ) != std::strlen( show2.team[0].name )
         || std::strcmp( M_teams[0].name, show2.team[0].name )
         || M_teams[0].score != show2.team[0].score
         || std::strlen( M_teams[1].name ) != std::strlen( show2.team[1].name )
         || std::strcmp( M_teams[1].name, show2.team[1].name )
         || M_teams[1].score != show2.team[1].score )
    {
        std::strncpy( M_teams[0].name, show2.team[0].name,
                      sizeof( M_teams[0].name ) );
        M_teams[0].score = show2.team[0].score;
        std::strncpy ( M_teams[1].name, show2.team[1].name,
                       sizeof( M_teams[1].name ) );;
        M_teams[1].score = show2.team[1].score;

        serializeImpl( os, show2.team[0], show2.team[1] );
    }

    short_showinfo_t2 short_show2;

    short_show2.ball = show2.ball;

    for ( int i = 0; i < MAX_PLAYER * 2; ++i )
    {
        short_show2.pos[i] = show2.pos[i];
    }

    short_show2.time = show2.time;

    serializeImpl( os, short_show2 );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const short_showinfo_t2 & show2 )
{
    short mode = htons( SHOW_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    os.write( reinterpret_cast< const char * >( &show2 ),
              sizeof( short_showinfo_t2 ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const msginfo_t & msg )
{
    short mode = htons( MSG_MODE );

    os.write( reinterpret_cast< char * >( &mode ),
              sizeof( short ) );

    os.write( reinterpret_cast< const char * >( &msg ),
              sizeof( msginfo_t ) );

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::ostream &
Serializer::serializeImpl( std::ostream & os,
                           const dispinfo_t2 & disp2 )
{
    switch ( ntohs( disp2.mode ) ) {
    case SHOW_MODE:
        serializeImpl( os, disp2.body.show );
        break;
    case MSG_MODE:
        serializeImpl( os, disp2.body.msg );
        break;
    case DRAW_MODE:
        break;
    case BLANK_MODE:
        break;
    case PM_MODE:
        serializeImpl( os, disp2.body.show.pmode );
        break;
    case TEAM_MODE:
        serializeImpl( os,
                       disp2.body.show.team[0],
                       disp2.body.show.team[1] );
        break;
    case PT_MODE:
        serializeImpl( os, disp2.body.ptinfo );
        break;
    case PARAM_MODE:
        serializeImpl( os, disp2.body.sparams );
        break;
    case PPARAM_MODE:
        serializeImpl( os, disp2.body.pparams );
        break;
    default:
        break;
    }

    return os;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Serializer::convert( const std::string & name,
                     const int score,
                     team_t & to )
{
    std::memset( to.name, 0, sizeof( to.name ) );
    std::strncpy( to.name,
                  name.c_str(),
                  std::min( sizeof( to.name ), name.length() ) );
    to.score = hitons( score );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
Serializer::convert( const showinfo_t2 & from,
                     showinfo_t & to )
{
    // playmode
    to.pmode = from.pmode;

    // team
    for ( int i = 0; i < 2; ++i )
    {
        std::strncpy( to.team[i].name, from.team[i].name, 16 );
        to.team[i].score = from.team[i].score;
    }

    // ball
    to.pos[0].side = htons( NEUTRAL );
    to.pos[0].x = hdtons( nltohd( from.ball.x ) );
    to.pos[0].y = hdtons( nltohd( from.ball.y ) );

    // left team
    short unum = 1;
    for ( int i = 0; i < MAX_PLAYER; ++i, ++unum )
    {
        to.pos[i+1].enable = from.pos[i].mode;
        to.pos[i+1].side = htons( LEFT );
        to.pos[i+1].unum = htons( unum );
        to.pos[i+1].angle = htons( static_cast< short >
                                      ( nltohd( from.pos[i].body_angle )
                                        * 360.0 / M_PI ) );
        to.pos[i+1].x = hdtons( nltohd( from.pos[i].x ) );
        to.pos[i+1].y = hdtons( nltohd( from.pos[i].y ) );
    }

    // right team
    unum = 1;
    for ( int i = MAX_PLAYER; i < MAX_PLAYER *2; ++i, ++unum )
    {
        to.pos[i+1].enable = from.pos[i].mode;
        to.pos[i+1].side = htons( RIGHT );
        to.pos[i+1].unum = htons( unum );
        to.pos[i+1].angle = htons( static_cast< short >
                                      ( nltohd( from.pos[i].body_angle )
                                        * 360.0 / M_PI ) );
        to.pos[i+1].x = hdtons( nltohd( from.pos[i].x ) );
        to.pos[i+1].y = hdtons( nltohd( from.pos[i].y ) );
    }

    // time
    to.time = from.time;

}

/*-------------------------------------------------------------------*/
/*!

*/
void
Serializer::convert( const std::string & from,
                     msginfo_t & to )
{
    to.board = htons( MSG_BOARD );
    std::memset( to.message, 0, sizeof( to.message ) );
    std::strncpy( to.message,
                  from.c_str(),
                  std::min( sizeof( to.message ), from.length() ) );
}

} // end of namespace
} // end of namespace
