
#include "win_src/StepDebug_Define.h"
#include <rcsc/player/world_model.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/penalty_kick_state.h>
#include <rcsc/player/action_effector.h>
#include <rcsc/player/visual_sensor.h>
#include <rcsc/player/audio_sensor.h>
#include <rcsc/player/fullstate_sensor.h>


namespace rcsc {

	template <class T, class O>
		class Container_ReadFromDataFile
	{
	public:
		Container_ReadFromDataFile( const char* pData, T* pPC, DWORD dwRegCode )
		{
			bReturn = true;
	
			// check head

			STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pData;
			if( pHead->_dwRegCode != dwRegCode )
			{
				ASSERT(false);
				bReturn = false;
				return;
			}

			pPC->clear();
			
			// read PlayerObject
			size_t nCount = ( pHead->_dwSize - pHead->size() )/sizeof(O);
			for( size_t i=0; i<nCount; i++ )
			{
				pPC->push_back( *((O*)(pHead->_pData+i*sizeof(O))) );
			}
		};

		bool GetReturn()
		{
			return bReturn;
		};

	private:
		bool bReturn;
	};

/*
	structure of WorldModel in data log file
	_____________________________
	|	[HEAD]					|
	|___________________________|
	|	Variables				|
	|___________________________|
	|	Static Variable			|
	|___________________________|
	|	PlayerCont	[H]			|
	|___________________________|
	|	PlayerPtrCont	[H]		|
	|___________________________|
	|	M_localize	[H]			|
	|___________________________|
	|	M_intercept_table	[H]	|
	|___________________________|
	|	M_penalty_kick_state [H]|
	|___________________________|
	|	SelfObject	[H]			|
	|___________________________|

	[H] or [HEAD] means that the data includes the structure of STEPDEBUG_HEAD
*/

	bool WorldModel::ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pWM_Head = (STEPDEBUG_HEAD*)pCurrent;
		if( pWM_Head->_dwRegCode != REGCODE_WorldModel )
		{
			ASSERT(false);
			return false;
		}

		// read Variables
		{
			memcpy( (void*)this, pWM_Head->_pData, ((long)&M_localize - (long)this) );
			pCurrent = pCurrent + pWM_Head->size() + ((long)&M_localize - (long)this);
		}

		// read Static Variable
		{
			memcpy( (void*)&Vector2D::ERROR_VALUE, pCurrent, sizeof(Vector2D::ERROR_VALUE ));
			pCurrent += sizeof(Vector2D::ERROR_VALUE );

			memcpy( (void*)&Vector2D::INVALIDATED, pCurrent, sizeof(Vector2D::INVALIDATED ));
			pCurrent += sizeof(Vector2D::INVALIDATED );

//			memcpy( (void*)&AngleDeg::EPSILON, pCurrent, sizeof(AngleDeg::EPSILON ));
			pCurrent += sizeof(AngleDeg::EPSILON );

//			memcpy( (void*)&AngleDeg::DEG2RAD, pCurrent, sizeof(AngleDeg::DEG2RAD ));
			pCurrent += sizeof(AngleDeg::DEG2RAD );

//			memcpy( (void*)&AngleDeg::RAD2DEG, pCurrent, sizeof(AngleDeg::RAD2DEG ));
			pCurrent += sizeof(AngleDeg::RAD2DEG );

//			memcpy( (void*)&WorldModel::DIR_STEP, pCurrent, sizeof(WorldModel::DIR_STEP ));
			pCurrent += sizeof(WorldModel::DIR_STEP );

		}

		// read PlayerCont
		{
			if( !PlayerCont_ReadFromDataFile( pCurrent, &M_teammates ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
			
			if( !PlayerCont_ReadFromDataFile( pCurrent, &M_opponents ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

			if( !PlayerCont_ReadFromDataFile( pCurrent, &M_unknown_players ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
		}

		// read PlayerPtrCont
		{
			if( !PlayerPtrCont_ReadFromDataFile( pCurrent, &M_teammates_from_self ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
			
			if( !PlayerPtrCont_ReadFromDataFile( pCurrent, &M_opponents_from_self ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
			
			if( !PlayerPtrCont_ReadFromDataFile( pCurrent, &M_teammates_from_ball ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
			
			if( !PlayerPtrCont_ReadFromDataFile( pCurrent, &M_opponents_from_ball ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
			
		}

		// read object
		{
			if( M_localize == NULL )
				(Localization *)M_localize = new Localization( this->M_our_side );

			if( !M_localize->ReadFromDataFile( pCurrent ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

			if( !M_intercept_table->ReadFromDataFile( pCurrent ) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

			 if( !M_penalty_kick_state->ReadFromDataFile( pCurrent ) )
				 return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			if( !M_self.ReadFromDataFile( pCurrent ) )
				 return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

		}


		return true;
	};



	bool WorldModel::PlayerCont_ReadFromDataFile( const char* pData, const PlayerCont* pPC ) const
	{
/*		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pData;
		if( pHead->_dwRegCode != REGCODE_WM_PlayerCont )
		{
			ASSERT(false);
			return false;
		}

		((PlayerCont*)pPC)->clear();
		
		// read PlayerObject
		size_t nCount = ( pHead->_dwSize - pHead->size() )/sizeof(PlayerObject);
		for( size_t i=0; i<nCount; i++ )
		{
			((PlayerCont*)pPC)->push_back( *((PlayerObject*)(pHead->_pData+i*sizeof(PlayerObject))));
		}

		return true;
*/
		Container_ReadFromDataFile<PlayerCont, PlayerObject>
			cr( pData, (PlayerCont*)pPC, REGCODE_WM_PlayerCont );

		return cr.GetReturn();
	};

	bool WorldModel::PlayerPtrCont_ReadFromDataFile( const char* pData, const PlayerPtrCont* pPPC ) const
	{
		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pData;
		if( pHead->_dwRegCode != REGCODE_WM_PlayerPtrCont )
		{
			ASSERT(false);
			return false;
		}

		((PlayerPtrCont*)pPPC)->clear();
		
		// read PlayerObject
		size_t nCount = ( pHead->_dwSize - pHead->size() )/sizeof(DWORD);
		for( size_t i=0; i<nCount; i++ )
		{
			DWORD dbPlayerObjectPosition = *((DWORD*)(pHead->_pData+i*sizeof(DWORD)));
			DWORD nIndex = dbPlayerObjectPosition&0x0000FFFF;

			if( (dbPlayerObjectPosition&0xFFFF0000) == 0 )					// invalid
				continue;
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00010000 )	// M_teammates
			{
				if( nIndex < M_teammates.size() )
				{
					PlayerCont::const_iterator itr= M_teammates.begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					((PlayerPtrCont*)pPPC)->push_back( (rcsc::PlayerObject *)pPO );
				}
			}
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00020000 )	// M_opponents
			{
				if( nIndex < M_opponents.size() )
				{
					PlayerCont::const_iterator itr= M_opponents.begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					((PlayerPtrCont*)pPPC)->push_back( (rcsc::PlayerObject *)pPO );
				}
			}
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00020000 )	// M_unknown_players
			{
				if( nIndex < M_unknown_players.size() )
				{
					PlayerCont::const_iterator itr= M_unknown_players.begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					((PlayerPtrCont*)pPPC)->push_back( (rcsc::PlayerObject *)pPO );
				}
			}
		}

		return true;
	};


/*
	structure of InterceptTable in data log file
	_____________________________
	|	[HEAD]					|
	|___________________________|
	|	Variables				|
	|___________________________|
	|	M_fastest_teammate		|	-> 0x0001000? / 0x0002000? / 0x0003000?
	|___________________________|
	|	M_fastest_opponent		|	-> 0x0001000? / 0x0002000? / 0x0003000?
	|___________________________|
	|	M_ball_pos_cache [H]	|
	|___________________________|
	|	M_self_cache [H]		|
	|___________________________|

	[H] or [HEAD] means that the data includes the structure of STEPDEBUG_HEAD
*/

	bool InterceptTable::ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		if( pHead->_dwRegCode != REGCODE_InterceptTable )
		{
			ASSERT(false);
			return false;
		}

		// read Variables
		{
			memcpy( (void*)this, pHead->_pData, ((long)&M_fastest_teammate - (long)this) );
			pCurrent = pCurrent + pHead->size() + ((long)&M_fastest_teammate - (long)this);
		}

		// read M_fastest_teammate
		{
			DWORD dbPlayerObjectPosition = *((DWORD*)pCurrent);
			DWORD nIndex = dbPlayerObjectPosition&0x0000FFFF;

			if( (dbPlayerObjectPosition&0xFFFF0000) == 0 )					// invalid
			{
				memset( (void *)&M_fastest_teammate, 0, sizeof(M_fastest_teammate) );
			}
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00010000 )	// M_teammates
			{
				if( nIndex < M_world.teammates().size() )
				{
					PlayerCont::const_iterator itr= M_world.teammates().begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					memcpy( (void*)&M_fastest_teammate, &pPO, sizeof(M_fastest_teammate) );
				}
			}
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00020000 )	// M_opponents
			{
				if( nIndex < M_world.opponents().size() )
				{
					PlayerCont::const_iterator itr= M_world.opponents().begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					memcpy( (void*)&M_fastest_teammate, &pPO, sizeof(M_fastest_teammate) );
				}
			}
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00020000 )	// M_unknown_players
			{
				if( nIndex < M_world.unknownPlayers().size() )
				{
					PlayerCont::const_iterator itr= M_world.unknownPlayers().begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					memcpy( (void*)&M_fastest_teammate, &pPO, sizeof(M_fastest_teammate) );
				}
			}
			pCurrent += sizeof(DWORD);
		}

		// read M_fastest_opponent
		{
			DWORD dbPlayerObjectPosition = *((DWORD*)pCurrent);
			DWORD nIndex = dbPlayerObjectPosition&0x0000FFFF;

			if( (dbPlayerObjectPosition&0xFFFF0000) == 0 )					// invalid
			{
				memset( (void *)&M_fastest_opponent, 0, sizeof(M_fastest_opponent) );
			}
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00010000 )	// M_teammates
			{
				if( nIndex < M_world.teammates().size() )
				{
					PlayerCont::const_iterator itr= M_world.teammates().begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					memcpy( (void*)&M_fastest_opponent, &pPO, sizeof(M_fastest_teammate) );
				}
			}
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00020000 )	// M_opponents
			{
				if( nIndex < M_world.opponents().size() )
				{
					PlayerCont::const_iterator itr= M_world.opponents().begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					memcpy( (void*)&M_fastest_opponent, &pPO, sizeof(M_fastest_teammate) );
				}
			}
			else if( (dbPlayerObjectPosition&0xFFFF0000) == 0x00020000 )	// M_unknown_players
			{
				if( nIndex < M_world.unknownPlayers().size() )
				{
					PlayerCont::const_iterator itr= M_world.unknownPlayers().begin();
					for( size_t j=0; j<nIndex; j++ ) itr++;

					const rcsc::PlayerObject * pPO = &(*itr);
					memcpy( (void*)&M_fastest_opponent, &pPO, sizeof(M_fastest_teammate) );
				}
			}
			pCurrent += sizeof(DWORD);
		}


		// read M_ball_pos_cache
		{
			if( !M_ball_pos_cache_ReadFromDataFile( pCurrent ) )
				return false;

			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
		}

		// read M_self_cache
		{
			if( !M_self_cache_ReadFromDataFile( pCurrent ) )
				return false;

			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
		}

		return true;

	}


	bool InterceptTable::M_ball_pos_cache_ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		if( pHead->_dwRegCode != REGCODE_InterceptTable1 )
		{
			ASSERT(false);
			return false;
		}

		((std::vector< Vector2D >*)(&M_ball_pos_cache))->clear();

		// read Vector2D
		size_t nCount = ( pHead->_dwSize - pHead->size() )/sizeof(Vector2D);
		for( size_t i=0; i<nCount; i++ )
		{
			((std::vector< Vector2D >*)(&M_ball_pos_cache))->push_back( *((Vector2D*)(pHead->_pData+i*sizeof(Vector2D))));
		}

		return true;
	}


	bool InterceptTable::M_self_cache_ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		if( pHead->_dwRegCode != REGCODE_InterceptTable2 )
		{
			ASSERT(false);
			return false;
		}

		((std::vector< InterceptInfo >*)(&M_self_cache))->clear();

		// read InterceptInfo
		size_t nCount = ( pHead->_dwSize - pHead->size() )/sizeof(InterceptInfo);
		for( size_t i=0; i<nCount; i++ )
		{
			((std::vector< InterceptInfo >*)(&M_self_cache))->push_back( *((InterceptInfo*)(pHead->_pData+i*sizeof(InterceptInfo))));
		}

		return true;
	}


	bool PenaltyKickState::ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		if( pHead->_dwRegCode != REGCODE_PenaltyKickState )
		{
			ASSERT(false);
			return false;
		}

		memcpy( (void*)this, pHead->_pData, sizeof(PenaltyKickState) );


		return true;
	}

	bool ActionEffector::ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		if( pHead->_dwRegCode != REGCODE_ActionEffector )
		{
			ASSERT(false);
			return false;
		}

		memcpy( (void*)this, pHead->_pData, ((long)&M_say_message - (long)this) );


		return true;
	}

	bool SelfObject::ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		pCurrent += pHead->size();
		if( pHead->_dwRegCode != REGCODE_SelfObject )
		{
			ASSERT(false);
			return false;
		}

		memcpy( (void*)this, pHead->_pData, ((long)&(playerType().dashDistanceTable()) - (long)this) );
		pCurrent += ((long)&(playerType().dashDistanceTable()) - (long)this);


		((std::vector< double > &)playerType().dashDistanceTable()).clear();
		int nCount = (pHead->_dwSize - (pCurrent - pData)) / sizeof( double );
		for( int i=0; i<nCount; i++ )
		{
			((std::vector< double > &)playerType().dashDistanceTable()).push_back( ((double*)pCurrent)[i] );
		}


		return true;
	}


	bool VisualSensor::ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		pCurrent += pHead->size();
		if( pHead->_dwRegCode != REGCODE_PAI_VisualSensor )
		{
			ASSERT(false);
			return false;
		}

		// M_time
		memcpy( (void*)&M_time, pHead->_pData, sizeof( M_time ) );
		pCurrent += sizeof( M_time );

		// container:	M_balls, M_markers, M_behind_markers, 
		//				M_lines, M_teammates, M_unknown_teammates,
		//				M_opponents, M_unknown_opponents, M_unknown_players
		{
			// M_balls
			Container_ReadFromDataFile<BallCont, BallT>
				crM_balls( pCurrent, (BallCont*)&M_balls, REGCODE_PAI_VisualSensor_BallCont );
			if( !crM_balls.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			// M_markers
			Container_ReadFromDataFile<MarkerCont, MarkerT>
				crM_markerss( pCurrent, (MarkerCont*)&M_markers, REGCODE_PAI_VisualSensor_MarkerCont );
			if( !crM_markerss.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			// M_behind_markers
			Container_ReadFromDataFile<MarkerCont, MarkerT>
				crM_behind_markerss( pCurrent, (MarkerCont*)&M_behind_markers, REGCODE_PAI_VisualSensor_MarkerCont );
			if( !crM_behind_markerss.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			// M_lines
			Container_ReadFromDataFile<LineCont, LineT>
				crM_lines( pCurrent, (LineCont*)&M_lines, REGCODE_PAI_VisualSensor_LineCont );
			if( !crM_lines.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			// M_teammates
			Container_ReadFromDataFile<PlayerCont, PlayerT>
				crM_teammates( pCurrent, (PlayerCont*)&M_teammates, REGCODE_PAI_VisualSensor_PlayerCont );
			if( !crM_teammates.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			// M_unknown_teammates
			Container_ReadFromDataFile<PlayerCont, PlayerT>
				crM_unknown_teammates( pCurrent, (PlayerCont*)&M_unknown_teammates, REGCODE_PAI_VisualSensor_PlayerCont );
			if( !crM_unknown_teammates.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			// M_opponents
			Container_ReadFromDataFile<PlayerCont, PlayerT>
				crM_opponents( pCurrent, (PlayerCont*)&M_opponents, REGCODE_PAI_VisualSensor_PlayerCont );
			if( !crM_opponents.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			// M_unknown_opponents
			Container_ReadFromDataFile<PlayerCont, PlayerT>
				crM_unknown_opponents( pCurrent, (PlayerCont*)&M_unknown_opponents, REGCODE_PAI_VisualSensor_PlayerCont );
			if( !crM_unknown_opponents.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;


			// M_unknown_players
			Container_ReadFromDataFile<PlayerCont, PlayerT>
				crM_unknown_players( pCurrent, (PlayerCont*)&M_unknown_players, REGCODE_PAI_VisualSensor_PlayerCont );
			if( !crM_unknown_players.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

		}


		// write M_opponent_team_name
		{
			(std::string&)M_opponent_team_name = pCurrent;

			pCurrent += 128;
		}
		return true;
	}


	bool AudioSensor::ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		pCurrent += pHead->size();
		if( pHead->_dwRegCode != REGCODE_PAI_AudioSensor )
		{
			ASSERT(false);
			return false;
		}

		// read variables
		{
			memcpy( (void *)this, pCurrent, ((long)&(M_player_message.message_) - (long)this) );
			pCurrent += ((long)&(M_player_message.message_) - (long)this);
		}

		// read M_player_message.message_
		{
			(std::string&)M_player_message.message_ = pCurrent;

			pCurrent += 512;
		}


		// read M_player_message.info_
		{
			if( !ReadFromDataFile_info(pCurrent) )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;
		}

		// read M_freeform_message
		{
			memcpy( (void *)&M_freeform_message.time_, pCurrent, sizeof(M_freeform_message.time_) );
			pCurrent += sizeof( M_freeform_message.time_ );

			(std::string&)M_freeform_message.message_ = pCurrent;
			pCurrent += 512;
		}

		// read M_trainer_message
		{
			memcpy( (void *)&M_trainer_message.time_, pCurrent, sizeof(M_trainer_message.time_) );
			pCurrent += sizeof( M_trainer_message.time_ );

			(std::string&)M_trainer_message.message_ = pCurrent;
			pCurrent += 512;
		}

		return true;
	}


	bool AudioSensor::ReadFromDataFile_info( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		pCurrent += pHead->size();
		if( pHead->_dwRegCode != REGCODE_PAI_AudioSensor_info )
		{
			ASSERT(false);
			return false;
		}

		// read data
		{
			((std::vector< PlayerMessage::Info >&)M_player_message.info_).clear();

			size_t nCount = ( pHead->_dwSize - pHead->size() ) / ( sizeof(PlayerMessage::Type) + 512 );

			for( size_t i=0; i<nCount; i++ )
			{
				PlayerMessage::Info info;
				memcpy( &info.type_, pCurrent + i * ( sizeof(PlayerMessage::Type) + 512 ), sizeof(PlayerMessage::Type) );
				info.str_ = pCurrent + i * ( sizeof(PlayerMessage::Type) + 512 ) + sizeof(PlayerMessage::Type);

				((std::vector< PlayerMessage::Info >&)M_player_message.info_).push_back( info );
			}

		}

		return true;
	}

	bool FullstateSensor::ReadFromDataFile( const char* pData ) const
	{
		const char* pCurrent = pData;

		// check head

		STEPDEBUG_HEAD*		pHead = (STEPDEBUG_HEAD*)pCurrent;
		pCurrent += pHead->size();
		if( pHead->_dwRegCode != REGCODE_PAI_FullstateSensor )
		{
			ASSERT(false);
			return false;
		}


		// read variables
		{
			memcpy( (void *)this, pCurrent, ((long)&(M_left_team) - (long)this) );
			pCurrent += ((long)&(M_left_team) - (long)this);
		}

		// read PlayerCont
		{
			// M_left_team
			Container_ReadFromDataFile<PlayerCont, PlayerT>
				crM_left_team( pCurrent, (PlayerCont*)&M_left_team, REGCODE_PAI_FullstateSensor_PlayerCont );
			if( !crM_left_team.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

			// M_right_team
			Container_ReadFromDataFile<PlayerCont, PlayerT>
				crM_right_team( pCurrent, (PlayerCont*)&M_right_team, REGCODE_PAI_FullstateSensor_PlayerCont );
			if( !crM_right_team.GetReturn() )
				return false;
			pCurrent += ((STEPDEBUG_HEAD*)pCurrent)->_dwSize;

		}

		return true;
	}

}