#include "win_src/StepDebug_Define.h"
#include <rcsc/player/penalty_kick_state.h>
#include <rcsc/player/action_effector.h>
#include <rcsc/player/self_object.h>
#include <rcsc/player/world_model.h>
#include <rcsc/player/intercept_table.h>
#include <rcsc/player/player_agent.h>
#include <rcsc/player/visual_sensor.h>
#include <rcsc/player/audio_sensor.h>
#include <rcsc/player/fullstate_sensor.h>


namespace rcsc {

template <class T> bool Container_Write2StepDebugFile(FILE *pStepDebugFile, const T* pPC, DWORD dwRegCode )    
{
	// check pointer
	if( pStepDebugFile == 0 )
		return false;
	if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
		return false;

	long lStart = ftell( pStepDebugFile );
	if( lStart == -1 )
		return false;

	// write head
	STEPDEBUG_HEAD	wm_head;
	{
		wm_head._dwRegCode		= dwRegCode;
		fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
	}

	// write PlayerObject
	{
		T::const_iterator	itr;
		for( itr = pPC->begin(); itr != pPC->end(); itr++ )
		{
			fwrite( &(*itr), sizeof( (*itr) ), 1, pStepDebugFile ); 
		}
	}

	// rewrite head with size
	{
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lEnd		= ftell( pStepDebugFile );
		if( lEnd == -1 )
			return false;

		wm_head._dwSize	= lEnd - lStart;
		if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
			return false;
		fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

		fseek(pStepDebugFile,0L,SEEK_END);
	}

	return true;
}    


	/*!
      \brief write current world model to a file
      \param file handle
      \return true - success, false - failed.
	*/

/*
	structure of WorldModel in data log file
	_____________________________
	|	[HEAD]					|
	|___________________________|
	|	Variables				|
	|___________________________|
	|	Static Variable			|
	|___________________________|
	|	PlayerCont	[H]			|
	|___________________________|
	|	PlayerPtrCont	[H]		|
	|___________________________|
	|	M_localize	[H]			|
	|___________________________|
	|	M_intercept_table	[H]	|
	|___________________________|
	|	M_penalty_kick_state [H]|
	|___________________________|
	|	SelfObject	[H]			|
	|___________________________|

	[H] or [HEAD] means that the data includes the structure of STEPDEBUG_HEAD
*/

	bool WorldModel::Write2StepDebugFile( FILE* pStepDebugFile ) const
	{
		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_WorldModel;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write variable and static
		{
			// The variales, defined before M_localize, are saved by memory
			fwrite( this, ((long)&M_localize - (long)this), 1, pStepDebugFile );

			fwrite( &Vector2D::ERROR_VALUE, sizeof( Vector2D::ERROR_VALUE ), 1, pStepDebugFile );
			fwrite( &Vector2D::INVALIDATED, sizeof( Vector2D::INVALIDATED ), 1, pStepDebugFile );

			fwrite( &AngleDeg::EPSILON, sizeof( AngleDeg::EPSILON ), 1, pStepDebugFile );
			fwrite( &AngleDeg::DEG2RAD, sizeof( AngleDeg::DEG2RAD ), 1, pStepDebugFile );
			fwrite( &AngleDeg::RAD2DEG, sizeof( AngleDeg::RAD2DEG ), 1, pStepDebugFile );

//			fwrite( &StaminaModel::DEFAULT_RECOVERY_MAX, sizeof( StaminaModel::DEFAULT_RECOVERY_MAX ), 1, pStepDebugFile );

			fwrite( &WorldModel::DIR_STEP, sizeof( WorldModel::DIR_STEP ), 1, pStepDebugFile );

//			fwrite( &BallObject::S_pos_count_thr, sizeof( BallObject::S_pos_count_thr ), 1, pStepDebugFile );
//			fwrite( &BallObject::S_rpos_count_thr, sizeof( BallObject::S_rpos_count_thr ), 1, pStepDebugFile );
//			fwrite( &BallObject::S_vel_count_thr, sizeof( BallObject::S_vel_count_thr ), 1, pStepDebugFile );
		}

		// write teamname
		{
			// TODO
		}

		// write PlayerCont
		{
			if( !PlayerCont_Write2StepDebugFile( pStepDebugFile, &M_teammates ))
				return false;
			if( !PlayerCont_Write2StepDebugFile( pStepDebugFile, &M_opponents ))
				return false;
			if( !PlayerCont_Write2StepDebugFile( pStepDebugFile, &M_unknown_players ))
				return false;
		}

		// write PlayerPtrCont
		{
			if( !PlayerPtrCont_Write2StepDebugFile( pStepDebugFile, &M_teammates_from_self ))
				return false;
			if( !PlayerPtrCont_Write2StepDebugFile( pStepDebugFile, &M_opponents_from_self ))
				return false;
			if( !PlayerPtrCont_Write2StepDebugFile( pStepDebugFile, &M_teammates_from_ball ))
				return false;
			if( !PlayerPtrCont_Write2StepDebugFile( pStepDebugFile, &M_opponents_from_ball ))
				return false;
		}

		// write object
		{

			if( !M_localize->Write2StepDebugFile( pStepDebugFile ) )
				return false;
			
			if( !M_intercept_table->Write2StepDebugFile( pStepDebugFile ) )
				return false;

			 if( !M_penalty_kick_state->Write2StepDebugFile( pStepDebugFile ) )
				 return false;
		}

		// write SelfObject
		{
			if( !M_self.Write2StepDebugFile( pStepDebugFile ) )
				return false;
		}


		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lEnd == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}


		fflush( pStepDebugFile );
		return true;
	}

	bool WorldModel::PlayerCont_Write2StepDebugFile(FILE *pStepDebugFile, const PlayerCont* pPC) const
	{
/*		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_WM_PlayerCont;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write PlayerObject
		{
			PlayerCont::const_iterator	itr;
			for( itr = pPC->begin(); itr != pPC->end(); itr++ )
			{
				fwrite( &(*itr), sizeof( (*itr) ), 1, pStepDebugFile ); 
			}
		}

		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lEnd == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
		*/
		return Container_Write2StepDebugFile( pStepDebugFile, pPC, REGCODE_WM_PlayerCont);
	}

	bool WorldModel::PlayerPtrCont_Write2StepDebugFile(FILE *pStepDebugFile, const rcsc::PlayerPtrCont *pPPC) const
	{
		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_WM_PlayerPtrCont;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write PlayerObject* as a dword
		{
			DWORD dbPlayerObjectPosition(0);
			PlayerPtrCont::const_iterator	itr;
			for( itr = pPPC->begin(); itr != pPPC->end(); itr++ )
			{
				// find the position and record it to file

				PlayerCont::const_iterator	itr2;
				
				// search in M_teammates
				dbPlayerObjectPosition = 0x00010000;
				for( itr2 = M_teammates.begin(); itr2 != M_teammates.end(); itr2++ )
				{
					if( (*itr) == &(*itr2) )
					{
						fwrite( &dbPlayerObjectPosition, sizeof(dbPlayerObjectPosition), 1, pStepDebugFile );
						break;
					}
					dbPlayerObjectPosition++;
				}
				if( itr2 != M_teammates.end() )
					continue;

				// search in M_opponents
				dbPlayerObjectPosition = 0x00020000;
				for( itr2 = M_opponents.begin(); itr2 != M_opponents.end(); itr2++ )
				{
					if( (*itr) == &(*itr2) )
					{
						fwrite( &dbPlayerObjectPosition, sizeof(dbPlayerObjectPosition), 1, pStepDebugFile );
						break;
					}
					dbPlayerObjectPosition++;
				}
				if( itr2 != M_opponents.end() )
					continue;

				// search in M_unknown_players
				dbPlayerObjectPosition = 0x00030000;
				for( itr2 = M_unknown_players.begin(); itr2 != M_unknown_players.end(); itr2++ )
				{
					if( (*itr) == &(*itr2) )
					{
						fwrite( &dbPlayerObjectPosition, sizeof(dbPlayerObjectPosition), 1, pStepDebugFile );
						break;
					}
					dbPlayerObjectPosition++;
				}
				if( itr2 == M_unknown_players.end() )
				{
					ASSERT(false);
					std::cerr<<" [Write2DataFile error] " << __FILE__ << ":" << __LINE__ << std::endl;
				}

			}
		}

		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lEnd == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
	}

	bool PenaltyKickState::Write2StepDebugFile( FILE* pStepDebugFile ) const
	{
		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_PenaltyKickState;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write data
		{

			fwrite( this, sizeof(PenaltyKickState), 1, pStepDebugFile );
			
		}

		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lStart == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
	}


	bool ActionEffector::Write2StepDebugFile( FILE* pStepDebugFile ) const
	{

		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_ActionEffector;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write data
		{

			fwrite( this, ((long)&M_say_message - (long)this), 1, pStepDebugFile );
			
		}

		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lStart == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
	}

	bool SelfObject::Write2StepDebugFile( FILE* pStepDebugFile ) const
	{

		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_SelfObject;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write data
		{

			fwrite( this, ((long)&(playerType().dashDistanceTable()) - (long)this), 1, pStepDebugFile );
			
			std::vector< double >::const_iterator	itr = playerType().dashDistanceTable().begin();
			for(;itr!=playerType().dashDistanceTable().end(); itr++)
			{
				fwrite( &(*itr), sizeof( double ), 1, pStepDebugFile );
			}
			
		}

		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lStart == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
	}

	bool VisualSensor::Write2StepDebugFile( FILE* pStepDebugFile ) const
	{

		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_PAI_VisualSensor;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write data
		{

			fwrite( &M_time, sizeof( M_time ), 1, pStepDebugFile );
		}

		// container:	M_balls, M_markers, M_behind_markers, 
		//				M_lines, M_teammates, M_unknown_teammates,
		//				M_opponents, M_unknown_opponents, M_unknown_players
		{
			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_balls, 
												REGCODE_PAI_VisualSensor_BallCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_markers, 
												REGCODE_PAI_VisualSensor_MarkerCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_behind_markers, 
												REGCODE_PAI_VisualSensor_MarkerCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_lines, 
												REGCODE_PAI_VisualSensor_LineCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_teammates, 
												REGCODE_PAI_VisualSensor_PlayerCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_unknown_teammates, 
												REGCODE_PAI_VisualSensor_PlayerCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_opponents, 
												REGCODE_PAI_VisualSensor_PlayerCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_unknown_opponents, 
												REGCODE_PAI_VisualSensor_PlayerCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_unknown_players, 
												REGCODE_PAI_VisualSensor_PlayerCont ) )
				return false;
		}

		// write M_opponent_team_name
		{
			char buff[128]={0};
			if( M_opponent_team_name.size() >= 128 )
				memcpy( buff, M_opponent_team_name.c_str(), 127 );
			else
				strcpy( buff, M_opponent_team_name.c_str() );

			fwrite( buff, 128, 1, pStepDebugFile );
		}

		// write M_marker_map, M_marker_map_old
		{

		}

		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lStart == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
	}

	bool AudioSensor::Write2StepDebugFile( FILE* pStepDebugFile ) const
	{

		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_PAI_AudioSensor;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write variables
		{
			fwrite( this, ((long)&(M_player_message.message_) - (long)this), 1, pStepDebugFile );
		}

		// write M_player_message.message_
		{
			char buff[512]={0};
			if( M_player_message.message_.size() >= 512 )
				memcpy( buff, M_player_message.message_.c_str(), 511 );
			else
				strcpy( buff, M_player_message.message_.c_str() );

			fwrite( buff, 512, 1, pStepDebugFile );
		}

		// write M_player_message.info_
		{
			if( !Write2StepDebugFile_info(pStepDebugFile) )
				return false;
		}

		// write M_freeform_message
		{
			fwrite( &M_freeform_message.time_, sizeof(M_freeform_message.time_), 1, pStepDebugFile );

			char buff[512]={0};
			if( M_freeform_message.message_.size() >= 512 )
				memcpy( buff, M_freeform_message.message_.c_str(), 511 );
			else
				strcpy( buff, M_freeform_message.message_.c_str() );

			fwrite( buff, 512, 1, pStepDebugFile );
		}

		// write M_trainer_message
		{
			fwrite( &M_trainer_message.time_, sizeof(M_trainer_message.time_), 1, pStepDebugFile );

			char buff[512]={0};
			if( M_trainer_message.message_.size() >= 512 )
				memcpy( buff, M_trainer_message.message_.c_str(), 511 );
			else
				strcpy( buff, M_trainer_message.message_.c_str() );

			fwrite( buff, 512, 1, pStepDebugFile );
		}



		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lStart == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
	}

	bool AudioSensor::Write2StepDebugFile_info( FILE* pStepDebugFile ) const
	{

		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_PAI_AudioSensor_info;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write data
		{
			for( size_t i=0; i<M_player_message.info_.size(); i++ )
			{
				
				fwrite( &M_player_message.info_[i].type_, sizeof(M_player_message.info_[i].type_), 1, pStepDebugFile );

				char buff[512]={0};
				if( M_player_message.info_[i].str_.size() >= 512 )
					memcpy( buff, M_player_message.info_[i].str_.c_str(), 511 );
				else
					strcpy( buff, M_player_message.info_[i].str_.c_str() );

				fwrite( buff, 512, 1, pStepDebugFile );
			}
		}


		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lStart == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
	}


	bool FullstateSensor::Write2StepDebugFile( FILE* pStepDebugFile ) const
	{

		// check pointer
		if( pStepDebugFile == 0 )
			return false;
		if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
			return false;

		long lStart = ftell( pStepDebugFile );
		if( lStart == -1 )
			return false;

		// write head
		STEPDEBUG_HEAD	wm_head;
		{
			wm_head._dwRegCode		= REGCODE_PAI_FullstateSensor;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );
		}

		// write variables
		{
			fwrite( this, ((long)&(M_left_team) - (long)this), 1, pStepDebugFile );
		}

		// write PlayerCont
		{
			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_left_team, 
												REGCODE_PAI_FullstateSensor_PlayerCont ) )
				return false;

			if( !Container_Write2StepDebugFile( pStepDebugFile, 
												&M_right_team, 
												REGCODE_PAI_FullstateSensor_PlayerCont ) )
				return false;

		}



		// rewrite head with size
		{
			if( fseek(pStepDebugFile,0L,SEEK_END) != 0 )
				return false;

			long lEnd		= ftell( pStepDebugFile );
			if( lStart == -1 )
				return false;

			wm_head._dwSize	= lEnd - lStart;
			if( fseek( pStepDebugFile, lStart, SEEK_SET ) != 0 )
				return false;
			fwrite( &wm_head, wm_head.size(), 1, pStepDebugFile );

			fseek(pStepDebugFile,0L,SEEK_END);
		}

		return true;
	}
}