// -*-c++-*-

/*!
  \file fedit_config.cpp
  \brief Formatin Editor configuration class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "fedit_config.h"

#include <rcsc/param/param_map.h>
#include <rcsc/param/cmd_line_parser.h>

#include <iostream>

const double FEditConfig::PITCH_LENGTH = 105.0;
const double FEditConfig::PITCH_WIDTH = 68.0;
const double FEditConfig::PITCH_MARGIN = 5.0;
const double FEditConfig::CENTER_CIRCLE_R = 9.15;
const double FEditConfig::PENALTY_AREA_LENGTH = 16.5;
const double FEditConfig::PENALTY_AREA_WIDTH = 40.32;
const double FEditConfig::PENALTY_CIRCLE_R = 9.15; // 7.1
const double FEditConfig::PENALTY_SPOT_DIST = 11.0;
const double FEditConfig::GOAL_AREA_LENGTH = 5.5;
const double FEditConfig::GOAL_AREA_WIDTH = 18.2;
const double FEditConfig::GOAL_DEPTH = 2.5;
const double FEditConfig::GOAL_WIDTH = 14.02;
const double FEditConfig::GOAL_POST_R = 0.06;

/*-------------------------------------------------------------------*/
/*!
  singleton interface
*/
FEditConfig &
FEditConfig::instance()
{
    static FEditConfig s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditConfig::FEditConfig()
{
    init();
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditConfig::~FEditConfig()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditConfig::init()
{
    M_maximize = false;
    M_full_screen = false;

    M_pos_x = -1;
    M_pos_y = -1;
    M_width = -1;
    M_height = -1;

    M_auto_backup = true;
    M_logging = false;
}

/*-------------------------------------------------------------------*/
/*!
  analyze command line options
*/
bool
FEditConfig::parseCmdLine( int argc,
                           char ** argv )
{
    rcsc::ParamMap system_options( "System Options" );
    rcsc::ParamMap editor_options( "Editor Options" );
    rcsc::ParamMap view_options( "View Preference Options" );

    bool help = false;
    bool version = false;
    system_options.add()
        ( "help", "h",
          rcsc::BoolSwitch( &help ),
          "print this message." )
        ( "version", "v",
          rcsc::BoolSwitch( &version ),
          "print version." )
        ;

    editor_options.add()
        ( "auto-backup", "",
          &M_auto_backup,
          "make backup files automatically" )
        ( "logging", "",
          rcsc::BoolSwitch( &M_logging ),
          "logging formation editor operation." )
        ;

    view_options.add()
        ( "maximize", "",
          rcsc::BoolSwitch( &M_maximize ),
          "start with a maximized frame." )
        ( "full-screen", "",
          rcsc::BoolSwitch( &M_full_screen ),
          "start with a full screen frame." )
        ( "pos-x", "",
          &M_pos_x,
          "set left x position of a main frame." )
        ( "pos-y", "",
          &M_pos_y,
          "set top y position of a main frame." )
        ( "width", "",
          &M_width,
          "set width of a main frame." )
        ( "height", "",
          &M_height,
          "set height of a main frame." )
        ;

    rcsc::CmdLineParser parser( argc, argv );

    parser.parse( system_options );
    parser.parse( editor_options );
    parser.parse( view_options );

    if ( help
         || parser.failed() )
    {
        std::cout << "Usage: " << "fedit"
                  << " [options ... ]"
                  << std::endl;
        system_options.printHelp( std::cout, false );
        editor_options.printHelp( std::cout );
        view_options.printHelp( std::cout );
        return false;
    }

    if ( version )
    {
        std::cout << "fedit" << " Version " << VERSION
                  << std::endl;
        return false;
    }

    return true;
}

#if 0
{
    namespace po = boost::program_options;

    po::options_description visibles( "Allowed options:" );

    visibles.add_options()
        ( "help,h",
          "generates this message." )
        ( "version,v",
          "print version." )
        ( "maximize",
          po::bool_switch( &M_maximize )->default_value( false ),
          "start with a maximized frame." )
        ( "full-screen",
          po::bool_switch( &M_full_screen )->default_value( false ),
          "start with a full screen frame." )
        ( "pos-x",
          po::value< long >( &M_pos_x )->default_value( -1, "" ),
          "set left x position of the main frame." )
        ( "pos-y",
          po::value< long >( &M_pos_y )->default_value( -1, "" ),
          "set top y position of the main frame." )
        ( "width",
          po::value< long >( &M_width )->default_value( -1, "" ),
          "set width of the main frame." )
        ( "height",
          po::value< long >( &M_height )->default_value( -1, "" ),
          "set height of the main frame." )
        ( "logging",
          po::bool_switch( &M_logging )->default_value( false, "off" ),
          "logging formation editor operation" )
        ;

    po::options_description all_desc( "All options:" );
    all_desc.add( visibles );

    bool help = false;
    try
    {
        po::variables_map vm;
        po::command_line_parser parser( argc, argv );
        parser.options( all_desc );
        po::store( parser.run(), vm );
        po::notify( vm );

        if ( vm.count( "help" ) )
        {
            help = true;
        }
        else if ( vm.count( "version" ) )
        {
            std::cout << PACKAGE_NAME << " Version " << VERSION
                      << std::endl;
            return false;
        }
    }
    catch ( std::exception & e )
    {
        std::cerr << e.what() << std::endl;
        help = true;
    }

    if ( help )
    {
        std::cout << "Usage: " << PACKAGE_NAME
                  << " [options ... ] [<GameLogFile>]\n";
        std::cout << visibles << std::endl;
        return false;
    }

    return true;
}
#endif
