// -*-c++-*-

/*!
  \file fedit_frame.cpp
  \brief formation editor frame class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/spinctrl.h>
#endif

#include "fedit_frame.h"

#include "id.h"
#include "fedit_config.h"
#include "fedit_canvas.h"
#include "fedit_data.h"
#include "fedit_dialog.h"

#include <rcsc/formation/formation_bpn.h>
#include <rcsc/formation/formation_dt.h>
#include <rcsc/formation/formation_ngnet.h>
#include <rcsc/formation/formation_rbf.h>
#include <rcsc/formation/formation_sbsp.h>

#include <iostream>

#include "xpm/chase.xpm"
#include "xpm/delete.xpm"
#include "xpm/hand.xpm"
#include "xpm/train.xpm"
#include "xpm/record.xpm"
#include "xpm/replace.xpm"
#include "xpm/save.xpm"

#if defined(__WXGTK__) || defined(__WXX11__) || defined(__WXMOTIF__) || defined(__WXMAC__)
#  include "xpm/soccerwindow2-nostr.xpm"
#endif

/*-------------------------------------------------------------------*/
/*!

*/
FEditFrame::FEditFrame()
    : wxFrame( static_cast< wxWindow * >( 0 ), -1,
               wxT( "Formation Editor" ),
               wxDefaultPosition,
               wxSize( 620, 480 ) )
    , M_editor_canvas( static_cast< FEditCanvas * >( 0 ) )
    , M_editor_dialog( static_cast< FEditDialog * >( 0 ) )
    , M_train_data_index( static_cast< wxSpinCtrl * >( 0 ) )
{
    // set minimal window size
    this->SetSizeHints( 340, 240 );
    this->SetSize( FEditConfig::instance().framePosX(),
                   FEditConfig::instance().framePosY(),
                   FEditConfig::instance().frameWidth(),
                   FEditConfig::instance().frameHeight() );

    createWindows();
    connectEvents();


    if ( FEditConfig::instance().maximize() )
    {
        this->Maximize( true );
    }

    if ( FEditConfig::instance().fullScreen() )
    {
        this->ShowFullScreen( true );
    }

    SetIcon( wxICON( soccerwindow2_nostr ) );
}

/*-------------------------------------------------------------------*/
/*!
  destructing main frame.
*/
FEditFrame::~FEditFrame()
{
    //std::cerr << "delete FEditFrame" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!
  creates dynamic event signal connect
*/
void
FEditFrame::connectEvents()
{
    // ---------------------------------------------------
    // register wx event hander
    // close frame
    Connect( wxID_ANY, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( FEditFrame::handleCloseEvent ) );

    // command event : menu
    // file
    Connect( FEDIT_NEW, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuNew ) );
    Connect( FEDIT_OPEN, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuOpen ) );
    Connect( FEDIT_SAVE_ALL, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuSaveAll ) );
    Connect( FEDIT_SAVE, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuSave ) );
    Connect( FEDIT_SAVE_AS, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuSaveAs ) );
    Connect( FEDIT_OPEN_TRAIN_DATA, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuOpenTrainData ) );
    Connect( FEDIT_SAVE_TRAIN_DATA, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuSaveTrainData ) );
    Connect( FEDIT_SAVE_TRAIN_DATA_AS, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuSaveTrainDataAs ) );
    Connect( FEDIT_QUIT, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuQuit ) );

    // spin ctrl
    //Connect( FEDIT_SPIN_TRAIN_DATA, wxEVT_COMMAND_TEXT_UPDATED,
    //wxTextEventHandler( FEditFrame::handleSpinTrainData ) );
    Connect( FEDIT_SPIN_TRAIN_DATA, wxEVT_COMMAND_SPINCTRL_UPDATED,
             wxSpinEventHandler( FEditFrame::handleSpinTrainData ) );

    // command
    Connect( FEDIT_RESET_BALL, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleResetBall ) );
    Connect( FEDIT_PLAYER_AUTO_MOVE, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handlePlayerAutoMove ) );

    Connect( FEDIT_SHOW_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FEditFrame::handleMenuShowDialog ) );

    // toolbar tool
    Connect( FEDIT_BALL_DRAGGABLE, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditFrame::handleBallDraggable ) );

    Connect( FEDIT_REPLACE_DATA, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditFrame::handleReplaceTrainData ) );
    Connect( FEDIT_DELETE_DATA, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditFrame::handleDeleteTrainData ) );

    Connect( FEDIT_TRAINING, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditFrame::handleTrain ) );
    Connect( FEDIT_RECORD_POSITION, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( FEditFrame::handleRecordPosition ) );


    ///////////////////////////////////////////////
    // update UI
    Connect( FEDIT_BALL_DRAGGABLE, wxEVT_UPDATE_UI,
             wxUpdateUIEventHandler( FEditFrame::handleUpdateBallDraggable ) );
    Connect( FEDIT_PLAYER_AUTO_MOVE, wxEVT_UPDATE_UI,
             wxUpdateUIEventHandler( FEditFrame::handleUpdatePlayerAutoMove ) );
}

/*-------------------------------------------------------------------*/
/*!
  \brief create all windows that has this mainframe as a parent
*/
void
FEditFrame::createWindows()
{
    // create attached tools.
    createMenuBar();
    createToolBar();
    createStatusBar();


    M_editor_canvas = new FEditCanvas( this );
    M_editor_canvas->SetSize( this->GetClientSize() );
}

/*-------------------------------------------------------------------*/
/*!
  create & realize menu & menu bar.
  called from constructor.
*/
void
FEditFrame::createMenuBar()
{
    // create menu
    wxMenuBar * menubar = new wxMenuBar();

    {
        wxMenu * menu = new wxMenu;
        menu->Append( FEDIT_NEW,
                      _( "New formation\tctrl-n" ),
                      _( "Create new formation data" ) );
        menu->AppendSeparator();
        menu->Append( FEDIT_OPEN,
                      _( "Open formation\tctrl-o" ),
                      _( "Open formation file" ) );
        menu->Append( FEDIT_OPEN_TRAIN_DATA,
                      _( "Open training data" ),
                      _( "Open training data file" ) );
        menu->AppendSeparator();
        menu->Append( FEDIT_SAVE_ALL,
                      _( "Save all\tctrl-s" ),
                      _( "Save formation and training data" ) );
        /*
        menu->Append( FEDIT_SAVE,
                      _( "Save formation" ),
                      _( "Save formation" ) );
        menu->Append( FEDIT_SAVE_TRAIN_DATA,
                      _( "Save trainig data" ),
                      _( "Save training data" ) );
        */
        menu->AppendSeparator();
        menu->Append( FEDIT_SAVE_AS,
                      _( "Save formation as..." ),
                      _( "Save data to the new file" ) );
        menu->Append( FEDIT_SAVE_TRAIN_DATA_AS,
                      _( "Save trainig data as..." ),
                      _( "Save training data to the new file" ) );
        menu->AppendSeparator();
        menu->Append( FEDIT_QUIT,
                      _( "&Quit\tctrl-q" ),
                      _( "Quit application" ) );
        menubar->Append( menu, _( "File" ) );
    }
#if 0
    {
        wxMenu * menu = new wxMenu;
        menu->Append( FEDIT_RESET_BALL,
                      _( "Ball Reset" ),
                      _( "Ball is dropped on the field center point." ) );
        menu->Append( FEDIT_BALL_DRAGGABLE,
                      _( "Enable Ball Drag" ),
                      _( "Enable to move the ball by mouse" ) );
        menu->Append( FEDIT_PLAYER_AUTO_MOVE,
                      _( "Player Move" ),
                      _( "Toggle player automatic move mode" ) );
        menubar->Append( menu, _( "Edit" ) );

    }
    {
        wxMenu * menu = new wxMenu;
        menu->Append( FEDIT_SHOW_DIALOG,
                      _( "Dialog"  ),
                      _( "Toggle control dialog" ) );

        menubar->Append( menu, _( "View" ) );
    }
#endif
    // append menubar to FEditFrame.
    this->SetMenuBar( menubar );
}

/*-------------------------------------------------------------------*/
/*!
  create & realize toolbar.
  called from constructor.
*/
void
FEditFrame::createToolBar()
{

    wxToolBar * tbar = this->CreateToolBar( wxTB_HORIZONTAL
                                            | wxTB_FLAT
                                            | wxTB_TEXT
                                            );
    ////////////////////////////////////////////////////////
    tbar->AddTool( FEDIT_SAVE_ALL,
                   _( "Save" ),
                   wxBitmap( save_xpm ),
                   _( "Save modiried formation and training data" ) );
    tbar->AddTool( FEDIT_BALL_DRAGGABLE,
                   _( "Ball Drag" ),
                   wxBitmap( hand_xpm ),
                   _( "Toggle ball draggable status" ),
                   wxITEM_CHECK );
    tbar->AddCheckTool( FEDIT_PLAYER_AUTO_MOVE,
                        _( "Auto Move" ),
                        wxBitmap( chase_xpm ),
                        wxNullBitmap,
                        _( "Toggle player automatic move mode" ) );
    tbar->AddSeparator();

    tbar->AddTool( FEDIT_REPLACE_DATA,
                   _(  "Replace" ),
                   wxBitmap( replace_xpm ),
                   _( "Replace the current indexed training data by the screen status" ) );
    tbar->AddTool( FEDIT_DELETE_DATA,
                   _(  "Delete" ),
                   wxBitmap( delete_xpm ),
                   _( "Delete the current indexed training data" ) );

    M_train_data_index = new wxSpinCtrl( tbar, FEDIT_SPIN_TRAIN_DATA,
                                         wxT( "0" ),
                                         wxDefaultPosition, wxDefaultSize,
                                         wxSP_ARROW_KEYS | wxSP_WRAP,
                                         0, 0, 0 );
    tbar->AddControl( M_train_data_index );

    tbar->AddSeparator();

    tbar->AddTool( FEDIT_TRAINING,
                   _( "Train" ),
                   wxBitmap( train_xpm ),
                   _( "Perform only train using recorded snapshot" ) );

    tbar->AddTool( FEDIT_RECORD_POSITION,
                   _( "Record" ),
                   wxBitmap( record_xpm ),
                   _( "Record current players' position as training data and Perform train" ) );
}

/*-------------------------------------------------------------------*/
/*!
  create & realize status bar.
  called from constructor.
*/
void
FEditFrame::createStatusBar()
{
    // already exist
    if ( this->GetStatusBar() )
    {
        return;
    }

    const int chw = this->GetCharWidth();
    const int widths[] = { -1, 20 * chw };
    this->CreateStatusBar( WXSIZEOF( widths ),
                           wxST_SIZEGRIP,
                           FEDIT_STATUSBAR );
    this->SetStatusText( wxT( "Ready" ), 0 );
    this->GetStatusBar()->SetStatusWidths( WXSIZEOF( widths ), widths );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::quit()
{
    if ( M_editor_data
         && M_editor_data->saveChanged() & wxCANCEL )
    {
        return;
    }

    if ( M_editor_data )
    {
        M_editor_data.reset();
    }

    this->Destroy();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    //std::cerr << "FEditFrame::handleCloseEvent()" << std::endl;
    quit();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuNew( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data
         && M_editor_data->saveChanged() & wxCANCEL )
    {
        return;
    }

    // select formation training type
    wxString choices[4] = { wxString( rcsc::FormationDT::name().c_str(), *wxConvCurrent ),
                            wxString( rcsc::FormationBPN::name().c_str(), *wxConvCurrent ),
                            wxString( rcsc::FormationRBF::name().c_str(), *wxConvCurrent ),
                            wxString( rcsc::FormationNGNet::name().c_str(), *wxConvCurrent ) //,
                            //wxString( rcsc::FormationSBSP::name().c_str(), *wxConvCurrent )
    };

    wxSingleChoiceDialog dlg( this,
                              wxT( "Choice a training method:" ),
                              wxT( "Create New Formation" ),
                              WXSIZEOF( choices ),
                              choices );
    if ( dlg.ShowModal() != wxID_OK )
    {
        return;
    }

#ifdef UNICODE
    std::string type_name =  (const char*)(dlg.GetStringSelection().mbc_str());
#else
    std::string type_name =  (const char*)(dlg.GetStringSelection().c_str());
#endif

    if ( M_editor_data )
    {
        M_editor_data.reset();
    }

    this->SetTitle( wxT( "Formation Editor - New Formation -" ) );

    // create new data

    M_editor_data = boost::shared_ptr< FEditData >( new FEditData( this ) );


    M_editor_data->setCanvas( M_editor_canvas );
    M_editor_data->setDialog( M_editor_dialog );

    M_editor_canvas->setData( M_editor_data );
    M_editor_dialog->setData( M_editor_data );

    M_editor_data->setTrainingTypeName( type_name );
    M_editor_data->createDefaultParam();

    if ( ! M_editor_data->formation() )
    {
        std::cerr << "***ERROR*** Failed to initialize formation data"
                  << std::endl;
        M_editor_data.reset();
        return;
    }

    M_editor_data->moveBallTo( 0.0, 0.0 );
    M_editor_dialog->update();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuOpen( wxCommandEvent & WXUNUSED( event ) )
{
    showOpenDialog();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuOpenTrainData( wxCommandEvent & WXUNUSED( event ) )
{
    showOpenTrainDataDialog();
}


/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuSaveAll( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->save();
        M_editor_data->saveTrainData();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuSave( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->save();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuSaveAs( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->saveAs();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuSaveTrainData( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->saveTrainData();
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuSaveTrainDataAs( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->saveTrainDataAs();
    }
}


/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuQuit( wxCommandEvent & WXUNUSED( event ) )
{
    //std::cerr << "FEditFrame::handleCloseEvent()" << std::endl;
    quit();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleResetBall( wxCommandEvent & event )
{
    if ( M_editor_data )
    {
        M_editor_data->moveBallTo( 0.0, 0.0 );
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handlePlayerAutoMove( wxCommandEvent & event )
{
    if ( M_editor_data )
    {
        M_editor_data->togglePlayerAutoMove();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::handleUpdatePlayerAutoMove( wxUpdateUIEvent & event )
{
    if ( M_editor_data )
    {
        event.Check( M_editor_data->isPlayerAutoMove() );
    }
    else
    {
        event.Check( false );
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FEditFrame::handleMenuShowDialog( wxCommandEvent & WXUNUSED( event ) )
{
    showEditDialog();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::showEditDialog()
{
    if ( M_editor_dialog )
    {
        M_editor_dialog->Show( ! M_editor_dialog->IsShown() );
    }
    else
    {
        M_editor_dialog = new FEditDialog( this );
        M_editor_dialog->Show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::handleBallDraggable( wxCommandEvent & event )
{
    if ( M_editor_data )
    {
        M_editor_data->toggleBallDraggable();
        M_editor_canvas->draw();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::handleUpdateBallDraggable( wxUpdateUIEvent & event )
{
    if ( M_editor_data )
    {
        event.Check( M_editor_data->isBallDraggable() );
    }
    else
    {
        event.Check( false );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::handleSpinTrainData( wxSpinEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->focusTrainData( M_train_data_index->GetValue() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::handleReplaceTrainData( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        if ( M_editor_data->replaceTrainData( M_train_data_index->GetValue() ) )
        {
            M_editor_data->train();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::handleDeleteTrainData( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        if ( M_editor_data->deleteTrainData( M_train_data_index->GetValue() ) )
        {
            M_editor_data->train();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::handleRecordPosition( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        if ( M_editor_data->recordCurrentPosition() )
        {
            M_editor_data->train();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::handleTrain( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_editor_data )
    {
        M_editor_data->train();
    }
}

/*-------------------------------------------------------------------*/
/*!
  \brief show file open dialog, and load view data from .rcg[.gz] file
*/
void
FEditFrame::showOpenDialog()
{
    if ( M_editor_data
         && M_editor_data->saveChanged() & wxCANCEL )
    {
        return;
    }

    wxString wild_card( wxT( "Formation file (*.conf)|*.conf"
                             "|All files (*.*)|*.*" ) );
    wxFileDialog file_dialog( this, // parent window
                              wxT( "Open" ), // dialog name shown on titlebar
                              wxT( "" ), // default dir
                              wxT( "" ), // default file
                              wild_card,
                              wxOPEN
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return;
    }
#ifdef UNICODE
    std::string file_path = (const char*)(file_dialog.GetPath().mbc_str());
#else
    std::string file_path = (const char*)(file_dialog.GetPath().c_str());
#endif

    M_editor_data = boost::shared_ptr< FEditData >( new FEditData( this ) );
    if ( ! M_editor_data->open( file_path ) )
    {
        M_editor_data.reset();
        return;
    }

    if ( ! M_editor_data->getFormation() )
    {
        M_editor_data->createDefaultParam();
    }

    M_editor_data->setCanvas( M_editor_canvas );
    M_editor_data->setDialog( M_editor_dialog );

    M_editor_canvas->setData( M_editor_data );
    M_editor_dialog->setData( M_editor_data );
    M_editor_dialog->update();

    M_editor_data->moveBallTo( 0.0, 0.0 );

    int answer = wxMessageBox( wxT( "Do you open a training data?" ),
                               wxT( "Option" ), // title
                               wxYES_NO,
                               this );
    if ( answer != wxYES )
    {
        return;
    }

    showOpenTrainDataDialog();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::showOpenTrainDataDialog()
{
    if ( ! M_editor_data )
    {
        return;
    }

    wxString wild_card( wxT( "Training data file (*.dat)|*.dat"
                             "|All files (*.*)|*.*" ) );
    wxFileDialog file_dialog( this, // parent window
                              wxT( "Open" ), // dialog name shown on titlebar
                              wxT( "" ), // default dir
                              wxT( "" ), // default file
                              wild_card,
                              wxOPEN
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return;
    }

#ifdef UNICODE
    std::string file_path = (const char*)(file_dialog.GetPath().mbc_str());
#else
    std::string file_path = (const char*)(file_dialog.GetPath().c_str());
#endif

    M_editor_data->openTrainData( file_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::setTrainDataIndexMax( const int max_idx )
{
    M_train_data_index->SetRange( 0, max_idx );
    M_train_data_index->SetValue( 0 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditFrame::updatePositionLabel( const double & x,
                                 const double & y )
{
    if ( GetStatusBar()
         && GetStatusBar()->IsShown() )
    {
        wxString pos_str;
        pos_str.Printf( wxT( "(%.2f %.2f)" ), x, y );
        GetStatusBar()->SetStatusText( pos_str, 1 );
    }
}
