// -*-c++-*-

/*!
  \file debug_log_data.h
  \brief debug log data Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCERWINDOW2_DEBUG_LOG_DATA_H
#define SOCCERWINDOW2_DEBUG_LOG_DATA_H

#include <string>
#include <list>
#include <map>

//! one cycle debug data snapshot
class DebugLogData {
public:
    enum Type {
        LEVEL_0 = 0x0000,
        LEVEL_1 = 0x0001,
        LEVEL_2 = 0x0002,
        LEVEL_3 = 0x0004,
        LEVEL_4 = 0x0008,
        LEVEL_5 = 0x0010,
        LEVEL_6 = 0x0020,
        LEVEL_7 = 0x0040,
        LEVEL_8 = 0x0080,
        LEVEL_9 = 0x0100,
        LEVEL_10 = 0x0200,
        LEVEL_11 = 0x0400,
        LEVEL_12 = 0x0800,
        LEVEL_13 = 0x1000,
        LEVEL_14 = 0x2000,
        LEVEL_15 = 0x4000,
        LEVEL_16 = 0x8000,
    };

    struct PointT {
        int level_;
        double x_;
        double y_;
    };
    struct LineT {
        int level_;
        double x1_;
        double y1_;
        double x2_;
        double y2_;
    };
    struct CircleT {
        int level_;
        double x_;
        double y_;
        double r_;
    };
    struct MessageT {
        int level_;
        double x_;
        double y_;
        std::string message_;
    };

    /**********************************************************
    Log Message Line Format
    Line := <Time> <Level> <Type> <Content>
    Time := integer value
    Level := integer value
    Type := {T|P|L|C|M}
        T : text
        P : point
        L: line
        C: circle
        M: message;
    Text := <Str>
    Point := <x:Real> <y:Real>
    Line := <x1:Real> <y1:Real> <x2:Real> <y2:Real>
    Circle := <x:Real> <y:Real> <r:Real>
    Message := <x:Real> <y:Real> <Str>
    **********************************************************/

    // pair.first is log level
    typedef std::list< std::pair< int, std::string > > TextList;

    typedef std::list< PointT > PointCont;
    typedef std::list< LineT > LineCont;
    typedef std::list< CircleT > CircleCont;
    typedef std::list< MessageT > MessageCont;
private:
    long M_cycle;

    //! normal message that is append to debug message box.
    TextList M_text_list;

    // key: level, value: data set for it.
    PointCont M_point_cont;
    LineCont M_line_cont;
    CircleCont M_circle_cont;
    MessageCont M_message_cont;

public:
    explicit
    DebugLogData( const long & cycle )
        : M_cycle( cycle )
      { }

    void setCycle( const long & cycle )
      {
          M_cycle = cycle;
      }

    const
    long & cycle() const
      {
          return M_cycle;
      }

    bool parse( const char * buf );

    const
    TextList & textList() const
      {
          return M_text_list;
      }
    const
    PointCont & pointCont() const
      {
          return M_point_cont;
      }
    const
    LineCont & lineCont() const
      {
          return M_line_cont;
      }
    const
    CircleCont & circleCont() const
      {
          return M_circle_cont;
      }
    const
    MessageCont & messageCont() const
      {
          return M_message_cont;
      }

private:
    bool addCircle( const int level,
                    const char * buf );
    bool addMessage( const int level,
                     const char * buf );
    bool addLine( const int level,
                  const char * buf );
    bool addPoint( const int level,
                   const char * buf );
};

#endif
