// -*-c++-*-

/*!
  \file main_window.cpp
  \brief main application window class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "main_window.h"

#include "image_save_dialog.h"
#include "detail_dialog.h"
#include "player_type_dialog.h"
#include "view_config_dialog.h"
#include "debug_message_window.h"
#include "field_canvas.h"
#include "debug_server.h"
#include "monitor_client.h"
#include "log_player.h"
#include "log_player_tool_bar.h"
#include "dir_selector.h"
#include "app_config.h"

#include <string>
#include <iostream>
#include <cstring>

#include <csignal> // ::kill()
#include <cstdlib> // system()
#include <sys/types.h> // SIGINT


#include "xpm/soccerwindow2.xpm"
#include "xpm/soccerwindow2-nostr.xpm"

#include "xpm/open_rcg.xpm"

#include "xpm/logplayer_live_mode.xpm"

/*-------------------------------------------------------------------*/
/*!

*/
MainWindow::MainWindow()
    : M_log_player( new LogPlayer( M_main_data, this ) )
    , M_detail_dialog( static_cast< DetailDialog * >( 0 ) )
    , M_player_type_dialog( static_cast< PlayerTypeDialog * >( 0 ) )
    , M_view_config_dialog( static_cast< ViewConfigDialog * >( 0 ) )
    , M_debug_message_window( static_cast< DebugMessageWindow * >( 0 ) )
    , M_monitor_client( static_cast< MonitorClient * >( 0 ) )
    , M_cursor_override( false )
{
    //    QPixmapCache::setCacheLimit( 1024 * 24 );

    loadSettings();

    createActions();
    createMenus();
    createToolBars();
    createStatusBar();
    // central widget
    createFieldCanvas();
    // control dialogs
    createViewConfigDialog();

    connect( M_log_player, SIGNAL( updated() ),
             this, SIGNAL( viewUpdated() ) );

    this->setWindowIcon( QIcon( QPixmap( soccerwindow2_xpm ) ) );
    this->setWindowTitle( tr( PACKAGE_NAME ) );

    this->setMinimumSize( 280, 220 );
    this->resize( AppConfig::instance().frameWidth() > 0
                  ? AppConfig::instance().frameWidth()
                  : 640,
                  AppConfig::instance().frameHeight() > 0
                  ? AppConfig::instance().frameHeight()
                  : 480 );
    this->move( AppConfig::instance().framePosX() > 0
                ? AppConfig::instance().framePosX() > 0
                : this->x(),
                AppConfig::instance().framePosY() > 0
                ? AppConfig::instance().framePosY()
                : this->y() );

    // this->setWindowOpacity( 0.5 ); // window transparency

    if ( AppConfig::instance().hideToolBar() )
    {
        M_log_player_tool_bar->hide();
    }

    if ( AppConfig::instance().hideStatusBar() )
    {
        this->statusBar()->hide();
    }

    if ( AppConfig::instance().maximize() )
    {
        this->showMaximized();
    }

    if ( AppConfig::instance().fullScreen() )
    {
        this->showFullScreen();
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
MainWindow::~MainWindow()
{
    std::cerr << "delete MainWindow" << std::endl;

    if ( AppConfig::instance().killServer() )
    {
        killServer();
    }
    saveSettings();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::init()
{
    if ( ! AppConfig::instance().gameLogFilePath().empty() )
    {
        loadRCG( QString::fromStdString( AppConfig::instance().gameLogFilePath() ) );
        return;
    }

    if ( AppConfig::instance().connect() )
    {
        std::string host = AppConfig::instance().host();
        if ( host.empty() )
        {
            host = "localhost";
        }

        connectMonitorTo( host.c_str() );

        if ( AppConfig::instance().debugServerMode() )
        {
            startDebugServer();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::loadSettings()
{
    QSettings settings( QDir::homePath() + "/.soccerwindow2-qt4",
                        QSettings::IniFormat );
    //QSettings settings( QSettings::IniFormat,
    //                    QSettings::UserScope,
    //                    PACKAGE_NAME );
    //QSettings settings( QSettings::NativeFormat,
    //                    QSettings::UserScope,
    //                    "rctools",
    //                    PACKAGE_NAME );


    settings.beginGroup( "AppConfig" );

    if ( AppConfig::instance().gameLogDir().empty() )
    {
        AppConfig::instance()
            .setGameLogDir( settings.value( "gameLogDir", "" )
                            .toString()
                            .toStdString() );
    }

    if ( AppConfig::instance().debugLogDir().empty() )
    {
        AppConfig::instance()
            .setDebugLogDir( settings.value( "debugLogDir", "" )
                             .toString()
                             .toStdString() );
    }

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveSettings()
{
    QSettings settings( QDir::homePath() + "/.soccerwindow2-qt4",
                        QSettings::IniFormat );
    //QSettings settings( QSettings::IniFormat,
    //                    QSettings::UserScope,
    //                    PACKAGE_NAME );
    //QSettings settings( QSettings::NativeFormat,
    //                    QSettings::UserScope,
    //                    "rctools",
    //                    PACKAGE_NAME );

    settings.beginGroup( "AppConfig" );

    settings.setValue( "gameLogDir", AppConfig::instance().gameLogDir().c_str() );
    settings.setValue( "debugLogDir", AppConfig::instance().debugLogDir().c_str() );

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActions()
{
    createActionsFile();
    createActionsMonitor();
    createActionsView();
    createActionsTool();
    createActionsHelp();

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsFile()
{
    M_open_rcg_act = new QAction( QIcon( QPixmap( open_rcg_xpm ) ),
                                  tr( "&Open rcg file..." ), this );
    M_open_rcg_act->setShortcut( tr( "Ctrl+O" ) );
    M_open_rcg_act->setStatusTip( tr( "Open RoboCup Game Log file" ) );
    connect( M_open_rcg_act, SIGNAL( triggered() ), this, SLOT( openRCG() ) );

    M_save_rcg_act = new QAction( tr( "Save rcg file as..." ), this );
    M_save_rcg_act->setStatusTip( tr( "Save RoboCup Game Log file" ) );
    connect( M_save_rcg_act, SIGNAL( triggered() ), this, SLOT( saveRCG() ) );

    M_open_debug_view_act = new QAction( tr( "Open debug view" ), this );
    M_open_debug_view_act
        ->setStatusTip( tr( "Open the directory where debug view logs exist" ) );
    connect( M_open_debug_view_act, SIGNAL( triggered() ),
             this, SLOT( openDebugView() ) );

    M_save_debug_view_act = new QAction( tr( "Save debug view" ), this );
    M_save_debug_view_act
        ->setStatusTip( tr( "Save debug view logs to the directory..." ) );
    connect( M_save_debug_view_act, SIGNAL( triggered() ),
             this, SLOT( saveDebugView() ) );

    M_exit_act = new QAction( tr( "&Quit" ), this );
    M_exit_act->setShortcut( tr( "Ctrl+Q" ) );
    M_exit_act->setStatusTip( tr( "Exit the application." ) );
    connect( M_exit_act, SIGNAL( triggered() ), this, SLOT( close() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsMonitor()
{
    M_kick_off_act = new QAction( tr( "&KickOff" ), this );
    M_kick_off_act->setShortcut( tr( "Ctrl+K" ) );
    M_kick_off_act->setStatusTip( tr( "Start the game" ) );
    M_kick_off_act->setEnabled( false );
    connect( M_kick_off_act, SIGNAL( triggered() ), this, SLOT( kickOff() ) );

    M_set_live_mode_act = new QAction( QIcon( QPixmap( logplayer_live_mode_xpm ) ),
                                       tr( "&Live Mode" ), this );
    M_set_live_mode_act->setShortcut( tr( "Ctrl+L" ) );
    M_set_live_mode_act->setStatusTip( tr( "set monitor to live mode" ) );
    M_set_live_mode_act->setEnabled( false );
    connect( M_set_live_mode_act, SIGNAL( triggered() ),
             this, SLOT( setLiveMode() ) );

    M_connect_monitor_act = new QAction( tr( "&Connect" ), this );
    M_connect_monitor_act->setShortcut( tr( "Ctrl+C" ) );
    M_connect_monitor_act
        ->setStatusTip( "Connect to the rcssserver on localhost" );
    M_connect_monitor_act->setEnabled( true );
    connect( M_connect_monitor_act, SIGNAL( triggered() ),
             this, SLOT( connectMonitor() ) );

    M_connect_monitor_to_act = new QAction( tr( "Connect &to ..." ), this );
    M_connect_monitor_to_act
        ->setStatusTip( tr( "Connect to the rcssserver on other host" ) );
    M_connect_monitor_to_act->setEnabled( true );
    connect( M_connect_monitor_to_act, SIGNAL( triggered() ),
             this, SLOT( connectMonitorTo() ) );

    M_disconnect_monitor_act = new QAction( tr( "&Disconnect" ), this );
    M_disconnect_monitor_act->setStatusTip( tr( "Disonnect from rcssserver" ) );
    M_disconnect_monitor_act->setEnabled( false );
    connect( M_disconnect_monitor_act, SIGNAL( triggered() ),
             this, SLOT( disconnectMonitor() ) );

    M_kill_server_act = new QAction( tr( "&Kill server" ), this );
    M_kill_server_act->setStatusTip( tr( "Kill the rcssserver process" ) );
    M_kill_server_act->setEnabled( false );
    connect( M_kill_server_act, SIGNAL( triggered() ),
             this, SLOT( killServer() ) );

    M_restart_server_act = new QAction( tr( "(Re)&start server" ), this );
    M_restart_server_act->setStatusTip( tr( "(Re)start rcssserver" ) );
    connect( M_restart_server_act, SIGNAL( triggered() ),
             this, SLOT( restartServer() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsView()
{
    M_toggle_tool_bar_act = new QAction( tr( "&Tool Bar" ), this );
    M_toggle_tool_bar_act->setStatusTip( tr( "Show/Hide Tool Bar" ) );
    connect( M_toggle_tool_bar_act, SIGNAL( triggered() ),
             this, SLOT( toggleToolBar() ) );

    M_toggle_status_bar_act = new QAction( tr( "&Status Bar" ), this );
    M_toggle_status_bar_act->setStatusTip( tr( "Show/Hide Status Bar" ) );
    connect( M_toggle_status_bar_act, SIGNAL( triggered() ),
             this, SLOT( toggleStatusBar() ) );

    M_full_screen_act = new QAction( tr( "&Full Screen" ), this );
    M_full_screen_act->setShortcut( tr( "Alt+Enter" ) );
    //M_full_screen_act->setShortcut( tr( "Alt+Enter" ) );
    //M_full_screen_act->setShortcut( Qt::ALT + Qt::Key_Return );
    //M_full_screen_act->setShortcut( Qt::ALT + Qt::Key_Enter );
    M_full_screen_act->setStatusTip( tr( "Toggle Full Screen" ) );
    connect( M_full_screen_act, SIGNAL( triggered() ),
             this, SLOT( toggleFullScreen() ) );
    //(void) new QShortcut( Qt::ALT + Qt::Key_Enter, this, SLOT( toggleFullScreen() ) );
    (void) new QShortcut( Qt::ALT + Qt::Key_Return,
                          this, SLOT( toggleFullScreen() ) );

    M_show_player_type_dialog_act = new QAction( tr( "&Player Type List" ), this );
    M_show_player_type_dialog_act->setShortcut( tr( "Ctrl+T" ) );
    M_show_player_type_dialog_act
        ->setStatusTip( tr( "Show player type parameters dialog" ) );
    connect( M_show_player_type_dialog_act, SIGNAL( triggered() ),
             this, SLOT( showPlayerTypeDialog() ) );

    M_show_detail_dialog_act = new QAction( tr( "&Object Detail" ), this );
    M_show_detail_dialog_act->setShortcut( tr( "Ctrl+I" ) );
    M_show_detail_dialog_act
        ->setStatusTip( tr( "Show detail information dialog" ) );
    connect( M_show_detail_dialog_act, SIGNAL( triggered() ),
             this, SLOT( showDetailDialog() ) );

    // qt style menu group
    M_style_act_group = new QActionGroup( this );
    Q_FOREACH ( QString style_name, QStyleFactory::keys() )
        {
            QAction * subaction = new QAction( M_style_act_group );
            subaction->setText( style_name );
            subaction->setData( style_name );
            subaction->setCheckable( true );
            if ( style_name.toLower()
                 == QApplication::style()->objectName().toLower() )
            {
                subaction->setChecked( true );
            }
            connect( subaction, SIGNAL( triggered( bool ) ),
                     this, SLOT( changeStyle( bool ) ) );
        }

    M_show_view_config_dialog_act = new QAction( tr( "&View Preference" ), this );
    M_show_view_config_dialog_act->setShortcut( tr( "Ctrl+V" ) );
    M_show_view_config_dialog_act
        ->setStatusTip( tr( "Show view preference dialog" ) );
    connect( M_show_view_config_dialog_act, SIGNAL( triggered() ),
             this, SLOT( showViewConfigDialog() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsTool()
{
    M_show_debug_message_window_act = new QAction( tr( "Debug &Message" ), this );
    M_show_debug_message_window_act->setShortcut( tr( "Ctrl+D" ) );
    M_show_debug_message_window_act
        ->setStatusTip( tr( "Show debug message window" ) );
    connect( M_show_debug_message_window_act, SIGNAL( triggered() ),
             this, SLOT( showDebugMessageWindow() ) );

    M_start_debug_server_act = new QAction( tr( "Start Debug Server" ), this );
    M_start_debug_server_act
        ->setStatusTip( tr( "Start to handle debug client messages." ) );
    M_start_debug_server_act->setEnabled( false );
    connect( M_start_debug_server_act, SIGNAL( triggered() ),
             this, SLOT( startDebugServer() ) );

    M_stop_debug_server_act = new QAction( tr( "Stop Debug Server" ), this );
    M_stop_debug_server_act
        ->setStatusTip( tr( "Disconnect debug connections and stop the debug server." ) );
    M_stop_debug_server_act->setEnabled( false );
    connect( M_stop_debug_server_act, SIGNAL( triggered() ),
             this, SLOT( stopDebugServer() ) );

    M_show_image_save_dialog_act = new QAction( tr( "Save &Image" ), this );
    M_show_image_save_dialog_act
        ->setStatusTip( tr( "Save game log data as image files" ) );
    connect( M_show_image_save_dialog_act, SIGNAL( triggered() ),
             this, SLOT( showImageSaveDialog() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsHelp()
{
    M_about_act = new QAction( QIcon( QPixmap( soccerwindow2_nostr_xpm ) ),
                               tr( "&About" ), this );
    M_about_act->setStatusTip( tr( "Show the about dialog." ) );
    connect( M_about_act, SIGNAL( triggered() ), this, SLOT( about() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenus()
{
    createMenuFile();
    createMenuMonitor();
    createMenuView();
    createMenuTool();
    createMenuHelp();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuFile()
{
    QMenu * menu = menuBar()->addMenu( tr( "&File" ) );
    menu->addAction( M_open_rcg_act );
    menu->addAction( M_save_rcg_act );
    menu->addSeparator();
    menu->addAction( M_open_debug_view_act );
    menu->addAction( M_save_debug_view_act );
    menu->addSeparator();
    menu->addAction( M_exit_act );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuMonitor()
{
    QMenu * menu = menuBar()->addMenu( tr( "&Monitor" ) );
    menu->addAction( M_kick_off_act );
    menu->addAction( M_set_live_mode_act );
    menu->addSeparator();
    menu->addAction( M_connect_monitor_act );
    menu->addAction( M_connect_monitor_to_act );
    menu->addAction( M_disconnect_monitor_act );
    menu->addSeparator();
    menu->addAction( M_kill_server_act );
    menu->addAction( M_restart_server_act );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuView()
{
    QMenu * menu = menuBar()->addMenu( tr( "&View" ) );
    menu->addAction( M_toggle_tool_bar_act );
    menu->addAction( M_toggle_status_bar_act );
    menu->addSeparator();
    menu->addAction( M_full_screen_act );
    menu->addSeparator();
    menu->addAction( M_show_player_type_dialog_act );
    menu->addAction( M_show_detail_dialog_act );
    menu->addSeparator();
    {
        QMenu * submenu = menu->addMenu( tr( "Qt &Style" ) );
        Q_FOREACH ( QAction * action, M_style_act_group->actions() )
            {
                submenu->addAction( action );
            }
    }
    menu->addAction( M_show_view_config_dialog_act );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuTool()
{
    QMenu * menu = menuBar()->addMenu( tr( "&Tool" ) );
    menu->addAction( M_show_debug_message_window_act );
    menu->addSeparator();
    menu->addAction( M_start_debug_server_act );
    menu->addAction( M_stop_debug_server_act );
    menu->addSeparator();
    menu->addAction( M_show_image_save_dialog_act );

}
/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuHelp()
{
    QMenu * menu = menuBar()->addMenu( tr( "&Help" ) );
    menu->addAction( M_about_act );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createToolBars()
{
    M_log_player_tool_bar = new LogPlayerToolBar( M_log_player,
                                                  M_main_data );

    M_log_player_tool_bar->addSeparator();

    M_log_player_tool_bar->addAction( M_set_live_mode_act );


    connect( this,  SIGNAL( viewUpdated() ),
             M_log_player_tool_bar, SLOT( updateSlider() ) );


    this->addToolBar( M_log_player_tool_bar );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createStatusBar()
{
    this->statusBar()->showMessage( tr( "Ready" ) );

    M_position_label = new QLabel( tr( "(0.0, 0.0)" ) );

    int min_width
        = M_position_label->fontMetrics().width(  tr( "(-60.0, -30.0)" ) )
        + 16;
    M_position_label->setMinimumWidth( min_width );
    M_position_label->setAlignment( Qt::AlignRight );

    this->statusBar()->addPermanentWidget( M_position_label );

    //QSlider * slider = new QSlider( Qt::Horizontal );
    //this->statusBar()->addWidget( slider );

    //QLineEdit * edit = new QLineEdit( tr( "0" ) );
    //edit->setMaximumSize( 36, 20 );
    //this->statusBar()->addWidget( edit );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createFieldCanvas()
{
    M_field_canvas = new FieldCanvas( M_main_data );
    //M_field_canvas->setParent( this );
    this->setCentralWidget( M_field_canvas );

    M_field_canvas->setFocus();

    connect( this, SIGNAL( viewUpdated() ),
             M_field_canvas, SLOT( update() ) );
             //M_field_canvas, SLOT( repaint() ) );

    connect( M_field_canvas, SIGNAL( mouseMoved( const QPoint & ) ),
             this, SLOT( updatePositionLabel( const QPoint & ) ) );

    connect( M_field_canvas, SIGNAL( dropBall( const QPoint & ) ),
             this, SLOT( dropBall( const QPoint & ) ) );
    connect( M_field_canvas, SIGNAL( freeKickLeft( const QPoint & ) ),
             this, SLOT( freeKickLeft( const QPoint & ) ) );
    connect( M_field_canvas, SIGNAL( freeKickRight( const QPoint & ) ),
             this, SLOT( freeKickRight( const QPoint & ) ) );


    // create & set context menus
    {
        QMenu * menu = new QMenu( M_field_canvas );
        menu->addAction( M_open_rcg_act );
        menu->addAction( M_connect_monitor_act );
        menu->addAction( M_restart_server_act );

        M_field_canvas->setNormalMenu( menu );
    }
    {
        QMenu * menu = new QMenu( M_field_canvas );
        menu->addAction( M_open_rcg_act );
        menu->addAction( M_connect_monitor_act );
        menu->addSeparator();
        menu->addAction( M_kill_server_act );
        menu->addAction( M_restart_server_act );

        M_field_canvas->setSystemMenu( menu );
    }
    {
        QMenu * menu = new QMenu( M_field_canvas );
        menu->addAction( M_kick_off_act );
        menu->addSeparator();
        menu->addAction( tr( "Drop Ball" ),
                         M_field_canvas, SLOT( dropBall() ) );
        menu->addAction( tr( "Free Kick Left" ),
                         M_field_canvas, SLOT( freeKickLeft() ) );
        menu->addAction( tr( "Free Kick Right" ),
                         M_field_canvas, SLOT( freeKickRight() ) );
        menu->addSeparator();

        QAction * drop_ball_there
            = new QAction( tr( "Drop Ball There" ), this );
        drop_ball_there->setStatusTip( tr( "Drop ball at the current ball position." ) );
        connect( drop_ball_there, SIGNAL( triggered() ),
                 this, SLOT( dropBallThere() ) );
        menu->addAction( drop_ball_there );

        M_field_canvas->setMonitorMenu( menu );
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createViewConfigDialog()
{
    if ( M_view_config_dialog )
    {
        return;
    }

    M_view_config_dialog
        = new ViewConfigDialog( this,
                                M_main_data,
                                M_main_data.getViewConfig() );

    //M_view_config_dialog->setParent( this, Qt::Dialog );
    M_view_config_dialog->hide();

    connect( M_view_config_dialog, SIGNAL( configured() ),
             this, SIGNAL( viewUpdated() ) );

    connect( M_view_config_dialog, SIGNAL( canvasResized( const QSize & ) ),
             this, SLOT( resizeCanvas( const QSize & ) ) );

    connect( M_field_canvas, SIGNAL( focusChanged( const QPoint & ) ),
             M_view_config_dialog, SLOT( setFocusPoint( const QPoint & ) ) );

    connect( M_view_config_dialog, SIGNAL( configured() ),
             M_field_canvas, SLOT( setRedrawAllFlag()  ) );

    // register short cut keys
    {
        // z
        QAction * act = new QAction( tr( "ZoomIn" ), this );
        act->setShortcut( Qt::Key_Z );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( zoomIn() ) );
    }
    {
        // x
        QAction * act = new QAction( tr( "ZoomOut" ), this );
        act->setShortcut( Qt::Key_X );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( zoomOut() ) );
    }
    {
        // Ctrl + z
        QAction * act = new QAction( tr( "ZoomOut" ), this );
        act->setShortcut( Qt::CTRL + Qt::Key_Z );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( zoomOut() ) );
    }
    {
        // i
        QAction * act = new QAction( tr( "Fit" ), this );
        act->setShortcut( Qt::Key_I );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( unzoom() ) );
    }
    {
        // e
        QAction * act = new QAction( tr( "Toggle Enlarge" ), this );
        act->setShortcut( Qt::Key_E );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleEnlarge() ) );
    }

    {
        // k
        QAction * act = new QAction( tr( "Toggle Keepaway" ), this );
        act->setShortcut( Qt::Key_K );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleKeepawayMode() ) );
    }

    // player detail
    {
        // n
        QAction * act = new QAction( tr( "Show Player Number" ), this );
        act->setShortcut( Qt::Key_N );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowPlayerNumber() ) );
    }
    {
        // h
        QAction * act = new QAction( tr( "Show Hetero Number" ), this );
        act->setShortcut( Qt::Key_H );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowHeteroNumber() ) );
    }
    {
        // s
        QAction * act = new QAction( tr( "Show Staminar" ), this );
        act->setShortcut( Qt::Key_S );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowStamina() ) );
    }
    {
        // v
        QAction * act = new QAction( tr( "Show View Cone" ), this );
        act->setShortcut( Qt::Key_V );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowViewCone() ) );
    }
    {
        // c
        QAction * act = new QAction( tr( "Show Control Area" ), this );
        act->setShortcut( Qt::Key_C );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowControlArea() ) );
    }

    // show/hide
    {
        // t
        QAction * act = new QAction( tr( "Show Score Board" ), this );
        act->setShortcut( Qt::Key_T );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowScoreBoard() ) );
    }
    {
        // Ctrl + b
        QAction * act = new QAction( tr( "Show Ball" ), this );
        act->setShortcut( Qt::CTRL + Qt::Key_B );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowBall() ) );
    }
    {
        // Ctrl + p
        QAction * act = new QAction( tr( "Show Players" ), this );
        act->setShortcut( Qt::CTRL + Qt::Key_P );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowPlayers() ) );
    }
    {
        // f
        QAction * act = new QAction( tr( "Show Flags" ), this );
        act->setShortcut( Qt::Key_F );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowFlags() ) );
    }
    {
        // o
        QAction * act = new QAction( tr( "Show Offside Line" ), this );
        act->setShortcut( Qt::Key_O );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                 M_view_config_dialog, SLOT( toggleShowOffsideLine() ) );
    }

    // number 1-10
    for ( int i = 0; i < 10; ++i )
    {
        {
            QAction * act = new QAction( QString( "Selct Left %1" ).arg( i ), this );
            act->setShortcut( Qt::Key_0 + i );
            this->addAction( act );
            connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( selectPlayerWithKey() ) );
        }
        {
            QAction * act = new QAction( QString( "Selct Right %1" ).arg( i ), this );
            act->setShortcut( Qt::CTRL + Qt::Key_0 + i );
            this->addAction( act );
            connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( selectPlayerWithKey() ) );
        }
    }
    // number 11
    {
        {
            QAction * act = new QAction( tr( "Selct Left 11" ), this );
            act->setShortcut( Qt::Key_Minus );
            this->addAction( act );
            connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( selectPlayerWithKey() ) );
        }
        {
            QAction * act = new QAction( tr( "Selct Right 11" ), this );
            act->setShortcut( Qt::CTRL + Qt::Key_Minus );
            this->addAction( act );
            connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( selectPlayerWithKey() ) );
        }
    }
    // b
    {
        QAction * act = new QAction( tr( "Focus Ball" ), this );
        act->setShortcut( Qt::Key_B );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( toggleFocusBall() ) );
    }
    // p
    {
        QAction * act = new QAction( tr( "Focus Player" ), this );
        act->setShortcut( Qt::Key_P );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( toggleFocusPlayer() ) );
    }
    // a
    {
        QAction * act = new QAction( tr( "Select auto all" ), this );
        act->setShortcut( Qt::Key_A );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( toggleSelectAutoAll() ) );
    }
    // l
    {
        QAction * act = new QAction( tr( "Select auto left" ), this );
        act->setShortcut( Qt::Key_L );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( toggleSelectAutoLeft() ) );
    }
    // r
    {
        QAction * act = new QAction( tr( "Select auto right" ), this );
        act->setShortcut( Qt::Key_R );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( toggleSelectAutoRight() ) );
    }
    // u
    {
        QAction * act = new QAction( tr( "Unselect" ), this );
        act->setShortcut( Qt::Key_U );
        this->addAction( act );
        connect( act, SIGNAL( triggered() ),
                     M_view_config_dialog, SLOT( setUnselect() ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::closeEvent( QCloseEvent * event )
{
    event->ignore();

    //QCoreApplication::instance()->quit();
    qApp->quit();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::resizeEvent( QResizeEvent * event )
{
    event->accept();

    if ( M_view_config_dialog
         && M_view_config_dialog->isVisible() )
    {
        M_view_config_dialog->updateFieldScale();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::wheelEvent( QWheelEvent * event )
{
    if ( event->delta() < 0 )
    {
        M_log_player->stepForward();
    }
    else
    {
        M_log_player->stepBack();
    }

    event->accept();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::openRCG()
{
    std::cerr << "MainWindow::openRCG()" << std::endl;

#ifdef HAVE_LIBRCSC_GZ
    QString filter( tr( "Game Log files (*.rcg *.rcg.gz);;"
                        "All files (*)" ) );
#else
    QString filter( tr( "Game Log files (*.rcg);;"
                        "All files (*)" ) );
#endif

    QString default_dir
        = QString::fromStdString( AppConfig::instance().gameLogDir() );

    QString file_path = QFileDialog::getOpenFileName( this,
                                                      tr( "Choose a game log file to open" ),
                                                      default_dir,
                                                      filter );

    if ( file_path.isEmpty() )
    {
        std::cerr << "MainWindow::opneRCG() empty file path" << std::endl;
        return;
    }

    std::cerr << "selected file = [" << file_path.toStdString() << std::endl;

    loadRCG( file_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::loadRCG( const QString & file_path )
{
    disconnectMonitor();
    M_log_player->stop();

    if ( ! M_main_data.loadRCG( file_path.toStdString() ) )
    {
        QString err_msg = tr( "Failed to read [" );
        err_msg += file_path;
        err_msg += tr( "]" );
        QMessageBox::critical( this,
                               tr( "Error" ),
                               err_msg,
                               QMessageBox::Ok, QMessageBox::NoButton );
        return;
    }

    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        QString err_msg = tr( "Empty log file [" );
        err_msg += file_path;
        err_msg += tr( "]" );
        QMessageBox::critical( this,
                               tr( "Error" ),
                               err_msg,
                               QMessageBox::Ok, QMessageBox::NoButton );
        return;
    }

    QFileInfo file_info( file_path );

    AppConfig::instance().setGameLogDir( file_info.absolutePath().toStdString() );

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->updateData();
    }

    if ( M_debug_message_window )
    {
        M_debug_message_window->clearAll();
    }

    if ( M_view_config_dialog )
    {
        M_view_config_dialog->unzoom();
    }

    emit viewUpdated();

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveRCG()
{
    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        QMessageBox::warning( this,
                              tr( "Error" ),
                              tr( "No Monitor View Data!" ) );
        return;
    }

    QString default_file_name;
    {
        const MonitorViewConstPtr latest = M_main_data.viewHolder().latestViewData();

        if ( latest )
        {
            default_file_name
                = QDateTime::currentDateTime().toString( "yyyyMMddhhmm-" );

            QString left_team = QString::fromStdString( latest->leftTeam().name() );
            QString left_score = QString::number( latest->leftTeam().score() );

            QString right_team = QString::fromStdString( latest->rightTeam().name() );
            QString right_score = QString::number( latest->rightTeam().score() );

            default_file_name += left_team;
            default_file_name += tr( "_" );
            default_file_name += left_score;
            default_file_name += tr( "-" );
            default_file_name += right_team;
            default_file_name += tr( "_" );
            default_file_name += right_score;

            default_file_name += tr( ".rcg" );
        }
    }

#ifdef HAVE_LIBRCSC_GZ
    QString filter( tr( "Game Log files (*.rcg *.rcg.gz);;"
                        "All files (*)" ) );
#else
    QString filter( tr( "Game Log files (*.rcg);;"
                        "All files (*)" ) );
#endif

    QString default_dir
        = QString::fromStdString( AppConfig::instance().gameLogDir() );
    if ( ! default_file_name.isEmpty() )
    {
        default_dir += tr( "/" );
        default_dir += default_file_name;
    }

    QString file_path = QFileDialog::getSaveFileName( this,
                                                      tr( "Save a game log file as" ),
                                                      default_dir,
                                                      filter );

    if ( file_path.isEmpty() )
    {
        std::cerr << "MainWindow::saveRCG() empty file path" << std::endl;
        return;
    }

    std::string file_path_std = file_path.toStdString();

    std::cerr << "save game log data to the file = [" << file_path_std
              << std::endl;

    // update game log dir
    QFileInfo file_info( file_path );
    AppConfig::instance().setGameLogDir( file_info.absolutePath().toStdString() );

    // check gzip usability
    bool is_gzip = false;
    if ( file_path_std.length() > 3
         && file_path_std.compare( file_path_std.length() - 3, 3, ".gz" ) == 0 )
    {
#ifdef HAVE_LIBRCSC_GZ
        if ( file_path_std.length() <= 7
             || file_path_std.compare( file_path_std.length() - 4, 4, ".rcg.gz" ) != 0 )
        {
            file_path_std == ".rcg.gz";
        }
        is_gzip = true;
#else
        // erase '.gz'
        file_path_std.erase( file_path_std.length() - 3 );
#endif
    }

    // check the extention string
    if ( ! is_gzip )
    {
        if ( file_path_std.length() <= 4
             || file_path_std.compare( file_path_std.length() - 4, 4, ".rcg" ) != 0 )
        {
            file_path_std += ".rcg";
        }
    }

    M_main_data.saveRCG( file_path_std );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::openDebugView()
{
    std::cerr << "MainWindow::openDebugView()" << std::endl;

    QString default_dir
        = QString::fromStdString( AppConfig::instance().debugLogDir() );
#if 0
    QString dir_path
        = QFileDialog::getExistingDirectory( this,
                                             tr( "Choose a debug view log directory" ),
                                             default_dir );
#endif
    DirSelector selector( this,
                          tr( "Choose a debug view log directory" ),
                          default_dir );

    if ( ! selector.exec() )
    {
        std::cerr << "MainWindow::openDebugView() canceled" << std::endl;
        return;
    }

    QString dir_path = selector.dirPath();

    if ( dir_path.isEmpty() )
    {
        std::cerr << "MainWindow::openDebugView() empty dir path" << std::endl;
        return;
    }

    std::cerr << "open debug view. dir = [" << dir_path.toStdString() << std::endl;

    M_main_data.getViewHolder().loadDebugView( dir_path.toStdString() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveDebugView()
{
    std::cerr << "MainWindow::saveDebugView()" << std::endl;
#if 0
    QString dir_path
        = QFileDialog::getExistingDirectory( this,
                                             tr( "Choose a directory to save a debug view logs" ) );

#endif
    DirSelector selector( this,
                          tr( "Choose a debug view log directory" ),
                          QDir::currentPath() );

    if ( ! selector.exec() )
    {
        std::cerr << "MainWindow::openDebugView() canceled" << std::endl;
        return;
    }

    QString dir_path = selector.dirPath();

    if ( dir_path.isEmpty() )
    {
        std::cerr << "MainWindow::openDebugView() empty dir path" << std::endl;
        return;
    }

    std::cerr << "save debug view. dir = [" << dir_path.toStdString() << std::endl;


    M_main_data.viewHolder().saveDebugView( dir_path.toStdString() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::kickOff()
{
    std::cerr << "MainWindow::kickOff()" << std::endl;
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        M_monitor_client->sendKickOff();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::setLiveMode()
{
    std::cerr << "MainWindow::setLiveMode()" << std::endl;

    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        M_log_player->setLiveMode();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::connectMonitor()
{
    std::cerr << "MainWindow::connectMonitor()" << std::endl;
    connectMonitorTo( "localhost" );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::connectMonitorTo()
{
    std::cerr << "MainWindow::connectMonitorTo()" << std::endl;

    bool ok = true;
    QString text = QInputDialog::getText( this,
                                          tr( "Input sserver host name" ),
                                          tr( "Host name: "),
                                          QLineEdit::Normal,
                                          tr( "localhost" ),
                                          & ok );
    if ( ok
         && ! text.isEmpty() )
    {
        connectMonitorTo( text.toStdString().c_str() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::connectMonitorTo( const char * hostname )
{
    if ( std::strlen( hostname ) == 0 )
    {
        std::cerr << "Empty host name! Connection failed!" << std::endl;
        return;
    }

    std::cerr << "Connect to rcssserver on [" << hostname << "]" << std::endl;

    M_monitor_client
        = boost::shared_ptr< MonitorClient >
        ( new MonitorClient( M_main_data.getViewHolder(),
                             hostname,
                             AppConfig::instance().port(),
                             AppConfig::instance().clientVersion() ) );

    if ( ! M_monitor_client->isConnected() )
    {
        std::cerr << "Conenction failed." << std::endl;
        M_monitor_client.reset();
        return;
    }

    // reset all data
    M_main_data.clear();

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->hide();
    }

    if ( M_debug_message_window )
    {
        M_debug_message_window->clearAll();
    }

    if ( M_view_config_dialog )
    {
        M_view_config_dialog->unzoom();
    }

    if ( M_debug_server )
    {
        M_debug_server.reset();
        startDebugServer();
    }

    AppConfig::instance().setMonitorClientMode( true );

    M_save_rcg_act->setEnabled( false );

    M_kick_off_act->setEnabled( true );
    M_set_live_mode_act->setEnabled( true );
    M_connect_monitor_act->setEnabled( false );
    M_connect_monitor_to_act->setEnabled( false );
    M_disconnect_monitor_act->setEnabled( true );
    M_kill_server_act->setEnabled( true );

    M_start_debug_server_act->setEnabled( true );
    M_stop_debug_server_act->setEnabled( false );
    M_show_image_save_dialog_act->setEnabled( false );

    connect( M_monitor_client.get(), SIGNAL( received() ),
             this, SLOT( receiveMonitorPacket() ) );
    connect( M_monitor_client.get(), SIGNAL( timeout() ),
             this, SLOT( disconnectMonitor() ) );

    M_log_player->setLiveMode();

    M_monitor_client->sendDispInit();

    if ( M_cursor_override )
    {
        QApplication::restoreOverrideCursor();
        M_cursor_override = false;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::disconnectMonitor()
{
    //std::cerr << "MainWindow::disconnectMonitor()" << std::endl;
    if ( M_monitor_client )
    {
        M_monitor_client->disconnect();

        disconnect( M_monitor_client.get(), SIGNAL( received() ),
                    this, SLOT( receiveMonitorPacket() ) );

        disconnect( M_monitor_client.get(), SIGNAL( timeout() ),
                    this, SLOT( disconnectMonitor() ) );

        M_monitor_client.reset();
    }

    if ( M_debug_server )
    {
        M_debug_server.reset();
    }

    AppConfig::instance().setMonitorClientMode( false );

    M_save_rcg_act->setEnabled( true );

    M_kick_off_act->setEnabled( false );
    M_set_live_mode_act->setEnabled( false );
    M_connect_monitor_act->setEnabled( true );
    M_connect_monitor_to_act->setEnabled( true );
    M_disconnect_monitor_act->setEnabled( false );
    M_kill_server_act->setEnabled( false );

    M_start_debug_server_act->setEnabled( false );
    M_stop_debug_server_act->setEnabled( false );
    M_show_image_save_dialog_act->setEnabled( true );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::killServer()
{
    std::cerr << "MainWindow::killServer()" << std::endl;

    disconnectMonitor();

    AppConfig::instance().setKillServer( false );

    if ( AppConfig::instance().serverPID() != 0 )
    {
        AppConfig::instance().setServerPID( 0 );
        ::kill( pid_t( AppConfig::instance().serverPID() ), SIGINT );
    }
    else
    {
        //std::system( "killall -s INT rcssserver" );
        QString command( "killall -s INT rcssserver" );
        QProcess::execute( command );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::startServer()
{
    std::cerr << "MainWindow::startServer()" << std::endl;

    AppConfig::instance().setServerPID( 0 );
    AppConfig::instance().setKillServer( true );

    //std::string rcssserver = AppConfig::instance().serverPath();
    //rcssserver += " > /dev/null 2>&1 &";
    //std::system( rcssserver.c_str() );
    QString command
        = QString::fromStdString( AppConfig::instance().serverPath() );
    QProcess::startDetached( command );

    if ( ! M_cursor_override )
    {
        QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );
        M_cursor_override = true;
    }

    QTimer::singleShot( 1000,
                        this, SLOT( connectMonitor() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::restartServer()
{
    std::cerr << "MainWindow::restartServer()" << std::endl;
    if ( M_monitor_client )
    {
        killServer();

        if ( ! M_cursor_override )
        {
            QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ) );
            M_cursor_override = true;
        }

        QTimer::singleShot( 1000,
                            this, SLOT( startServer() ) );
    }
    else
    {
        startServer();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleToolBar()
{
    M_log_player_tool_bar->setVisible( ! M_log_player_tool_bar->isVisible() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleStatusBar()
{
    this->statusBar()->setVisible( ! this->statusBar()->isVisible() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleFullScreen()
{
    if ( this->isFullScreen() )
    {
        this->showNormal();
    }
    else
    {
        this->showFullScreen();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showPlayerTypeDialog()
{
    std::cerr << "MainWindow::showPlayerTypeDialog()" << std::endl;

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->setVisible( ! M_player_type_dialog->isVisible() );
    }
    else
    {
        M_player_type_dialog = new PlayerTypeDialog( this, M_main_data );
        M_player_type_dialog->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showDetailDialog()
{
    std::cerr << "MainWindow::showDetailDialog()" << std::endl;

    if ( M_detail_dialog )
    {
        M_detail_dialog->setVisible( ! M_detail_dialog->isVisible() );
    }
    else
    {
        M_detail_dialog = new DetailDialog( this, M_main_data );
        connect( this,  SIGNAL( viewUpdated() ),
                 M_detail_dialog, SLOT( updateLabels() ) );

        M_detail_dialog->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::changeStyle( bool checked )
{
    if ( ! checked )
    {
        return;
    }

    QAction * action = qobject_cast< QAction * >( sender() );
    QStyle * style = QStyleFactory::create( action->data().toString() );
    Q_ASSERT( style );

    QApplication::setStyle( style );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showViewConfigDialog()
{
    std::cerr << "MainWindow::showViewConfigDialog()" << std::endl;

    M_view_config_dialog->setVisible( ! M_view_config_dialog->isVisible() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showDebugMessageWindow()
{
    std::cerr << "MainWindow::showDebugMessageWindow()" << std::endl;

    if ( M_debug_message_window )
    {
        M_debug_message_window
            ->setVisible( ! M_debug_message_window->isVisible() );
    }
    else
    {
        M_debug_message_window = new DebugMessageWindow( this,
                                                         M_main_data );
        connect( M_debug_message_window, SIGNAL( configured() ),
                 this, SIGNAL( viewUpdated() ) );

        M_debug_message_window->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::startDebugServer()
{
    std::cerr << "MainWindow::startDebugServer()" << std::endl;

    if ( M_debug_server )
    {
        // the server instance has already existed.
        return;
    }

    std::cerr << "Start Debug Server" << std::endl;

    int port = static_cast< int >( AppConfig::instance().debugServerPort() );

    M_debug_server
        = boost::shared_ptr< DebugServer >
        ( new DebugServer( M_main_data.getViewHolder(), port ) );

    if ( ! M_debug_server->isConnected() )
    {
        std::cerr << "failed to create Debug Server" << std::endl;
        M_debug_server.reset();
        return;
    }

    M_start_debug_server_act->setEnabled( false );
    M_stop_debug_server_act->setEnabled( true );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::stopDebugServer()
{
    std::cerr << "MainWindow::stopDebugServer()" << std::endl;

    if ( M_debug_server )
    {
        std::cerr << "Stop Debug Server" << std::endl;
        M_debug_server.reset();
    }


    M_start_debug_server_act->setEnabled( M_monitor_client
                                          ? true
                                          : false );
    M_stop_debug_server_act->setEnabled( false );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showImageSaveDialog()
{
    std::cerr << "MainWindow::showImageSaveDialog()" << std::endl;

    M_log_player->stop();

    ImageSaveDialog dlg( this,
                         M_field_canvas,
                         M_main_data );

    dlg.exec();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::about()
{
    QString msg( tr( PACKAGE_NAME ) );
    msg += tr( "-" );
    msg += tr( VERSION );
    msg += tr( "\n\n" );
    msg += tr( "soccerwindow2 is a viewer applicaton for\n"
               "the RoboCup Soccer Simulator.\n"
               "  http://sserver.sourceforge.net/\n"
               "\n"
               "soccerwindow2 Development Site:\n"
               "  http://rctools.sourceforge.jp/\n"
               "Author:\n"
               "  Hidehisa Akiyama <akky@users.sourceforge.jp>" );

    QMessageBox::about( this,
                        tr( "About soccerwindow2" ),
                        msg );

    // from Qt 4.1 documents
    /*
      about() looks for a suitable icon in four locations:

      1. It prefers parent->icon() if that exists.
      2. If not, it tries the top-level widget containing parent.
      3. If that fails, it tries the active window.
      4. As a last resort it uses the Information icon.

      The about box has a single button labelled "OK".
    */
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::resizeCanvas( const QSize & size )
{
    if ( size.width() < this->minimumWidth()
         || size.height() < this->minimumHeight() )
    {
        std::cerr << "Too small canvas size ("
                  << size.width() << " "
                  << size.height() << ")"
                  << std::endl;
        return;
    }

    if ( centralWidget() )
    {
        if ( this->isMaximized()
             || this->isFullScreen() )
        {
            this->showNormal();
        }

        QRect rect = this->geometry();

        int width_diff = rect.width() - centralWidget()->width();
        int height_diff = rect.height() - centralWidget()->height();

        rect.setWidth( size.width() + width_diff );
        rect.setHeight( size.height() + height_diff );

        this->setGeometry( rect );

        //std::cerr << "centralWidget width = " << centralWidget()->width()
        //          << " height = " << centralWidget()->height()
        //          << std::endl;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::receiveMonitorPacket()
{
    //std::cerr << "receive monitor packet" << std::endl;

    if ( M_log_player->isLiveMode() )
    {
        M_log_player->showLive();
    }
    else
    {
        std::cerr << "receive monitor packet  no live" << std::endl;
        M_log_player_tool_bar->updateSlider();
    }


    if ( AppConfig::instance().autoQuitMode() )
    {
        if ( M_main_data.viewHolder().lastPlayMode() == rcsc::PM_TimeOver )
        {
            static QDateTime s_game_end_time;

            if ( ! s_game_end_time.isValid() )
            {
                s_game_end_time = QDateTime::currentDateTime();
            }
            else
            {
                if ( s_game_end_time.secsTo( QDateTime::currentDateTime() )
                     >= AppConfig::instance().waitSeconds() )
                {
                    std::cerr << "Elapsed " << AppConfig::instance().waitSeconds()
                              << " seconds after game end\n"
                              << "Exit..."
                              << std::endl;
                    qApp->quit();
                }
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::updatePositionLabel( const QPoint & point )
{
    if ( M_position_label
         && statusBar()
         && statusBar()->isVisible()
         )
    {
        double x = M_main_data.viewConfig().getFieldX( point.x() );
        double y = M_main_data.viewConfig().getFieldY( point.y() );

        char buf[32];
        std::snprintf( buf, 32,
                       "(%.2f, %.2f)",
                       x, y );

        M_position_label->setText( QString::fromAscii( buf ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::dropBallThere()
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        MonitorViewConstPtr view = M_main_data.viewHolder().latestViewData();
        if ( view )
        {
            std::cerr << "drop ball at current position "
                      << std::endl;
            M_monitor_client->sendDropBall( view->ball().x(),
                                            view->ball().y() );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::dropBall( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().getFieldX( point.x() );
        double y = M_main_data.viewConfig().getFieldY( point.y() );

        std::cerr << "drop ball to ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendDropBall( x, y );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::freeKickLeft( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().getFieldX( point.x() );
        double y = M_main_data.viewConfig().getFieldY( point.y() );

        std::cerr << "free kick left at ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendFreeKickLeft( x, y );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::freeKickRight( const QPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().getFieldX( point.x() );
        double y = M_main_data.viewConfig().getFieldY( point.y() );

        std::cerr << "free kick right at ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendFreeKickRight( x, y );
    }
}
