// -*-c++-*-

/*!
  \file plaeyr_painter.cpp
  \brief player painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "player_painter.h"

#include "draw_config.h"
// model
#include "main_data.h"

#include <rcsc/common/player_type.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/angle_deg.h>

#include <cstring>
#include <cstdio>
#include <cmath>

/*-------------------------------------------------------------------*/
/*

*/
inline
PlayerPainter::Param::Param( const Player & player,
                             const Ball & ball,
                             const ViewConfig & view_conf,
                             const rcsc::PlayerType & ptype )
    : x_( view_conf.getScreenX( player.x() ) )
    , y_( view_conf.getScreenY( player.y() ) )
    , body_radius_( view_conf.scale( ptype.playerSize() ) )
    , kick_radius_( view_conf.scale( ptype.kickableArea() ) )
    , have_full_effort_( player.hasFullEffort( ptype.effortMax() ) )
    , player_( player )
    , ball_( ball )
    , player_type_( ptype )
{

    if ( body_radius_ < 1 ) body_radius_ = 1;
    if ( kick_radius_ < 5 ) kick_radius_ = 5;

    draw_radius_ = ( view_conf.isEnlarged()
                     ? kick_radius_
                     : body_radius_ );
}

/*-------------------------------------------------------------------*/
/*

*/
PlayerPainter::PlayerPainter( const MainData & main_data )
    : M_main_data( main_data )
{
    for ( int i = 0; i < rcsc::MAX_PLAYER * 2; ++i )
    {
        M_stamina_state_caches[i] = 0;
    }

    {
        QPainter painter;
        QFontMetrics fm = painter.fontMetrics();

        M_comma_pixmap
            = boost::shared_ptr< QPixmap >
            ( new QPixmap( fm.width( QString::fromAscii( "," ) ), fm.height() ) );
        M_comma_pixmap->fill( Qt::transparent );

        painter.begin( M_comma_pixmap.get() );
        painter.setPen( DrawConfig::instance().playerNumberFontPen() );
        painter.setFont( DrawConfig::instance().playerFont() );

        painter.drawText( M_comma_pixmap->rect(),
                          QString::fromAscii( "," ) );
        painter.end();
        //M_comma_pixmap->setMask( M_comma_pixmap->createMaskFromColor( Qt::transparent ) );


        M_dot_pixmap
            = boost::shared_ptr< QPixmap >
            ( new QPixmap( fm.width( QString::fromAscii( "." ) ), fm.height() ) );
        M_dot_pixmap->fill( Qt::transparent );

        painter.begin( M_dot_pixmap.get() );
        painter.setPen( DrawConfig::instance().playerNumberFontPen() );
        painter.setFont( DrawConfig::instance().playerFont() );
        painter.drawText( M_dot_pixmap->rect(),
                          QString::fromAscii( "." ) );
        painter.end();
        //M_dot_pixmap->setMask( M_dot_pixmap->createMaskFromColor( Qt::transparent ) );


        M_t_pixmap
            = boost::shared_ptr< QPixmap >
            ( new QPixmap( fm.width( QString::fromAscii( "t" ) ), fm.height() ) );
        M_t_pixmap->fill( Qt::transparent );

        painter.begin( M_t_pixmap.get() );
        painter.setPen( DrawConfig::instance().playerNumberFontPen() );
        painter.setFont( DrawConfig::instance().playerFont() );
        painter.drawText( M_t_pixmap->rect(),
                          QString::fromAscii( "t" ) );
        painter.end();
        //M_t_pixmap->setMask( M_t_pixmap->createMaskFromColor( Qt::transparent ) );


        M_e_pixmap
            = boost::shared_ptr< QPixmap >
            ( new QPixmap( fm.width( QString::fromAscii( "e" ) ), fm.height() ) );
        M_e_pixmap->fill( Qt::transparent );

        painter.begin( M_e_pixmap.get() );
        painter.setPen( DrawConfig::instance().playerNumberFontPen() );
        painter.setFont( DrawConfig::instance().playerFont() );
        painter.drawText( M_e_pixmap->rect(),
                          QString::fromAscii( "e" ) );
        painter.end();
        //M_e_pixmap->setMask( M_e_pixmap->createMaskFromColor( Qt::transparent ) );


        M_r_pixmap
            = boost::shared_ptr< QPixmap >
            ( new QPixmap( fm.width( QString::fromAscii( "r" ) ), fm.height() ) );
        M_r_pixmap->fill( Qt::transparent );

        painter.begin( M_r_pixmap.get() );
        painter.setPen( DrawConfig::instance().playerNumberFontPen() );
        painter.setFont( DrawConfig::instance().playerFont() );
        painter.drawText( M_r_pixmap->rect(),
                          QString::fromAscii( "r" ) );
        painter.end();
        //M_r_pixmap->setMask( M_r_pixmap->createMaskFromColor( Qt::transparent ) );
    }

    for ( int i = 0; i < 12; ++i )
    {
        QString digit = QString::number( i );

        QPainter painter;
        QFontMetrics fm = painter.fontMetrics();

        M_number_pixmap[i]
            = boost::shared_ptr< QPixmap >
            ( new QPixmap( fm.width( digit ), fm.height() ) );
        M_number_pixmap[i]->fill( Qt::transparent );

        painter.begin( M_number_pixmap[i].get() );
        painter.setPen( DrawConfig::instance().playerNumberFontPen() );
        painter.setFont( DrawConfig::instance().playerFont() );
        painter.drawText( M_number_pixmap[i]->rect(),
                          digit );
        //M_number_pixmap[i]->setMask( M_number_pixmap[i]->createMaskFromColor( Qt::transparent ) );
    }

    for ( int i = 0; i <= 9; ++i )
    {
        QString digit = QString::number( i );

        QPainter painter;
        QFontMetrics fm = painter.fontMetrics();

        M_stamina_pixmap[i]
            = boost::shared_ptr< QPixmap >
            ( new QPixmap( fm.width( digit ), fm.height() ) );
        M_stamina_pixmap[i]->fill( Qt::transparent );

        painter.begin( M_stamina_pixmap[i].get() );
        painter.setPen( DrawConfig::instance().playerStaminaFontPen() );
        painter.setFont( DrawConfig::instance().playerFont() );
        painter.drawText( M_stamina_pixmap[i]->rect(),
                          digit );
        //M_stamina_pixmap[i]->setMask( M_stamina_pixmap[i]->createMaskFromColor( Qt::transparent ) );
    }
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::draw( QPainter & painter )
{
    if ( ! M_main_data.viewConfig().isShownPlayers() )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );

    if ( ! view )
    {
        return;
    }

    const Ball & ball = view->ball();

    const std::vector< Player >::const_iterator it_end = view->players().end();
    for ( std::vector< Player >::const_iterator it = view->players().begin();
          it != it_end;
          ++it )
    {
        const Param param( *it,
                           ball,
                           M_main_data.viewConfig(),
                           M_main_data.viewHolder().playerType( it->type() ) );

        drawBody( painter, param );
        drawShadow( painter, param );
        drawEdge( painter, param );

        if ( M_main_data.viewConfig().isSelectedPlayer( it->side(), it->unum() )
             && M_main_data.viewConfig().playerFutureCycle() > 0
             && it->hasDelta() )
        {
            drawFuture( painter, param );
        }

        if ( it->hasView()
             && M_main_data.viewConfig().isShownViewCone() )
        {
            drawViewCone( painter, param );
        }

        if ( M_main_data.viewConfig().isShownControlArea() )
        {
            drawControlArea( painter, param );
        }

        drawText( painter, param );
    }

}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::drawBody( QPainter & painter,
                         const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // decide base color
    painter.setPen( dconf.playerPen() );

    switch ( param.player_.side() ) {
    case rcsc::LEFT:
        if ( param.player_.isGoalie() )
        {
            painter.setBrush( dconf.leftGoalieBrush() );
        }
        else
        {
            painter.setBrush( dconf.leftTeamBrush() );
        }
        break;
    case rcsc::RIGHT:
        if ( param.player_.isGoalie() )
        {
            painter.setBrush( dconf.rightGoalieBrush() );
        }
        else
        {
            painter.setBrush( dconf.rightTeamBrush() );
        }
        break;
    case rcsc::NEUTRAL:
        painter.setBrush( dconf.shadowBrush() );
        break;
    default:
        painter.setBrush( dconf.shadowBrush() );
        break;
    }


    // decide status color
    if ( ! param.player_.isAlive() )
    {
        painter.setBrush( dconf.shadowBrush() );
    }
    if ( param.player_.isKick() )
    {
        painter.setPen( dconf.kickPen() );
    }
    if ( param.player_.isKickFault() )
    {
        painter.setPen( dconf.kickFaultPen() );
        painter.setBrush( dconf.kickFaultBrush() );
    }
    if ( param.player_.isCatch() )
    {
        painter.setBrush( dconf.catchBrush() );
    }
    if ( param.player_.isCatchFault() )
    {
        painter.setBrush( dconf.catchFaultBrush() );
    }
    if ( param.player_.isTackle() )
    {
        painter.setPen( dconf.tacklePen() );
        painter.setBrush( dconf.tackleBrush() );
    }
    if ( param.player_.isTackleFault() )
    {
        painter.setPen( dconf.tacklePen() );
        painter.setBrush( dconf.tackleFaultBrush() );
    }
    if ( param.player_.isCollidedBall() )
    {
        painter.setBrush( dconf.collideBallBrush() );
    }
    if ( param.player_.isCollidedPlayer() )
    {
        painter.setBrush( dconf.collidePlayerBrush() );
    }

#if 0
    QRadialGradient gradient( param.x_,
                              param.y_,
                              param.draw_radius_,
                              param.x_ - param.draw_radius_ ,
                              param.y_ - param.draw_radius_ );
    gradient.setColorAt( 0.0, painter.brush().color() );
    gradient.setColorAt( 1.0, Qt::black );
    painter.setBrush( gradient );
#endif
    painter.drawEllipse( param.x_ - param.draw_radius_ ,
                         param.y_ - param.draw_radius_ ,
                         param.draw_radius_ * 2,
                         param.draw_radius_ * 2 );
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::drawShadow( QPainter & painter,
                           const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // set size
    int shadow_radius = static_cast< int >( param.draw_radius_ * 0.9 );
    // set angle
    //int shadow_start_dir
    //    = static_cast< int >( rint( ( - param.player_.body() + 90.0 ) * 16 ) );
    //double shadow_end_dir = shadow_start_dir + 180.0;

    // decide shadow color
#if 0
    double stamina_rate = ( param.player_.hasStamina()
                            ? ( param.player_.stamina()
                                / param.server_param_.staminaMax() )
                            : 1.0 );
    const QColour & base_col = dconf.playerShadowBrush().color();
    int r = 255 - static_cast< int >( (255 - base_col.red()) * stamina_rate );
    int g = 255 - static_cast< int >( (255 - base_col.green()) * stamina_rate );
    int b = 255 - static_cast< int >( (255 - base_col.blue()) * stamina_rate );
    QBrush tmp_brush( QColour( r, g, b ), Qt::SolidPattern );
    painter.setPen( dconf.transparentPen() );
    painter.setBrush( tmp_brush );
#elif 0
    // create temporary brush
    int col = 255
        - static_cast< int >( 255
                              * ( param.player_.hasStamina()
                                  ? ( param.player_.stamina()
                                      / rcsc::ServerParam::i().staminaMax() )
                                  : 1.0 ) );
    //QBrush tmp_brush( QColor( col, col, col ), Qt::SolidPattern );
    painter.setPen( dconf.transparentPen() );
    //painter.setBrush( tmp_brush );
    painter.setBrush( dconf.shadowBrush( col ) );
#else
    painter.setPen( dconf.transparentPen() );
    double stamina_rate = ( param.player_.hasStamina()
                            ? ( param.player_.stamina()
                                / rcsc::ServerParam::i().staminaMax() )
                            : 1.0 );
    //int level = 255 - (int)rint( 255 * rint( stamina_rate * 16.0 ) / 16.0 );
    int level = 255 - (int)rint( 255 * rint( stamina_rate * 8.0 ) / 8.0 );
    painter.setBrush( dconf.shadowBrush( level ) );
#endif
    // draw half circle of shadow
    QPainterPath shadow_path;
    shadow_path.moveTo( param.x_, param.y_ );
    shadow_path.arcTo( param.x_ - shadow_radius,
                       param.y_ - shadow_radius,
                       shadow_radius * 2,
                       shadow_radius * 2,
                       ( - param.player_.body() + 90.0 ),
                       180.0 );

    painter.drawPath( shadow_path );
    /*
      painter.drawPie( param.x_ - shadow_radius, // left x
      param.y_ - shadow_radius, // top y
      shadow_radius * 2, // width
      shadow_radius * 2, // height
      shadow_start_dir,
      180 * 16 ); // span angle
    */
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawEdge( QPainter & painter,
                         const PlayerPainter::Param & param )
{
    const DrawConfig & dconf = DrawConfig::instance();

#if 0
    static int s_count = 0;
    const int index = ( param.player_.side() == rcsc::LEFT
                        ? param.player_.unum() - 1
                        : param.player_.unum() - 1 + rcsc::MAX_PLAYER );
    const int pixmap_radius = param.kick_radius_ + 2;

    bool redraw = false;

    if ( ! M_edge_pixmap[index]
         || M_edge_pixmap[index]->width() != pixmap_radius * 2 )
    {
        M_edge_pixmap[index]
            = boost::shared_ptr< QPixmap >( new QPixmap( pixmap_radius * 2,
                                                         pixmap_radius * 2 ) );
        redraw = true;
    }

    if ( param.player_.hasStamina() )
    {
        if ( ! param.have_full_effort_ )
        {
            if ( M_stamina_state_caches[index] != 1 )
            {
                redraw = true;
                M_stamina_state_caches[index] = 1;
            }
        }
       else if  ( ! param.player_.hasFullRecovery() )
        {
            if ( M_stamina_state_caches[index] != 2 )
            {
                redraw = true;
                M_stamina_state_caches[index] = 2;
            }
        }
        else
        {
            if ( M_stamina_state_caches[index] != 0 )
            {
                redraw = true;
                M_stamina_state_caches[index] = 0;
            }
        }
    }

    if ( redraw )
    {
        std::cerr << ++s_count << " redraw " << param.player_.unum() << std::endl;
        M_edge_pixmap[index]->fill( Qt::transparent );

        QPainter pm_painter( M_edge_pixmap[index].get() );
        pm_painter.setRenderHint( QPainter::Antialiasing );
        pm_painter.setBrush( dconf.transparentBrush() );

        {
            if ( M_main_data.viewConfig().isEnlarged() )
            {
                pm_painter.setPen( dconf.realBodyPen() );

            }
            else if ( param.player_.side() == rcsc::LEFT )
            {
                pm_painter.setPen( dconf.leftTeamPen() );
            }
            else
            {
                pm_painter.setPen( dconf.rightTeamPen() );
            }

            int radius = ( M_main_data.viewConfig().isEnlarged()
                           ? param.body_radius_
                           : param.kick_radius_ );
            pm_painter.drawEllipse( pixmap_radius - radius,
                                    pixmap_radius - radius,
                                    radius * 2,
                                    radius * 2 );
        }

        if ( param.player_.hasStamina() )
        {
            if ( ! param.have_full_effort_ )
            {
                pm_painter.setPen( dconf.effortDecayedPen() );
                int radius = param.draw_radius_ + 2;
                pm_painter.drawEllipse( pixmap_radius - radius,
                                        pixmap_radius - radius,
                                        radius * 2,
                                        radius * 2 );
            }
        }
        else if  ( ! param.player_.hasFullRecovery() )
        {
            pm_painter.setPen( dconf.recoveryDecayedPen() );
            int radius = param.draw_radius_ + 2;
            pm_painter.drawEllipse( pixmap_radius - radius,
                                    pixmap_radius - radius,
                                    radius * 2,
                                    radius * 2 );
        }
        else
        {
            pm_painter.setPen( Qt::black );
            int radius = param.draw_radius_ + 2;
            pm_painter.drawEllipse( pixmap_radius - radius,
                                    pixmap_radius - radius,
                                    radius * 2,
                                    radius * 2 );
        }
        pm_painter.end();
    }

    painter.drawPixmap( param.x_ - pixmap_radius,
                        param.y_ - pixmap_radius,
                        *M_edge_pixmap[index] );

#else
    // draw body edge or kickable area edge
    if ( M_main_data.viewConfig().isEnlarged() )
    {
        // draw real body edge
        painter.setPen( dconf.realBodyPen() );
        painter.setBrush( dconf.transparentBrush() );
        painter.drawEllipse( param.x_ - param.body_radius_,
                             param.y_ - param.body_radius_,
                             param.body_radius_ * 2,
                             param.body_radius_ * 2 );
    }
    else
    {
        // draw kickable area edge
        if ( param.player_.side() == rcsc::LEFT )
        {
            painter.setPen( dconf.leftTeamPen() );
        }
        else
        {
            painter.setPen( dconf.rightTeamPen() );
        }
        painter.setBrush( dconf.transparentBrush() );
        painter.drawEllipse( param.x_ - param.kick_radius_,
                             param.y_ - param.kick_radius_,
                             param.kick_radius_ * 2,
                             param.kick_radius_ * 2 );
    }

    // draw stamina status if effort or recovery is decayed.
    if ( param.player_.hasStamina() )
    {
        if ( ! param.have_full_effort_ )
        {
            painter.setPen( dconf.effortDecayedPen() );
            painter.setBrush( dconf.transparentBrush() );
            int radius = param.draw_radius_ + 2;
            painter.drawEllipse( param.x_ - radius,
                                 param.y_ - radius,
                                 radius * 2,
                                 radius * 2 );
        }
        else if ( ! param.player_.hasFullRecovery() )
        {
            painter.setPen( dconf.recoveryDecayedPen() );
            painter.setBrush( dconf.transparentBrush() );
            int radius = param.draw_radius_ + 2;
            painter.drawEllipse( param.x_ - radius,
                                 param.y_ - radius,
                                 radius * 2,
                                 radius * 2 );
        }
    }
#endif

}


/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawFuture( QPainter & painter,
                           const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // draw future state
    rcsc::Vector2D ppos( param.player_.x(), param.player_.y() );
    rcsc::Vector2D pvel( param.player_.deltaX(), param.player_.deltaY() );

    const int last = M_main_data.viewConfig().playerFutureCycle();

    QPoint first_point( M_main_data.viewConfig().getScreenX( ppos.x ),
                        M_main_data.viewConfig().getScreenY( ppos.y ) ) ;
    QPoint last_point = first_point;

    // draw kickable area edge
    if ( param.player_.side() == rcsc::LEFT )
    {
        painter.setPen( dconf.rightTeamPen() );
    }
    else
    {
        painter.setPen( dconf.leftTeamPen() );
    }

    painter.setBrush( dconf.transparentBrush() );

    QPainterPath path;
    for ( int i = 0; i < last; ++i )
    {
        ppos += pvel;
        pvel *= param.player_type_.playerDecay();

        QPoint pt( M_main_data.viewConfig().getScreenX( ppos.x ),
                   M_main_data.viewConfig().getScreenY( ppos.y ) );
        if ( std::abs( last_point.x() - pt.x() ) < 1
             && std::abs( last_point.y() - pt.y() ) < 1 )
        {
            break;
        }

        path.addEllipse( pt.x() - 1,
                         pt.y() - 1,
                         2,
                         2 );
        path.addEllipse( pt.x() - param.kick_radius_,
                         pt.y() - param.kick_radius_,
                         param.kick_radius_ * 2,
                         param.kick_radius_ * 2 );

        last_point = pt;
    }

    painter.drawPath( path );

    // draw move line
    painter.setPen( dconf.debugTargetPen() );
    painter.drawLine( first_point, last_point );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawViewCone( QPainter & painter,
                             const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double view_width = param.player_.viewWidth();

    const double visible_dist = rcsc::ServerParam::i().visibleDistance();
    const int visible_radius = vconf.scale( visible_dist );

    const double view_start_angle = -param.player_.head() - view_width * 0.5;
    const double view_start_angle_real = ( param.player_.head() - view_width * 0.5 ) * rcsc::AngleDeg::DEG2RAD;
    const double view_end_angle_real = ( param.player_.head() + view_width * 0.5 ) * rcsc::AngleDeg::DEG2RAD;

    // draw enlarged view cone and feel area
    if ( vconf.isSelectedPlayer( param.player_.side(),
                                 param.player_.unum() ) )
    {
        const int UNUM_FAR = vconf.scale( 20.0 );
        const int TEAM_FAR = vconf.scale( 40.0 );
        const int TEAM_TOOFAR = vconf.scale( 60.0 );

        painter.setPen( dconf.linePen() );
        //painter.setBrush( dconf.transparentBrush() );

        int view_start_angle_int
            = static_cast< int >( rint( view_start_angle * 16 ) );
        int span_angle = static_cast< int >( rint( view_width * 16 ) );

        painter.drawArc( param.x_ - UNUM_FAR, // left x
                         param.y_ - UNUM_FAR, // toop y
                         UNUM_FAR * 2, // width
                         UNUM_FAR * 2, // height
                         view_start_angle_int,
                         span_angle );
        painter.drawArc( param.x_ - TEAM_FAR, // left x
                         param.y_ - TEAM_FAR, // toop y
                         TEAM_FAR * 2, // width
                         TEAM_FAR * 2, // height
                         view_start_angle_int,
                         span_angle );
        // pie, no an arc
        painter.drawArc( param.x_ - TEAM_TOOFAR, // left x
                         param.y_ - TEAM_TOOFAR, // toop y
                         TEAM_TOOFAR * 2, // width
                         TEAM_TOOFAR * 2, // height
                         view_start_angle_int,
                         span_angle );

        // draw feeling area circle
        painter.drawArc( param.x_ - visible_radius,
                         param.y_ - visible_radius,
                         visible_radius * 2,
                         visible_radius * 2,
                         0,
                         360 * 16 );
        // left side view cone end point x
        int lx = param.x_
            + vconf.scale( 60.0 * std::cos( view_start_angle_real ) );
        // left side view cone end point y
        int ly = param.y_
            + vconf.scale( 60.0  * std::sin( view_start_angle_real ) );
        // right side view cone end point x
        int rx = param.x_
            + vconf.scale( 60.0 * std::cos( view_end_angle_real ) );
        // right side view cone end point y
        int ry = param.y_
            + vconf.scale( 60.0 * std::sin( view_end_angle_real ) );
        painter.drawLine( lx, ly, param.x_, param.y_ );
        painter.drawLine( rx, ry, param.x_, param.y_ );
    }
    else
    {
        // draw normal view cone
        painter.setPen( dconf.viewConePen() );
        painter.setBrush( dconf.transparentBrush() );

        QPainterPath view_cone_path;
        view_cone_path.moveTo( param.x_, param.y_ );
        view_cone_path.arcTo( param.x_ - visible_radius,
                              param.y_ - visible_radius,
                              visible_radius * 2,
                              visible_radius * 2,
                              view_start_angle,
                              view_width );
        view_cone_path.closeSubpath();

        painter.drawPath( view_cone_path );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawControlArea( QPainter & painter,
                                const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    if ( param.player_.isGoalie() )
    {
        int catchable = vconf.scale( rcsc::ServerParam::i().catchableArea() );
        painter.setPen( ( param.player_.side() == rcsc::LEFT )
                        ? dconf.leftGoaliePen()
                        : dconf.rightGoaliePen() );
        painter.setBrush( dconf.transparentBrush() );;

        painter.drawEllipse( param.x_ - catchable,
                             param.y_ - catchable,
                             catchable * 2,
                             catchable * 2 );
    }

    rcsc::Vector2D ppos( param.player_.x(),
                         param.player_.y() );
    rcsc::Vector2D bpos( param.ball_.x(),
                         param.ball_.y() );
    rcsc::Vector2D player_to_ball = bpos - ppos;
    player_to_ball.rotate( - param.player_.body() );

    // draw tackle area & probability
    double tackle_dist = ( player_to_ball.x > 0.0
                           ? rcsc::ServerParam::i().tackleDist()
                           : rcsc::ServerParam::i().tackleBackDist() );
    double tackle_prob = ( std::pow( player_to_ball.absX() / tackle_dist,
                                     rcsc::ServerParam::i().tackleExponent() )
                           + std::pow( player_to_ball.absY() / rcsc::ServerParam::i().tackleWidth(),
                                       rcsc::ServerParam::i().tackleExponent() ) );
    if ( tackle_prob < 1.0 )
    {
        rcsc::AngleDeg body_angle = param.player_.body();
        rcsc::AngleDeg body_angle_side = body_angle + 90.0;
        double body_x = body_angle.cos();
        double body_y = body_angle.sin();
        double forward_x = rcsc::ServerParam::i().tackleDist() * body_x;
        double forward_y = rcsc::ServerParam::i().tackleDist() * body_y;
        double back_x = rcsc::ServerParam::i().tackleBackDist() * -body_x;
        double back_y = rcsc::ServerParam::i().tackleBackDist() * -body_y;
        double right_x = rcsc::ServerParam::i().tackleWidth() * body_angle_side.cos();
        double right_y = rcsc::ServerParam::i().tackleWidth() * body_angle_side.sin();

        QPoint pts[4];
        pts[0].setX( vconf.getScreenX( ppos.x + forward_x + right_x ) );
        pts[0].setY( vconf.getScreenY( ppos.y + forward_y + right_y ) );
        pts[1].setX( vconf.getScreenX( ppos.x + forward_x - right_x ) );
        pts[1].setY( vconf.getScreenY( ppos.y + forward_y - right_y ) );
        pts[2].setX( vconf.getScreenX( ppos.x + back_x - right_x ) );
        pts[2].setY( vconf.getScreenY( ppos.y + back_y - right_y ) );
        pts[3].setX( vconf.getScreenX( ppos.x + back_x + right_x ) );
        pts[3].setY( vconf.getScreenY( ppos.y + back_y + right_y ) );

        painter.setPen( dconf.tackleAreaPen() );
        painter.setBrush( dconf.transparentBrush() );

        painter.drawPolyline( pts, 4 );


        int text_radius = param.draw_radius_;
        if ( text_radius > 40 )
        {
            text_radius = 40;
        }

        painter.setFont( dconf.playerFont() );
        painter.setPen( dconf.tacklePen() );

        char msg[32];
        std::snprintf( msg, 32, "TakleProb=%.3f", 1.0 - tackle_prob );
        painter.drawText( param.x_ + text_radius,
                          param.y_ - text_radius + painter.fontMetrics().height(),
                          QString::fromAscii( msg ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawText( QPainter & painter,
                         const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const int text_radius = std::min( 40, param.draw_radius_ );

    if ( vconf.isShownPlayerNumber()
         && vconf.isShownHeteroNumber() )
    {
        int num_x = param.x_ + text_radius + 3;
        int num_y = param.y_ - M_number_pixmap[0]->height()/2;
        painter.drawPixmap( num_x, num_y,
                            *M_number_pixmap[param.player_.unum()] );
        num_x += M_number_pixmap[param.player_.unum()]->width();
        painter.drawPixmap( num_x, num_y,
                            *M_comma_pixmap );
        num_x += M_comma_pixmap->width();
        painter.drawPixmap( num_x, num_y,
                            *M_t_pixmap );
        num_x += M_t_pixmap->width();
        painter.drawPixmap( num_x, num_y,
                            *M_number_pixmap[param.player_.type()] );
    }
    else if ( vconf.isShownPlayerNumber() )
    {
        painter.drawPixmap( param.x_ + text_radius + 3,
                            param.y_ - M_number_pixmap[0]->height()/2,
                            *M_number_pixmap[param.player_.unum()] );
    }
    else if ( vconf.isShownHeteroNumber() )
    {
        int num_x = param.x_ + text_radius + 3;
        int num_y = param.y_ - M_number_pixmap[0]->height()/2;
        painter.drawPixmap( num_x, num_y,
                            *M_t_pixmap );
        num_x += M_t_pixmap->width();
        painter.drawPixmap( num_x, num_y,
                            *M_number_pixmap[param.player_.type()] );
    }

    if ( param.player_.hasStamina()
         && vconf.isShownStamina() )
    {
        // this player is selected
        if ( vconf.isSelectedPlayer( param.player_.side(),
                                     param.player_.unum() ) )
        {
#if 1
            int text_x = param.x_ - text_radius;
            int text_y = param.y_ - text_radius - M_stamina_pixmap[0]->height() + 3;
            int stamina = static_cast< int >( rint( param.player_.stamina() ) );

            int index = ( stamina / 1000 ) % 10;

            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            index = ( stamina / 100 ) % 10;
            painter.drawPixmap( text_x,
                                text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            index = ( stamina / 10 ) % 10;
            painter.drawPixmap( text_x,
                                text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            index = stamina % 10;
            painter.drawPixmap( text_x,
                                text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            painter.drawPixmap( text_x, text_y,
                                *M_comma_pixmap );
            text_x += M_comma_pixmap->width();

            painter.drawPixmap( text_x, text_y,
                                *M_e_pixmap );
            text_x += M_e_pixmap->width();

            int effort = static_cast< int >( rint( param.player_.effort() * 1000 ) );

            index = ( effort / 1000 ) % 10;
            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            painter.drawPixmap( text_x, text_y,
                                *M_dot_pixmap );
            text_x += M_dot_pixmap->width();

            index = ( effort / 100 ) % 10;
            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            index = ( effort / 10 ) % 10;
            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            index = effort % 10;
            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            int recovery = static_cast< int >( rint( param.player_.recovery() * 1000 ) );

            index = ( recovery / 1000 ) % 10;
            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            painter.drawPixmap( text_x, text_y,
                                *M_dot_pixmap );
            text_x += M_dot_pixmap->width();

            index = ( recovery / 100 ) % 10;
            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            index = ( recovery / 10 ) % 10;
            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
            text_x += M_number_pixmap[index]->width();

            index = recovery % 10;
            painter.drawPixmap( text_x, text_y,
                                *M_number_pixmap[index] );
#else
            char buf[32];
            std::snprintf( buf, 32,
                           "%4.0f,e%.3f,r%.3f",
                           param.player_.stamina(),
                           param.player_.effort(),
                           param.player_.recovery() );
            painter.setPen( dconf.playerNumberFontPen() );
            painter.drawText( param.x_ - text_radius,
                              param.y_ - text_radius - 3,
                              QString::fromAscii( buf ) );
#endif
        }
        else
        {
#if 1
            int text_x = param.x_ - text_radius;
            int text_y = param.y_ - text_radius - M_stamina_pixmap[0]->height() + 3;
            int stamina = static_cast< int >( rint( param.player_.stamina() ) );
            int index = ( stamina / 1000 ) % 10;

            painter.drawPixmap( text_x, text_y,
                                *M_stamina_pixmap[index] );
            text_x += M_stamina_pixmap[index]->width();

            index = ( stamina / 100 ) % 10;
            painter.drawPixmap( text_x,
                                text_y,
                                *M_stamina_pixmap[index] );
            text_x += M_stamina_pixmap[index]->width();

            index = ( stamina / 10 ) % 10;
            painter.drawPixmap( text_x,
                                text_y,
                                *M_stamina_pixmap[index] );
            text_x += M_stamina_pixmap[index]->width();

            index = stamina % 10;
            painter.drawPixmap( text_x,
                                text_y,
                                *M_stamina_pixmap[index] );
#else
            char buf[16];
            std::snprintf( buf, 16, "%4d",
                           static_cast< int >( rint( param.player_.stamina() ) ) );

            painter.setPen( DrawConfig::instance().playerStaminaFontPen() );
            painter.setFont( DrawConfig::instance().playerFont() );
            painter.drawText( param.x_ - text_radius,
                              param.y_ - text_radius - 3,
                              QString::fromAscii( buf ) );

#endif
        }
    }
}
