// -*-c++-*-

/*!
  \file debug_message_window.cpp
  \brief debug management window class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "debug_message_window.h"

#include "main_data.h"
#include "debug_log_holder.h"
#include "debug_log_data.h"
#include "debug_log_dir_dialog.h"
#include "app_config.h"

#include <cassert>
#include <cstdlib> // getenv
#include <iostream>

#include "xpm/debug_view_all.xpm"
#include "xpm/debug_view_self.xpm"
#include "xpm/debug_view_ball.xpm"
#include "xpm/debug_view_player.xpm"
#include "xpm/debug_view_comment.xpm"
#include "xpm/debug_view_line.xpm"
#include "xpm/debug_view_target.xpm"
#include "xpm/debug_view_message.xpm"

#include "xpm/open_dir.xpm"

#include "xpm/sync.xpm"
#include "xpm/logplayer_one_step_back.xpm"
#include "xpm/logplayer_one_step_forward.xpm"

#include "xpm/num_1.xpm"
#include "xpm/num_2.xpm"
#include "xpm/num_3.xpm"
#include "xpm/num_4.xpm"
#include "xpm/num_5.xpm"
#include "xpm/num_6.xpm"
#include "xpm/num_7.xpm"
#include "xpm/num_8.xpm"
#include "xpm/num_9.xpm"
#include "xpm/num_10.xpm"
#include "xpm/num_11.xpm"
#include "xpm/num_12.xpm"
#include "xpm/num_13.xpm"
#include "xpm/num_14.xpm"
#include "xpm/num_15.xpm"
#include "xpm/num_16.xpm"

/*-------------------------------------------------------------------*/
/*!

*/
DebugMessageWindow::DebugMessageWindow( QWidget * parent,
                                        MainData & main_data )
    : QMainWindow( parent )
    , M_main_data( main_data )
    , M_debug_holder( new DebugLogHolder )
    , M_level( 0xFFFF )
{
    //this->setWindowTitle( tr( "Debug Message" ) ); // qt4
    this->setCaption( tr( "Debug Message" ) ); // qt3

    createActions();
    createWindows();

    loadSettings();

    this->resize( 640, 480 );
}

/*-------------------------------------------------------------------*/
/*!

*/
DebugMessageWindow::~DebugMessageWindow()
{
    //std::cerr << "delete DebugMessageWindow" << std::endl;
    saveSettings();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::loadSettings()
{
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./DebugMessageWindow" );

    for ( int i = 0; i < 16; ++i )
    {
        char key[16];
        std::snprintf( key, 16, "debug%02d", i + 1 );

        //QVariant val = settings.value( QString::fromAscii( key ) );
        bool ok = false;
        bool enable = settings.readBoolEntry( QString::fromAscii( key ),
                                              true,
                                              &ok );
        if ( ok )
        {
            int level = DebugLogData::LEVEL_1 << i;
            if ( enable )
            {
                M_level |= level;
            }
            else
            {
                M_level &= ~level;
            }

            M_debug_level_act[i]->setOn( enable );
        }

    }

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::saveSettings()
{
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./DebugMessageWindow" );

    for ( int i = 0; i < 16; ++i )
    {
        char key[16];
        std::snprintf( key, 16, "debug%02d", i + 1 );

        settings.writeEntry( QString::fromAscii( key ),
                             M_debug_level_act[i]->isOn() );
    }

    settings.endGroup();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::createActions()
{
    // view switches

    M_show_debug_view_all_act
        = new QAction( QIconSet( QPixmap( debug_view_all_xpm ) ),
                       tr( "Toggle debug view" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_all_act->setStatusTip( tr( "Show/Hide all debug view" ) );
    M_show_debug_view_all_act->setToggleAction( true );
    M_show_debug_view_all_act->setOn( true );
    connect( M_show_debug_view_all_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewAll( bool ) ) );

    M_show_debug_view_self_act
        = new QAction( QIconSet( QPixmap( debug_view_self_xpm ) ),
                       tr( "Toggle debug view self" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_self_act->setStatusTip( tr( "Show/Hide debug view self" ) );
    M_show_debug_view_self_act->setToggleAction( true );
    M_show_debug_view_self_act->setOn( true );
    connect( M_show_debug_view_self_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewSelf( bool ) ) );

    M_show_debug_view_ball_act
        = new QAction( QIconSet( QPixmap( debug_view_ball_xpm ) ),
                       tr( "Toggle debug view ball" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_ball_act->setStatusTip( tr( "Show/Hide debug view ball" ) );
    M_show_debug_view_ball_act->setToggleAction( true );
    M_show_debug_view_ball_act->setOn( true );
    connect( M_show_debug_view_ball_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewBall( bool ) ) );

    M_show_debug_view_players_act
        = new QAction( QIconSet( QPixmap( debug_view_player_xpm ) ),
                       tr( "Toggle debug view players" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_players_act->setStatusTip( tr( "Show/Hide debug view players" ) );
    M_show_debug_view_players_act->setToggleAction( true );
    M_show_debug_view_players_act->setOn( true );
    connect( M_show_debug_view_players_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewPlayers( bool ) ) );

    M_show_debug_view_comment_act
        = new QAction( QIconSet( QPixmap( debug_view_comment_xpm ) ),
                       tr( "Toggle debug view comment" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_comment_act->setStatusTip( tr( "Show/Hide debug view comment" ) );
    M_show_debug_view_comment_act->setToggleAction( true );
    M_show_debug_view_comment_act->setOn( true );
    connect( M_show_debug_view_comment_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewComment( bool ) ) );

    M_show_debug_view_figure_act
        = new QAction( QIconSet( QPixmap( debug_view_line_xpm ) ),
                       tr( "Toggle debug view figures" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_figure_act->setStatusTip( tr( "Show/Hide debug view figures" ) );
    M_show_debug_view_figure_act->setToggleAction( true );
    M_show_debug_view_figure_act->setOn( true );
    connect( M_show_debug_view_figure_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewFigure( bool ) ) );

    M_show_debug_view_target_act
        = new QAction( QIconSet( QPixmap( debug_view_target_xpm ) ),
                       tr( "Toggle debug view target" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_target_act->setStatusTip( tr( "Show/Hide debug view target" ) );
    M_show_debug_view_target_act->setToggleAction( true );
    M_show_debug_view_target_act->setOn( true );
    connect( M_show_debug_view_target_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewTarget( bool ) ) );

    M_show_debug_view_message_act
        = new QAction( QIconSet( QPixmap( debug_view_message_xpm ) ),
                       tr( "Toggle debug view message" ),
                       QKeySequence(),
                       this );
    M_show_debug_view_message_act->setStatusTip( tr( "Show/Hide debug view message" ) );
    M_show_debug_view_message_act->setToggleAction( true );
    M_show_debug_view_message_act->setOn( true );
    connect( M_show_debug_view_message_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleShowDebugViewMessage( bool ) ) );

    //////////////////////////////////////////////
    // dir selection
    M_open_debug_log_dir_act
        = new QAction( QIconSet( QPixmap( open_dir_xpm ) ),
                       tr( "Open log dir..." ),
                       QKeySequence(),
                       this );
    M_open_debug_log_dir_act->setStatusTip( tr( "Open the debug log directory" ) );
    connect( M_open_debug_log_dir_act, SIGNAL( activated() ),
             this, SLOT( showDebugLogDirDialog() ) );

    //////////////////////////////////////////////
    // cycle control
    M_sync_act
        = new QAction( QIconSet( QPixmap( sync_xpm ) ),
                       tr( "Sync" ),
                       QKeySequence(),
                       this );
    M_sync_act->setStatusTip( tr( "Synchronize with field canvas" ) );
    connect( M_sync_act, SIGNAL( activated() ),
             this, SLOT( syncCycle() ) );

    M_decrement_act
        = new QAction( QIconSet( QPixmap( logplayer_one_step_back_xpm ) ),
                       tr( "Decrement" ),
                       QKeySequence(),
                       this );
    M_decrement_act->setStatusTip( tr( "Decrement message cycle" ) );
    connect( M_decrement_act, SIGNAL( activated() ),
             this, SLOT( decrementCycle() ) );

    M_increment_act
        = new QAction( QIconSet( QPixmap( logplayer_one_step_forward_xpm ) ),
                       tr( "Increment" ),
                       QKeySequence(),
                       this );
    M_increment_act->setStatusTip( tr( "Increment message cycle" ) );
    connect( M_increment_act, SIGNAL( activated() ),
             this, SLOT( incrementCycle() ) );

    //////////////////////////////////////////////
    // debug level switches
    int i = 0;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_1_xpm ) ),
                                        tr( "Level 1 System" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 1 System" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_1( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_2_xpm ) ),
                                        tr( "Level 2 Sensor" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 2 Sensor" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_2( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_3_xpm ) ),
                                        tr( "Level 3 WorldModel" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 3 WorldModel" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_3( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_4_xpm ) ),
                                        tr( "Level 4 BasicAction" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 4 BasicAction" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_4( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_5_xpm ) ),
                                        tr( "Level 5 Intercept" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 5 Intercept" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_5( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_6_xpm ) ),
                                        tr( "Level 6 Kick" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 6 Kick" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_6( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_7_xpm ) ),
                                        tr( "Level 7 Dribble" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 7 Dribble" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_7( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_8_xpm ) ),
                                        tr( "Level 8 Pass" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 8 Pass" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_8( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_9_xpm ) ),
                                        tr( "Level 9 Cross" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 9 Cross" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_9( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_10_xpm ) ),
                                        tr( "Level 10 Shoot" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 10 Shoot" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_10( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_11_xpm ) ),
                                        tr( "Level 11 Clear" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 11 Clear" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_11( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_12_xpm ) ),
                                        tr( "Level 12 Team" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 12 Team" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_12( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_13_xpm ) ),
                                        tr( "Level 13 Role" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 13 Role" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_13( bool ) ) );
    ++i;

    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_14_xpm ) ),
                                        tr( "Level 14" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 14" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_14( bool ) ) );
    ++i;


    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_15_xpm ) ),
                                        tr( "Level 15" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 15" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_15( bool ) ) );
    ++i;


    M_debug_level_act[i] = new QAction( QIconSet( QPixmap( num_16_xpm ) ),
                                        tr( "Level 16" ),
                                        QKeySequence(),
                                        this );
    M_debug_level_act[i]->setStatusTip( tr( "Debug Level 16" ) );
    M_debug_level_act[i]->setToggleAction( true );
    connect( M_debug_level_act[i], SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDebugLevel_16( bool ) ) );
    ++i;
}


/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::createWindows()
{
    createToolBar();

    M_tab_widget = new QTabWidget( this );

    connect( M_tab_widget, SIGNAL( currentChanged( QWidget * ) ),
             this, SLOT( changeCurrentTab( QWidget * ) ) );

    this->setCentralWidget( M_tab_widget );

    for ( int i = 0; i < 11; ++i )
    {
        M_message[i] = new QTextEdit( M_tab_widget );
        M_message[i]->setReadOnly( true );
        M_message[i]->setCaption( QString( "   %1 " ).arg( i + 1, 2 ) );
        M_tab_widget->addTab( M_message[i],
                              M_message[i]->caption() );
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::createToolBar()
{
    {
        QToolBar * tbar = new QToolBar();

        M_open_debug_log_dir_act->addTo( tbar );

        tbar->addSeparator();

        M_sync_act->addTo( tbar );
        M_decrement_act->addTo( tbar );
        M_increment_act->addTo( tbar );

        tbar->addSeparator();

        QLabel * find_label = new QLabel( tr( "Find: " ), tbar );
        M_find_box = new QLineEdit( tbar );
        connect( M_find_box, SIGNAL( returnPressed() ),
                 this, SLOT( findExistString() ) );
        connect( M_find_box, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( findString( const QString & ) ) );
        //QToolTip::add( M_find_box, tr( "Find the matched string from the below window" ) );

        QHButtonGroup * btn_group = new QHButtonGroup( tbar );
        btn_group->setFlat( true );
        btn_group->setInsideMargin( 0 );
        btn_group->setInsideSpacing( 0 );
        M_find_forward_rb = new QRadioButton( tr( "Down" ), btn_group );
        connect( M_find_forward_rb, SIGNAL( clicked() ),
                 this, SLOT( findExistString() ) );
        M_find_backward_rb = new QRadioButton( tr( "Up" ), btn_group );
        connect( M_find_backward_rb, SIGNAL( clicked() ),
                 this, SLOT( findExistString() ) );
        M_find_forward_rb->setChecked( true );

        this->addDockWindow( tbar, Qt::DockTop );
    }

    {
        QToolBar * tbar = new QToolBar();

        M_show_debug_view_all_act->addTo( tbar );
        M_show_debug_view_self_act->addTo( tbar );
        M_show_debug_view_ball_act->addTo( tbar );
        M_show_debug_view_players_act->addTo( tbar );
        M_show_debug_view_comment_act->addTo( tbar );
        M_show_debug_view_figure_act->addTo( tbar );
        M_show_debug_view_target_act->addTo( tbar );
        M_show_debug_view_message_act->addTo( tbar );

        tbar->addSeparator();

        for ( int i = 0; i < 16; ++i )
        {
            M_debug_level_act[i]->addTo( tbar );
        }

        this->addDockWindow( tbar, Qt::DockBottom );
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::loadDebugLogDir( const rcsc::SideID side,
                                    const std::string & dir_path )
{
    MonitorViewConstPtr ptr = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! ptr )
    {
        std::cerr << "No valid data!" << std::endl;
        return;
    }

    if ( ptr->leftTeam().name().empty() )
    {
        std::cerr << "Empty team name!" << std::endl;
        return;
    }

    if ( side == rcsc::LEFT )
    {
        if ( M_debug_holder->setDir( ptr->leftTeam().name(),
                                     M_tab_widget->currentPageIndex() + 1,
                                     dir_path ) )
        {
            syncCycle();
        }
    }
    else if ( side == rcsc::RIGHT )
    {
        if ( M_debug_holder->setDir( ptr->rightTeam().name(),
                                     M_tab_widget->currentPageIndex() + 1,
                                     dir_path ) )
        {
            syncCycle();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::changeCurrentTab( QWidget * )
{
    //std::cerr << "DebugMessageWindow  changed tab index to ["
    //          << widget->caption().ascii() << "]"
    //          << std::endl;
    syncCycle();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewAll( bool checked )
{
    if ( M_main_data.viewConfig().isShownDebugView() != checked )
    {
        M_main_data.getViewConfig().toggleShowDebugView();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewSelf( bool checked )
{
    if ( M_main_data.viewConfig().isShownDebugViewSelf() != checked )
    {
        M_main_data.getViewConfig().toggleShowDebugViewSelf();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewBall( bool checked )
{
    if ( M_main_data.viewConfig().isShownDebugViewBall() != checked )
    {
        M_main_data.getViewConfig().toggleShowDebugViewBall();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewPlayers( bool checked )
{
    if ( M_main_data.viewConfig().isShownDebugViewPlayers() != checked )
    {
        M_main_data.getViewConfig().toggleShowDebugViewPlayers();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewComment( bool checked )
{
    if ( M_main_data.viewConfig().isShownDebugViewComment() != checked )
    {
        M_main_data.getViewConfig().toggleShowDebugViewComment();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewFigure( bool checked )
{
    if ( M_main_data.viewConfig().isShownDebugViewFigure() != checked )
    {
        M_main_data.getViewConfig().toggleShowDebugViewFigure();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewTarget( bool checked )
{
    if ( M_main_data.viewConfig().isShownDebugViewTarget() != checked )
    {
        M_main_data.getViewConfig().toggleShowDebugViewTarget();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleShowDebugViewMessage( bool checked )
{
    if ( M_main_data.viewConfig().isShownDebugViewMessage() != checked )
    {
        M_main_data.getViewConfig().toggleShowDebugViewMessage();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::showDebugLogDirDialog()
{
    DebugLogDirDialog dlg( this );

    if ( ! dlg.exec() )
    {
        //std::cerr << "DebugLogDirDialog is not ok." << std::endl;
        return;
    }

    rcsc::SideID side = dlg.side();
    QString dir_path = dlg.dirPath();

    std::string dir_path_std = dlg.dirPath().ascii();
    std::cerr << "Select side = "
              << ( side == rcsc::LEFT ? "left" : "right" )
              << "  debug log dir = [" << dir_path_std << "]"
              << std::endl;

    AppConfig::instance().setDebugLogDir( dir_path_std );

    loadDebugLogDir( side, dir_path_std );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::syncCycle()
{
    MonitorViewConstPtr ptr = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! ptr )
    {
        //std::cerr << "sync debug cycle. no view data." << std::endl;
        return;
    }

    //std::cerr << "sync debug data cycle" << std::endl;
    // seek current cycle data
    if ( ! M_debug_holder->seekData( M_tab_widget->currentPageIndex() + 1,
                                     ptr->cycle() ) )
    {
        std::cerr << "No data! number = "
                  << M_tab_widget->currentPageIndex() + 1
                  << std::endl;
        return;
    }

    updateMessage();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::decrementCycle()
{
    if ( ! M_debug_holder->decDataCycle( M_tab_widget->currentPageIndex() + 1 ) )
    {
        return;
    }

    updateMessage();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::incrementCycle()
{
    if ( ! M_debug_holder->incDataCycle( M_tab_widget->currentPageIndex() + 1 ) )
    {
        //std::cerr << "Failed increment debug log" << std::endl;
        return;
    }

    updateMessage();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::findExistString()
{
    QString expr = M_find_box->text();
    if ( ! expr.isEmpty() )
    {
        findString( expr );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::findString( const QString & expr )
{
    M_message[ M_tab_widget->currentPageIndex() ]
        ->find( expr,
                false, // case insentive
                false, // no whole word
                M_find_forward_rb->isOn() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_1( bool checked )
{
    //M_level ^= DebugLogData::LEVEL_1; // reverse only 1 bit

    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_1;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_1;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_2( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_2;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_2;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_3( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_3;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_3;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_4( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_4;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_4;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_5( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_5;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_5;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_6( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_6;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_6;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_7( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_7;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_7;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_8( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_8;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_8;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_9( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_9;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_9;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_10( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_10;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_10;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_11( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_11;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_11;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_12( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_12;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_12;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_13( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_13;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_13;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_14( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_14;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_14;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_15( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_15;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_15;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::toggleDebugLevel_16( bool checked )
{
    if ( checked )
    {
        M_level |= DebugLogData::LEVEL_16;
    }
    else
    {
        M_level &= ~DebugLogData::LEVEL_16;
    }

    updateMessage();
    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::clearAll()
{
    M_debug_holder = boost::shared_ptr< DebugLogHolder >( new DebugLogHolder );

    for ( int i = 0; i < 11; ++i )
    {
        M_message[i ]->clear();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugMessageWindow::updateMessage()
{
    //std::cerr << "DebuMessageWindow::updateMessage()" << std::endl;

    const DebugLogHolder::DataList * data_list
        = M_debug_holder->getDataList( M_tab_widget->currentPageIndex() + 1 );
    if ( ! data_list )
    {
        std::cerr << "DebuMessageWindow::updateMessage() no data list" << std::endl;
        return;
    }

    if ( M_level == 0 )
    {
        std::cerr << "DebuMessageWindow::updateMessage() no debug level" << std::endl;
        return;
    }

    const long cur_cycle = M_debug_holder->getCurrentCycle();

    QTextEdit * text_edit = M_message[ M_tab_widget->currentPageIndex() ];

    // clear current data
    text_edit->clear();

    QString main_buffer;
    main_buffer.reserve( 1000 * 256 );
    int n_line = 0;

    int show_para = 0;
    int show_index = 0;
    const DebugLogHolder::DataList::const_iterator data_end = data_list->end();
    for ( DebugLogHolder::DataList::const_iterator data_it = data_list->begin();
          data_it != data_end;
          ++data_it )
    {
        if ( ! (*data_it) )
        {
            std::cerr << __FILE__ << ":" << __LINE__
                      << " Null data found!" << std::endl;
            continue;
        }

        if ( (*data_it)->cycle() <= cur_cycle )
        {
            //text_edit->getCursorPosition( &show_para, &show_index );
            show_para = text_edit->paragraphs() - 1;
        }

        QString cycle_line( "CYCLE " );
        cycle_line += QString::number( (*data_it)->cycle() );
        cycle_line += tr( " ------------------------------\n" );
        text_edit->append( cycle_line );

        const DebugLogData::TextList::const_iterator text_end = (*data_it)->textList().end();
        for ( DebugLogData::TextList::const_iterator text_it = (*data_it)->textList().begin();
              text_it != text_end;
              ++text_it )
        {
            // level check
            if ( M_level & text_it->first )
            {
                //main_buffer += QString::fromStdString( text_it->second );
                main_buffer += QString::fromAscii( text_it->second.c_str() );
                if ( ++n_line > 1000 )
                {
                    // append message to text control
                    text_edit->append( main_buffer );
                    main_buffer.remove( 0, main_buffer.length() );
                    n_line = 0;
                }
            }
        }

        if ( ! main_buffer.isEmpty() )
        {
            text_edit->append( main_buffer );
            main_buffer.remove( 0, main_buffer.length() );
        }
    }

    text_edit->setCursorPosition( show_para, 0 );
}
