// -*-c++-*-

/*!
  \file view_config_dialog.cpp
  \brief View Config Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "view_config_dialog.h"

#include "main_data.h"
#include "view_config.h"

#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::ViewConfigDialog( QWidget * parent,
                                    const MainData & main_data,
                                    ViewConfig & view_config )


    : QDialog( parent )
    , M_main_data( main_data )
    , M_view_config( view_config )
{
    this->setCaption( tr( "View Preference" ) );

    //QFont font = this->font();
    //font.setPointSize( 8 );
    //this->setFont( font );

    createWidgets();
}

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::~ViewConfigDialog()
{
    //std::cerr << "delete ViewConfigDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::createWidgets()
{
    QVBoxLayout * layout = new QVBoxLayout( this );
    layout->setMargin( 4 );
    //layout->setSpacing( 0 );

    // zoom options
    layout->addWidget( createZoomControls(),
                       0, Qt::AlignLeft );
    // canvas size
    layout->addWidget( createCanvasSizeControls(),
                       0, Qt::AlignLeft );
    // detail info options
    layout->addWidget( createPlayersDetailControls(),
                       0, Qt::AlignLeft );
    // show / hide info
    layout->addWidget( createShowControls(),
                       0, Qt::AlignLeft );
    // grass type
    layout->addWidget( createGrassTypeControls(),
                       0, Qt::AlignLeft );
    // show / hide computational geometry
    layout->addWidget( createCompGeomControls(),
                       0, Qt::AlignLeft );
    // focus selector
    layout->addWidget( createFocusControls(),
                       0, Qt::AlignLeft );
    // player selector
    layout->addWidget( createPlayerSelectionControls(),
                       0, Qt::AlignLeft );
    // trace cycle span
    layout->addWidget( createTraceControls(),
                       0, Qt::AlignLeft );
    // future cycle info
    layout->addWidget( createInertiaMoveControls(),
                       0, Qt::AlignLeft );

    //this->setLayout( layout ); // qt4
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createZoomControls()
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Zoom" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QVBoxLayout * top_layout = new QVBoxLayout( group_box );
    //top_layout->setMargin( 1 );
    //top_layout->setSpacing( 0 );

    const QFontMetrics metrics = this->fontMetrics();
    const int button_height = metrics.height() + 12;

    {
        QHBox * box = new QHBox( group_box );
        //box->setLineWidth( 0 );
        //QHBoxLayout * layout = new QHBoxLayout( top_layout );
        //layout->setSpacing( 0 );

        QPushButton * zoomin_button = new QPushButton( tr( "+" ), box );
        zoomin_button->setMaximumSize( 48, button_height );
        connect( zoomin_button, SIGNAL( clicked() ),
                 this, SLOT( zoomIn() ) );
        //layout->addWidget( zoomin_button );

        QPushButton * zoomout_button = new QPushButton( tr( "-" ), box );
        zoomout_button->setMaximumSize( 48, button_height );
        connect( zoomout_button, SIGNAL( clicked() ),
                 this, SLOT( zoomOut() ) );
        //layout->addWidget( zoomout_button );

        M_enlarge_cb = new QCheckBox( tr( "Enlarge Objects" ), box );
        M_enlarge_cb->setChecked( M_view_config.isEnlarged() );
        connect( M_enlarge_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickEnlarge( bool ) ) );
        //layout->addWidget( M_enlarge_cb );

        QPushButton * fit_button = new QPushButton( tr( "Fit" ), box );
        fit_button->setMaximumSize( 60, button_height );
        connect( fit_button, SIGNAL( clicked() ),
                 this, SLOT( unzoom() ) );
        //layout->addWidget( fit_button );

        //top_layout->addLayout( layout );
    }

    {
        QHBox * box = new QHBox( group_box );
        //QHBoxLayout * layout = new QHBoxLayout( top_layout );
        //layout->setSpacing( 0 );

        new QLabel( tr( "Scale: " ), box );
        //layout->addWidget( new QLabel( tr( "Scale:" ), box ) );
        //layout->addSpacing( 2 );

        M_scale_slider = new QSlider( Qt::Horizontal, box );
        M_scale_slider->setRange( static_cast< int >( ViewConfig::MIN_FIELD_SCALE
                                                      * 10.0 ),
                                  static_cast< int >( ViewConfig::MAX_FIELD_SCALE
                                                      * 10.0 ) );
        M_scale_slider->setValue( 80 );
        M_scale_slider->setSteps( 1, 1 );
        connect( M_scale_slider, SIGNAL( sliderMoved( int ) ),
                 this, SLOT( slideFieldScale( int ) ) );
        //layout->addWidget( M_scale_slider );

        //layout->addSpacing( 2 );

        M_scale_text = new QLabel( tr( " 000.0" ), box );
        //layout->addWidget( M_scale_text );

        //top_layout->addLayout( layout );
    }

    //group_box->setLayout( top_layout ); // qt4
    return group_box;;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createCanvasSizeControls()
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Canvas Size" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );
    //layout->setMargin( 1 );
    //layout->setSpacing( 0 );

    //layout->addWidget( new QLabel( tr( " Width:" ), group_box ) );
    new QLabel( tr( " Width:" ), group_box );

    M_canvas_width_text = new QLineEdit( tr( "640" ), group_box );
    M_canvas_width_text->setMaximumSize( 48, 24 );
    //layout->addWidget( M_canvas_width_text );

    //layout->addWidget( new QLabel( tr( " Height:" ), group_box ) );
    new QLabel( tr( " Height:" ), group_box );

    M_canvas_height_text = new QLineEdit( tr( "480" ), group_box );
    M_canvas_height_text->setMaximumSize( 48, 24 );
    //layout->addWidget( M_canvas_height_text );

    //layout->addSpacing( 12 );

    QPushButton * apply_canvas_size_btn = new QPushButton( tr( "apply" ), group_box );
    apply_canvas_size_btn->setMaximumSize( 60, this->fontMetrics().height() + 12 );
    connect( apply_canvas_size_btn, SIGNAL( clicked() ),
             this, SLOT( applyCanvasSize() ) );
    //layout->addWidget( apply_canvas_size_btn );

    //group_box->setLayout( layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createPlayersDetailControls()
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Players\' Detail" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );
    //layout->setMargin( 1 );
    //layout->setSpacing( 0 );

    {
        M_player_number_cb = new QCheckBox( tr( "Unum" ), group_box );
        M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );
        connect( M_player_number_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowPlayerNumber( bool ) ) );
        //layout->addWidget( M_player_number_cb );
    }
    {
        M_hetero_number_cb = new QCheckBox( tr( "Type" ), group_box );
        M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );
        connect( M_hetero_number_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowHeteroNumber( bool ) ) );
        //layout->addWidget( M_hetero_number_cb );
    }
    {
        M_stamina_cb = new QCheckBox( tr( "Stamina" ), group_box );
        M_stamina_cb->setChecked( M_view_config.isShownStamina() );
        connect( M_stamina_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowStamina( bool ) ) );
        //layout->addWidget( M_stamina_cb );
    }
    {
        M_view_cone_cb = new QCheckBox( tr( "View Cone" ), group_box );
        M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );
        connect( M_view_cone_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowViewCone( bool ) ) );
        //layout->addWidget( M_view_cone_cb );
    }
    {
        M_control_area_cb = new QCheckBox( tr( "Control Area" ), group_box );
        M_control_area_cb->setChecked( M_view_config.isShownControlArea() );
        connect( M_control_area_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowControlArea( bool ) ) );
        //layout->addWidget( M_control_area_cb );
    }

    //group_box->setLayout( layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createShowControls()
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Show" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );
    //layout->setMargin( 1 );
    //layout->setSpacing( 0 );

    {
        M_show_score_board_cb = new QCheckBox( tr( "Score Board" ), group_box );
        M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );
        connect( M_show_score_board_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowScoreBoard( bool ) ) );
        //layout->addWidget( M_show_score_board_cb );
    }
    {
        M_show_ball_cb = new QCheckBox( tr( "Ball" ), group_box );
        M_show_ball_cb->setChecked( M_view_config.isShownBall() );
        connect( M_show_ball_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowBall( bool ) ) );
        //layout->addWidget( M_show_ball_cb );
    }
    {
        M_show_players_cb = new QCheckBox( tr( "Player" ), group_box );
        M_show_players_cb->setChecked( M_view_config.isShownPlayers() );
        connect( M_show_players_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowPlayers( bool ) ) );
        //layout->addWidget( M_show_players_cb );
    }
    {
        M_show_flags_cb = new QCheckBox( tr( "Flag" ), group_box );
        M_show_flags_cb->setChecked( M_view_config.isShownFlags() );
        connect( M_show_flags_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowFlags( bool ) ) );
        //layout->addWidget( M_show_flags_cb );
    }
    {
        M_show_offside_line_cb = new QCheckBox( tr( "Offside Line" ), group_box );
        M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );
        connect( M_show_offside_line_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowOffsideLine( bool ) ) );
        //layout->addWidget( M_show_offside_line_cb );
    }

    //group_box->setLayout( layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createCompGeomControls()
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Compututational Geometry" ),
                                             this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );
    //layout->setMargin( 1 );
    //layout->setSpacing( 0 );

    {
        M_show_voronoi_cb = new QCheckBox( tr( "VoronoiDiagram" ), group_box );
        M_show_voronoi_cb->setChecked( M_view_config.isShownVoronoiDiagram() );
        connect( M_show_voronoi_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowVoronoiDiagram( bool ) ) );
        //layout->addWidget( M_show_voronoi_cb );
    }
    {
        M_show_delaunay_cb = new QCheckBox( tr( "DelaunayTrianglation" ), group_box );
        M_show_delaunay_cb->setChecked( M_view_config.isShownDelaunayTrianglation() );
        connect( M_show_delaunay_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowDelaunayTrianglation( bool ) ) );
        //layout->addWidget( M_show_delaunay_cb );
    }
    {
        M_voronoi_choice = new QComboBox( group_box );
        M_voronoi_choice->insertItem( tr( "All" ) );
        M_voronoi_choice->insertItem( tr( "Left Team" ) );
        M_voronoi_choice->insertItem( tr( "Right Team" ) );
        M_voronoi_choice->setMaximumSize( this->fontMetrics().width( tr( "Right Team" ) ) + 32,
                                          this->fontMetrics().height() + 12 );
        M_voronoi_choice->setCurrentItem( 0 );
        connect( M_voronoi_choice, SIGNAL( activated( int ) ),
                 this, SLOT( selectCompGeomSide( int ) ) );
        //layout->addWidget( M_voronoi_choice );
    }

    //group_box->setLayout( layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createFocusControls()
{
    QGroupBox * group_box = new QHButtonGroup( tr( "Focus" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );
    //layout->setMargin( 1 );
    //layout->setSpacing( 0 );

    {
        M_focus_ball_rb = new QRadioButton( tr( "Ball" ), group_box );
        connect( M_focus_ball_rb, SIGNAL( clicked() ),
                 this, SLOT( clickFocusBall() ) );
        //layout->addWidget( M_focus_ball_rb );
    }
    {
        M_focus_player_rb = new QRadioButton( tr( "Selected Player" ),
                                              group_box );
        connect( M_focus_player_rb, SIGNAL( clicked() ),
                 this, SLOT( clickFocusPlayer() ) );
        //layout->addWidget( M_focus_player_rb );
    }
    {
        M_focus_fix_rb = new QRadioButton( tr( "Fix" ), group_box );
        connect( M_focus_fix_rb, SIGNAL( clicked() ),
                 this, SLOT( setFocusFix() ) );
        //layout->addWidget( M_focus_fix_rb );
    }
    M_focus_fix_rb->setChecked( true );

    //group_box->setLayout( layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createPlayerSelectionControls()
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Player Selection" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QVBoxLayout * top_layout = new QVBoxLayout( group_box );
    //top_layout->setMargin( 1 );
    //top_layout->setSpacing( 0 );

    // selection type
    {
        QGroupBox * box = new QHButtonGroup( group_box );
        box->setFlat( true );
        box->setInsideMargin( 0 );
        box->setInsideSpacing( 0 );
        //QHBoxLayout * layout = new QHBoxLayout();
        //layout->setMargin( 0 );
        //layout->setSpacing( 0 );
        {
            M_select_all_rb = new QRadioButton( tr( "Auto" ), box );
            connect( M_select_all_rb, SIGNAL( clicked() ),
                     this, SLOT( clickSelectAutoAll() ) );
            //layout->addWidget( M_select_all_rb );
        }
        {
            M_select_left_rb = new QRadioButton( tr( "Auto left" ), box );
            connect( M_select_left_rb, SIGNAL( clicked() ),
                     this, SLOT( clickSelectAutoLeft() ) );
            //layout->addWidget( M_select_left_rb );
        }
        {
            M_select_right_rb = new QRadioButton( tr( "Auto right" ), box );
            connect( M_select_right_rb, SIGNAL( clicked() ),
                     this, SLOT( clickSelectAutoRight() ) );
            //layout->addWidget( M_select_right_rb );
        }
        {
            M_select_fix_rb = new QRadioButton( tr( "Fix" ), box );
            connect( M_select_all_rb, SIGNAL( clicked() ),
                     this, SLOT( clickSelectFix() ) );
            //layout->addWidget( M_select_fix_rb );
        }
        {
            M_unselect_rb = new QRadioButton( tr( "Unselect" ), box );
            connect( M_unselect_rb, SIGNAL( clicked() ),
                     this, SLOT( setUnselect() ) );
            //layout->addWidget( M_unselect_rb );
        }
        M_unselect_rb->setChecked( true );

        //top_layout->addLayout( layout );
    }

    {
        QFrame * box = new QFrame( group_box );
        QHBoxLayout * layout = new QHBoxLayout( box );
        //layout->setMargin( 0 );
        //layout->setSpacing( 0 );
        layout->addSpacing( 16 );

        M_player_choice = new QComboBox( box );
        M_player_choice->insertItem( tr ( "No Selection" ) );
        for ( int i = 1; i <= 11; ++i )
        {
            M_player_choice->insertItem( QString( "Left number: %1" ).arg( i ) );
        }
        for ( int i = 1; i <= 11; ++i )
        {
            M_player_choice->insertItem( QString( "Right number: %1" ).arg( i ) );
        }
        M_player_choice->setCurrentItem( 0 );
        int id_width = this->fontMetrics().width( tr( "Right Number: 11" ) );
        M_player_choice->setMaximumWidth( id_width + 40 );
        connect( M_player_choice, SIGNAL( activated( int ) ),
                 this, SLOT( selectPlayer( int ) ) );
        layout->addWidget( M_player_choice, 0, Qt::AlignLeft );

        //layout->addSpacing( 16 );

        //top_layout->addLayout( layout );
    }

    //group_box->setLayout( top_layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createGrassTypeControls()
{
    QGroupBox * group_box = new QHButtonGroup( tr( "Grass Type" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QHBoxLayout * layout = new QHBoxLayout( group_box );
    //layout->setMargin( 1 );
    //layout->setSpacing( 0 );

    {
        M_keepaway_mode_cb = new QCheckBox( tr( "Keepaway" ), group_box );
        M_keepaway_mode_cb->setChecked( M_view_config.keepawayMode() );
        connect( M_keepaway_mode_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickKeepawayMode( bool ) ) );
        //layout->addWidget( M_keepaway_mode_cb );
    }
    {
        M_grass_normal_rb = new QRadioButton( tr( "mono" ), group_box );
        connect( M_grass_normal_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassNormal() ) );
        //layout->addWidget( M_grass_normal_rb );
    }
    {
        M_grass_line_rb = new QRadioButton( tr( "lines" ), group_box );
        connect( M_grass_line_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassLine() ) );
        //layout->addWidget( M_grass_line_rb );
    }
    {
        M_grass_checker_rb = new QRadioButton( tr( "checker" ), group_box );
        connect( M_grass_checker_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassChecker() ) );
        //layout->addWidget( M_grass_checker_rb );
    }

    switch ( M_view_config.grassType() ) {
    case ViewConfig::GRASS_NORMAL:
        M_grass_normal_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_LINES:
        M_grass_line_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_CHECKER:
        M_grass_checker_rb->setChecked( true );
        break;
    default:
        M_grass_normal_rb->setChecked( true );
        break;
    }

    //group_box->setLayout( layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createTraceControls()
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Trace" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QVBoxLayout * top_layout = new QVBoxLayout( group_box );
    //top_layout->setMargin( 0 );
    //top_layout->setSpacing( 0 );

    //QHBoxLayout * first_layout = new QHBoxLayout();
    //first_layout->setMargin( 0 );
    //first_layout->setSpacing( 0 );
    QHBox * top_box = new QHBox( group_box );
    {
        QHGroupBox * ball_box = new QHGroupBox( tr( "Ball" ), top_box );
        ball_box->setInsideMargin( 5 );
        ball_box->setInsideSpacing( 2 );

        //QHBoxLayout * layout = new QHBoxLayout( ball_box );
        //layout->setMargin( 0 );
        //layout->setSpacing( 0 );

        M_ball_trace_start = new QLineEdit( tr( "0" ), ball_box );
        M_ball_trace_start->setMaximumSize( 36, 24 );
        connect( M_ball_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceStart( const QString & ) ) );
        //layout->addWidget( M_ball_trace_start, 0, Qt::AlignVCenter );

        //layout->addWidget( new QLabel( tr( "-" ), ball_box ),
        //0, Qt::AlignVCenter );
        new QLabel( tr( "-" ), ball_box );

        M_ball_trace_end = new QLineEdit( tr( "0" ), ball_box );
        M_ball_trace_end->setMaximumSize( 36, 24 );
        connect( M_ball_trace_end, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceEnd( const QString & ) ) );
        //layout->addWidget( M_ball_trace_end, 0, Qt::AlignVCenter );

        QPushButton * all_btn = new QPushButton( tr( "All" ), ball_box );
        all_btn->setMaximumSize( 48, this->fontMetrics().height() + 12 );
        all_btn->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        connect( all_btn, SIGNAL( clicked() ),
                 this, SLOT( clickBallTraceAll() ) );
        //layout->addWidget( all_btn, 0, Qt::AlignVCenter );

        M_ball_trace_cb = new QCheckBox( tr( "Auto" ), ball_box );
        M_ball_trace_cb->setChecked( false );
        connect( M_ball_trace_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickAutoBallTrace( bool ) ) );
        //layout->addWidget( M_ball_trace_cb, 0, Qt::AlignVCenter );

        //ball_box->setLayout( layout ); //qt4
        //first_layout->addWidget( ball_box );
    }
    //first_layout->addSpacing( 2 );
    {
        QHGroupBox * player_box = new QHGroupBox( tr( "Player" ), top_box );
        player_box->setInsideMargin( 5 );
        player_box->setInsideSpacing( 2 );

        //QHBoxLayout * layout = new QHBoxLayout( player_box );
        //layout->setMargin( 0 );
        //layout->setSpacing( 0 );

        M_player_trace_start = new QLineEdit( tr( "0" ), player_box );
        M_player_trace_start->setMaximumSize( 36, 24 );
        connect( M_player_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editPlayerTraceStart( const QString & ) ) );
        //layout->addWidget( M_player_trace_start, 0, Qt::AlignVCenter );

        //layout->addWidget( new QLabel( tr( "-" ), player_box ),
        //                   0, Qt::AlignVCenter );
        new QLabel( tr( "-" ), player_box );

        M_player_trace_end = new QLineEdit( tr( "0" ), player_box );
        M_player_trace_end->setMaximumSize( 36, 24 );
        connect( M_ball_trace_end, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceEnd( const QString & ) ) );
        //layout->addWidget( M_player_trace_end, 0, Qt::AlignVCenter );

        QPushButton * all_btn = new QPushButton( tr( "All" ), player_box );
        all_btn->setMaximumSize( 48, this->fontMetrics().height() + 12 );
        all_btn->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        connect( all_btn, SIGNAL( clicked() ),
                 this, SLOT( clickPlayerTraceAll() ) );
        //layout->addWidget( all_btn, 0, Qt::AlignVCenter );

        M_player_trace_cb =  new QCheckBox( tr( "Auto" ), player_box );
        M_player_trace_cb->setChecked( false );
        connect( M_player_trace_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickAutoPlayerTrace( bool ) ) );
        //layout->addWidget( M_player_trace_cb, 0, Qt::AlignVCenter );

        //player_box->setLayout( layout ); //qt4
        //first_layout->addWidget( player_box );
    }

    //top_layout->addLayout( first_layout );
    //top_layout->addSpacing( 1 );

    QFrame * bottom_box = new QFrame( group_box );
    QHBoxLayout * bottom_layout = new QHBoxLayout( bottom_box );
    //bottom_layout->setMargin( 4 );
    //bottom_layout->setSpacing( 0 );
    {
        QHGroupBox * trace_box = new QHGroupBox( tr( "Auto Trace Period" ),
                                                 bottom_box );
        trace_box->setInsideMargin( 5 );
        trace_box->setInsideSpacing( 2 );

        //QHBoxLayout * layout = new QHBoxLayout( trace_box );
        //layout->setMargin( 0 );
        //layout->setSpacing( 0 );

        //layout->addWidget( new QLabel( tr( "Start:" ), trace_box ),
        //                   0, Qt::AlignVCenter );
        new QLabel( tr( "Start:" ), trace_box );

        //layout->addSpacing( 2 );

        M_auto_trace_start = new QLineEdit( tr( "-10" ), trace_box );
        M_auto_trace_start->setMaximumSize( 36, 24 );
        connect( M_auto_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editAutoTraceStart( const QString & ) ) );
        //layout->addWidget( M_auto_trace_start, 0, Qt::AlignVCenter );

        //layout->addSpacing( 6 );

        //layout->addWidget( new QLabel( tr( "Period:" ), trace_box ),
        //                   0, Qt::AlignVCenter );
        new QLabel( tr( "Period:" ), trace_box );

        //layout->addSpacing( 2 );

        M_auto_trace_period = new QSpinBox( trace_box );
        M_auto_trace_period->setValue( 10 );
        M_auto_trace_period->setRange( 1, 12000 );
        //M_auto_trace_period->setMaximumSize( 36, 24 );
        connect( M_auto_trace_period, SIGNAL( valueChanged( int ) ),
                 this, SLOT( changeAutoTracePeriod( int ) ) );
        //layout->addWidget( M_auto_trace_period, 0, Qt::AlignVCenter );

        //trace_box->setLayout( layout ); //qt4
        bottom_layout->addWidget( trace_box, 5 );
    }

    bottom_layout->addSpacing( 12 );
    {
        //QHBoxLayout * layout = new QHBoxLayout();
        //layout->setMargin( 0 );
        //layout->setSpacing( 0 );

        //layout->addStretch( 1 );

        QPushButton * line_point_btn = new QPushButton( tr( "Line/Point" ),
                                                        bottom_box );
        line_point_btn->setMaximumSize( 80, this->fontMetrics().height() + 12 );
        connect( line_point_btn, SIGNAL( clicked() ),
                 this, SLOT( clickLinePointButton() ) );
        bottom_layout->addWidget( line_point_btn, 4 );

        bottom_layout->addStretch( 1 );
    }

    //top_layout->addLayout( second_layout );

    //group_box->setLayout( top_layout ); // qt4
    return group_box;;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createInertiaMoveControls()
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Inertia Move" ), this );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    //QHBoxLayout * top_layout = new QHBoxLayout( group_box );
    //top_layout->setMargin( 0 );
    //top_layout->setSpacing( 0 );

    //top_layout->addWidget( new QLabel( tr( "Ball:" ), group_box ) );
    new QLabel( tr( "Ball:" ), group_box );

    //top_layout->addSpacing( 2 );

    M_ball_future = new QSpinBox( group_box );
    //M_ball_future->setMinimumSize( 60, 24 );
    M_ball_future->setValue( 0 );
    M_ball_future->setRange( 0, 50 );
    connect( M_ball_future, SIGNAL( valueChanged( int ) ),
             this, SLOT( changeBallFutureCycle( int ) ) );
    //top_layout->addWidget( M_ball_future );

    //top_layout->addSpacing( 6 );

    //top_layout->addWidget( new QLabel( tr( "Player:" ), group_box ) );
    new QLabel( tr( "Player:" ), group_box );

    //top_layout->addSpacing( 2 );

    M_player_future = new QSpinBox( group_box );
    //M_player_future->setMaximumSize( 60, 24 );
    M_player_future->setValue( 0 );
    M_player_future->setRange( 0, 10 );
    connect( M_player_future, SIGNAL( valueChanged( int ) ),
             this, SLOT( changePlayerFutureCycle( int ) ) );
    //top_layout->addWidget( M_player_future );

    //group_box->setLayout( top_layout ); // qt4
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::showEvent( QShowEvent * event )
{
    updateAll();
    //event->accept();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::updateAll()
{
    M_enlarge_cb->setChecked( M_view_config.isEnlarged() );

    updateFieldScale();

    M_canvas_width_text
        ->setText( QString::number( M_view_config.canvasWidth() ) );

    M_canvas_height_text
        ->setText( QString::number( M_view_config.canvasHeight() ) );

    M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );
    M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );
    M_stamina_cb->setChecked( M_view_config.isShownStamina() );
    M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );
    M_control_area_cb->setChecked( M_view_config.isShownControlArea() );

    M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );
    M_show_ball_cb->setChecked( M_view_config.isShownBall() );
    M_show_players_cb->setChecked( M_view_config.isShownPlayers() );
    M_show_flags_cb->setChecked( M_view_config.isShownFlags() );
    M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );

    switch ( M_view_config.grassType() ) {
    case ViewConfig::GRASS_NORMAL:
        M_grass_normal_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_LINES:
        M_grass_line_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_CHECKER:
        M_grass_checker_rb->setChecked( true );
        break;
    default:
        M_grass_normal_rb->setChecked( true );
        break;
    }

    M_show_voronoi_cb->setChecked( M_view_config.isShownVoronoiDiagram() );
    M_show_delaunay_cb->setChecked( M_view_config.isShownDelaunayTrianglation() );

    switch ( M_view_config.voronoiTarget() ) {
    case rcsc::NEUTRAL:
        M_voronoi_choice->setCurrentItem( 0 );
        break;
    case rcsc::LEFT:
        M_voronoi_choice->setCurrentItem( 1 );
        break;
    case rcsc::RIGHT:
        M_voronoi_choice->setCurrentItem( 2 );
        break;
    default:
        M_voronoi_choice->setCurrentItem( 0 );
        break;
    }

    switch( M_view_config.focusType() ) {
    case ViewConfig::FOCUS_BALL:
        M_focus_ball_rb->setChecked( true );
        break;
    case ViewConfig::FOCUS_PLAYER:
        M_focus_player_rb->setChecked( true );
        break;
    case ViewConfig::FOCUS_POINT:
        M_focus_fix_rb->setChecked( true );
        break;
    default:
        M_focus_fix_rb->setChecked( true );
        break;
    }

    switch ( M_view_config.playerSelectType() ) {
    case ViewConfig::SELECT_AUTO_ALL:
        M_select_all_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_AUTO_LEFT:
        M_select_left_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_AUTO_RIGHT:
        M_select_right_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_UNSELECT:
        M_unselect_rb->setChecked( true );
        break;
    default:
        M_unselect_rb->setChecked( true );
        break;
    }

    M_ball_trace_start->setText( QString::number( M_view_config.ballTraceStart() ) );
    M_ball_trace_end->setText( QString::number( M_view_config.ballTraceEnd() ) );
    M_ball_trace_cb->setChecked( M_view_config.isBallAutoTrace() );

    M_player_trace_start->setText( QString::number( M_view_config.playerTraceStart() ) );
    M_player_trace_end->setText( QString::number( M_view_config.playerTraceEnd() ) );
    M_player_trace_cb->setChecked( M_view_config.isPlayerAutoTrace() );

    M_auto_trace_start->setText( QString::number( M_view_config.autoTraceStart() ) );
    M_auto_trace_period->setValue( M_view_config.autoTracePeriod() );

    M_ball_future->setValue( M_view_config.ballFutureCycle() );
    M_player_future->setValue( M_view_config.playerFutureCycle() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::updateFieldScale()
{
    M_scale_slider->setValue( static_cast< int >
                              ( M_view_config.fieldScale() * 10.0 ) );
    char buf[16];
    std::snprintf( buf, 16, "%.1f", M_view_config.fieldScale() );
    M_scale_text->setText( QString::fromAscii( buf ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::zoomIn()
{
    M_view_config.zoomIn();

    updateFieldScale();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::zoomOut()
{
    M_view_config.zoomOut();

    updateFieldScale();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::unzoom()
{
    M_view_config.unzoom();
    updateFieldScale();
    M_focus_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickEnlarge( bool checked )
{
    if ( M_view_config.isEnlarged() != checked )
    {
        M_view_config.toggleEnlarge();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleEnlarge()
{
    M_view_config.toggleEnlarge();
    M_enlarge_cb->setChecked( M_view_config.isEnlarged() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::slideFieldScale( int value )
{
    M_scale_text->setText( QString::number( value * 0.1 ) );

    M_view_config.setFieldScale( value * 0.1 );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::applyCanvasSize()
{
    bool ok_w = true;
    bool ok_h = true;
    int width = M_canvas_width_text->text().toInt( & ok_w );
    int height = M_canvas_height_text->text().toInt( & ok_h );

    if ( ok_w
         && ok_h
         && width > 0
         && height > 0 )
    {
        emit canvasResized( QSize( width, height ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowPlayerNumber( bool checked )
{
    if ( M_view_config.isShownPlayerNumber() != checked )
    {
        M_view_config.toggleShowPlayerNumber();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowPlayerNumber()
{
    M_view_config.toggleShowPlayerNumber();
    M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowHeteroNumber( bool checked )
{
    if ( M_view_config.isShownHeteroNumber() != checked )
    {
        M_view_config.toggleShowHeteroNumber();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowHeteroNumber()
{
    M_view_config.toggleShowHeteroNumber();
    M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowStamina( bool checked )
{
    if ( M_view_config.isShownStamina() != checked )
    {
        M_view_config.toggleShowStamina();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowStamina()
{
    M_view_config.toggleShowStamina();
    M_stamina_cb->setChecked( M_view_config.isShownStamina() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowViewCone( bool checked )
{
    if ( M_view_config.isShownViewCone() != checked )
    {
        M_view_config.toggleShowViewCone();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowViewCone()
{
    M_view_config.toggleShowViewCone();
    M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowControlArea( bool checked )
{
    if ( M_view_config.isShownControlArea() != checked )
    {
        M_view_config.toggleShowControlArea();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowControlArea()
{
    M_view_config.toggleShowControlArea();
    M_control_area_cb->setChecked( M_view_config.isShownControlArea() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowScoreBoard( bool checked )
{
    if ( M_view_config.isShownScoreBoard() != checked )
    {
        M_view_config.toggleShowScoreBoard();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowScoreBoard()
{
    M_view_config.toggleShowScoreBoard();
    M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowBall( bool checked )
{
    if ( M_view_config.isShownBall() != checked )
    {
        M_view_config.toggleShowBall();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowBall()
{
    M_view_config.toggleShowBall();
    M_show_ball_cb->setChecked( M_view_config.isShownBall() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowPlayers( bool checked )
{
    if ( M_view_config.isShownPlayers() != checked )
    {
        M_view_config.toggleShowPlayers();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowPlayers()
{
    M_view_config.toggleShowPlayers();
    M_show_players_cb->setChecked( M_view_config.isShownPlayers() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowFlags( bool checked )
{
    if ( M_view_config.isShownFlags() != checked )
    {
        M_view_config.toggleShowFlags();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowFlags()
{
    M_view_config.toggleShowFlags();
    M_show_flags_cb->setChecked( M_view_config.isShownFlags() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowOffsideLine( bool checked )
{
    if ( M_view_config.isShownOffsideLine() != checked )
    {
        M_view_config.toggleShowOffsideLine();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowOffsideLine()
{
    M_view_config.toggleShowOffsideLine();
    M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassNormal()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_NORMAL )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_NORMAL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassLine()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_LINES )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_LINES );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassChecker()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_CHECKER )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_CHECKER );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickKeepawayMode( bool checked )
{
    if ( M_view_config.keepawayMode() != checked )
    {
        M_view_config.toggleKeepawayMode();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleKeepawayMode()
{
    M_view_config.toggleKeepawayMode();
    M_keepaway_mode_cb->setChecked( M_view_config.keepawayMode() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowVoronoiDiagram( bool checked )
{
    if ( M_view_config.isShownVoronoiDiagram() != checked )
    {
        M_view_config.toggleShowVoronoiDiagram();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowDelaunayTrianglation( bool checked )
{
    if ( M_view_config.isShownDelaunayTrianglation() != checked )
    {
        M_view_config.toggleShowDelaunayTrianglation();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectCompGeomSide( int index )
{
    switch ( index ) {
    case 0:
        if ( M_view_config.voronoiTarget() != rcsc::NEUTRAL )
        {
            M_view_config.setVoronoiTarget( rcsc::NEUTRAL );
            emit configured();
        }
        break;
    case 1:
        if ( M_view_config.voronoiTarget() != rcsc::LEFT )
        {
            M_view_config.setVoronoiTarget( rcsc::LEFT );
            emit configured();
        }
        break;
    case 2:
        if ( M_view_config.voronoiTarget() != rcsc::RIGHT )
        {
            M_view_config.setVoronoiTarget( rcsc::RIGHT );
            emit configured();
        }
        break;
    default:
        std::cerr << "ViewConfigDialog::selectCompGeomSide() unsupported index "
                  << index
                  << std::endl;
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusBall()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_BALL )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_BALL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleFocusBall()
{
    if ( M_view_config.focusType() == ViewConfig::FOCUS_BALL )
    {
        setFocusFix();
    }
    else
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_BALL );
        M_focus_ball_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusPlayer()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_PLAYER )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_PLAYER );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleFocusPlayer()
{
    if ( M_view_config.focusType() == ViewConfig::FOCUS_PLAYER )
    {
        setFocusFix();
    }
    else
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_PLAYER );
        M_focus_player_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusFix()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_POINT )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_POINT );
        M_focus_fix_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusPoint( const QPoint & point )
{
    M_view_config.setFocusPoint( point.x(), point.y() );
    M_focus_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoAll()
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_AUTO_ALL )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoAll()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_AUTO_ALL )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
        M_select_all_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoLeft()
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_AUTO_LEFT )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_LEFT );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoLeft()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_AUTO_LEFT )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_LEFT );
        M_select_left_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoRight()
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_AUTO_RIGHT )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoRight()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_AUTO_RIGHT )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        M_select_right_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectFix()
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_FIX )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectFix()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_FIX )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_FIX );
        M_select_fix_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setUnselect()
{
    M_view_config.unselectPlayer();
    M_unselect_rb->setChecked( true );
    M_player_choice->setCurrentItem( 0 );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectPlayer( int number )
{
    // left side:  number = [1,11]
    // right side:  number = [12,22]
    if ( number <= 0
         || 22 < number )
    {
        setUnselect();
        return;
    }

    M_view_config.setPlayerSelectType( ViewConfig::SELECT_FIX );
    if ( number <= 11 )
    {
        M_view_config.setSelectedNumber( rcsc::LEFT, number );
    }
    else
    {
        M_view_config.setSelectedNumber( rcsc::RIGHT, number - 11 );
    }

    M_player_choice->setCurrentItem( number );
    M_select_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        //std::cerr << "editBallTraceStart " << value << std::endl;
        M_view_config.setBallTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallTraceEnd( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        //std::cerr << "editBallTraceEnd " << value << std::endl;
        M_view_config.setBallTraceEnd( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickBallTraceAll()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();

    if ( ! view.empty() )
    {
        M_ball_trace_cb->setChecked( false );

        if ( M_view_config.ballTraceStart() == 0
             && M_view_config.ballTraceEnd() == 0 )
        {
            M_ball_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_ball_trace_end->setText( QString::number( view.back()->cycle() ) );

            M_view_config.setBallTraceStart( view.front()->cycle() );
            M_view_config.setBallTraceEnd( view.back()->cycle() );
        }
        else if ( M_view_config.ballTraceStart() == view.front()->cycle()
                  && M_view_config.ballTraceEnd() == view.back()->cycle() )
        {
            M_ball_trace_start->setText( QString::number( 0 ) );
            M_ball_trace_end->setText( QString::number( 0 ) );

            M_view_config.setBallTraceStart( 0 );
            M_view_config.setBallTraceEnd( 0 );
        }
        else
        {
            M_ball_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_ball_trace_end->setText( QString::number( view.back()->cycle() ) );

            M_view_config.setBallTraceStart( view.front()->cycle() );
            M_view_config.setBallTraceEnd( view.back()->cycle() );
        }

        if ( M_view_config.isBallAutoTrace() )
        {
            M_view_config.toggleBallAutoTrace();
        }

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAutoBallTrace( bool checked )
{
    if ( M_view_config.isBallAutoTrace() != checked )
    {
        M_view_config.toggleBallAutoTrace();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        std::cerr << "editPlayerTraceStart " << value << std::endl;
        M_view_config.setPlayerTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerTraceEnd( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        std::cerr << "editPlayerTraceEnd " << value << std::endl;
        M_view_config.setPlayerTraceEnd( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickPlayerTraceAll()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();
    if ( ! view.empty() )
    {
        M_player_trace_cb->setChecked( false );

        if ( M_view_config.playerTraceStart() == 0
             && M_view_config.playerTraceEnd() == 0 )
        {
            M_player_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_player_trace_end->setText( QString::number( view.back()->cycle() ) );

            M_view_config.setPlayerTraceStart( view.front()->cycle() );
            M_view_config.setPlayerTraceEnd( view.back()->cycle() );
        }
        else if ( M_view_config.playerTraceStart() == view.front()->cycle()
                  && M_view_config.playerTraceEnd() == view.back()->cycle() )
        {
            M_player_trace_start->setText( QString::number( 0 ) );
            M_player_trace_end->setText( QString::number( 0 ) );

            M_view_config.setPlayerTraceStart( 0 );
            M_view_config.setPlayerTraceEnd( 0 );
        }
        else
        {
            M_player_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_player_trace_end->setText( QString::number( view.back()->cycle() ) );

            M_view_config.setPlayerTraceStart( view.front()->cycle() );
            M_view_config.setPlayerTraceEnd( view.back()->cycle() );
        }

        if ( M_view_config.isPlayerAutoTrace() )
        {
            M_view_config.togglePlayerAutoTrace();
        }

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAutoPlayerTrace( bool checked )
{
    if ( M_view_config.isPlayerAutoTrace() != checked )
    {
        M_view_config.togglePlayerAutoTrace();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editAutoTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        //std::cerr << "editAutTraceStart " << value << std::endl;
        M_view_config.setAutoTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changeAutoTracePeriod( int value )
{
    //std::cerr << "changedAutoTracePeriod " << value << std::endl;
    if ( value >= 1 )
    {
        M_view_config.setAutoTracePeriod( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickLinePointButton()
{
    M_view_config.toggleLineTrace();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changeBallFutureCycle( int value )
{
    M_view_config.setBallFutureCycle( value );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changePlayerFutureCycle( int value )
{
    M_view_config.setPlayerFutureCycle( value );

    emit configured();
}
