// -*-c++-*-

/*!
  \file main_frame.cpp
  \brief main application frame class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/config.h>   // wxFileConfig
#include <wx/fileconf.h> // wxFileConfig
#include <wx/filename.h>
#endif

#include "main_frame.h"

#include "id.h"
#include "soccerwindow_app.h"

#include "debug_message_frame.h"
#include "debug_server.h"
#include "detail_dialog.h"
#include "field_canvas.h"
#include "image_save_dialog.h"
#include "log_player.h"
#include "log_player_tool_bar.h"
#include "monitor_client.h"
#include "player_type_dialog.h"
#include "view_config_dialog.h"
// model
#include "app_config.h"
#include "main_data.h"
#include "monitor_view_data.h"

#include <rcsc/geom/vector_2d.h>

#include <boost/signals.hpp>
#include <boost/bind.hpp>

#include <iostream>
#include <csignal> // ::kill()
#include <cstdlib> // system()
#include <sys/types.h> // SIGINT

#if defined(__WXGTK__) || defined(__WXX11__) || defined(__WXMOTIF__) || defined(__WXMAC__)
#  include "xpm/soccerwindow2-nostr.xpm"
#endif

/*-------------------------------------------------------------------*/
/*!
  constructing main frame.
*/
MainFrame::MainFrame( const wxString & title )
    : wxFrame( static_cast< wxWindow * >( 0 ), -1, title,
               wxDefaultPosition, //wxPoint( 64, 32 ),
               wxSize( 620, 480 ) )
    , M_log_player( new LogPlayer( M_main_data ) )
    , M_detail_dialog( static_cast< DetailDialog * >( 0 ) )
    , M_player_type_dialog( static_cast< PlayerTypeDialog * >( 0 ) )
    , M_debug_message( static_cast< DebugMessageFrame * >( 0 ) )
{
    //this->Center();

    // load options from config file
    loadConfig();

    createMenuBar();
    createToolBar();
    createStatusBar();
    // create internal windows
    createWindows();

    connectEvent();

    M_log_player->emit_updated.connect( emit_viewUpdated );

    //this->SetAutoLayout( true );

    // set minimal window size
    this->SetSizeHints( 280, 220 );
    // set loaded frame size (or keep default size)
    this->SetSize( AppConfig::instance().framePosX(),
                   AppConfig::instance().framePosY(),
                   AppConfig::instance().frameWidth(),
                   AppConfig::instance().frameHeight() );

    // apply initial window options.
    if ( AppConfig::instance().hideToolBar() )
    {
        if ( this->GetToolBar() )
        {
            this->GetToolBar()->Hide();
            this->Fit();
        }
    }

    if ( AppConfig::instance().hideStatusBar() )
    {
        if ( this->GetStatusBar() )
        {
            this->GetStatusBar()->Hide();
            this->Fit();
        }
    }

    if ( AppConfig::instance().maximize() )
    {
        this->Maximize( true );
    }

    if ( AppConfig::instance().fullScreen() )
    {
        this->ShowFullScreen( true );
    }

    this->SetIcon( wxICON( soccerwindow2_nostr ) );
}

/*-------------------------------------------------------------------*/
/*!
  destructing main frame.
*/
MainFrame::~MainFrame()
{
    if ( AppConfig::instance().killServer() )
    {
        killServer();
    }
    saveConfig();

    //std::cerr << "delete MainFrame" << std::endl;

    delete M_log_player;
    M_log_player = static_cast< LogPlayer * >( 0 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::init()
{
    if ( ! AppConfig::instance().gameLogFilePath().empty() )
    {
        loadRCG( wxString( AppConfig::instance().gameLogFilePath().c_str(),
                           *wxConvCurrent ) );
        return;
    }

    if ( AppConfig::instance().connect() )
    {
        std::string host = AppConfig::instance().host();
        if ( host.empty() )
        {
            host = "localhost";
        }
        connectMonitorTo( host.c_str() );

        if ( AppConfig::instance().debugServerMode() )
        {
            startDebugServer();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!
  creates dynamic event signal connect
*/
void
MainFrame::connectEvent()
{
    // ---------------------------------------------------
    // wx event hander
    Connect( wxID_ANY, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( MainFrame::handleCloseEvent ) );
    Connect( wxID_ANY, wxEVT_SIZE,
             wxSizeEventHandler( MainFrame::handleSizeEvent ) );

    // ---------------------------------------------------

    // menu events: file
    Connect( SWID_SHOW_OPEN_RCG_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleOpenRCG ) );
    Connect( SWID_SHOW_SAVE_RCG_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleSaveRCG ) );
    Connect( SWID_SHOW_OPEN_DEBUG_VIEW_DIR_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleOpenDebugView ) );
    Connect( SWID_SHOW_SAVE_DEBUG_VIEW_DIR_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleSaveDebugView ) );
    Connect( SWID_QUIT, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleQuit ) );
    // menu events: monitor
    Connect( SWID_MONITOR_KICKOFF, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleKickOff ) );
    Connect( SWID_REQUEST_LIVE_MODE, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleSetLiveMode ) );
    Connect( SWID_MONITOR_CONNECT, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleConnectMonitor ) );
    Connect( SWID_SHOW_CONNECT_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleConnectMonitorTo ) );
    Connect( SWID_MONITOR_DISCONNECT, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleDisconnectMonitor ) );
    Connect( SWID_REQUEST_KILL_SERVER, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleKillServer ) );
    Connect( SWID_REQUEST_RESTART_SERVER, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleRestartServer ) );
    // menu events: view
    //Connect( SWID_TOGGLE_MENUBAR, wxEVT_COMMAND_MENU_SELECTED,
    //         wxCommandEventHandler( MainFrame::handleToggleMenuBar ) );
    Connect( SWID_TOGGLE_TOOLBAR, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleToggleToolBar ) );
    Connect( SWID_TOGGLE_STATUSBAR, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleToggleStatusBar ) );
    Connect( SWID_TOGGLE_FULLSCREEN, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleToggleFullScreen ) );
    Connect( SWID_SHOW_PLAYER_TYPE_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleShowPlayerTypeDialog ) );
    Connect( SWID_SHOW_DETAIL_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleShowDetailDialog ) );
    Connect( SWID_SHOW_VIEW_CONFIG_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleShowViewConfigDialog ) );
    // menu events: tool
    Connect( SWID_SHOW_DEBUG_MESSAGE_FRAME, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleShowDebugMessageFrame ) );
    Connect( SWID_DEBUG_SERVER_START, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleStartDebugServer ) );
    Connect( SWID_DEBUG_SERVER_STOP, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleStopDebugServer ) );
    Connect( SWID_SHOW_IMAGE_SAVE_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleShowImageSaveDialog ) );
    // menu events: help
    Connect( SWID_SHOW_ABOUT_DIALOG, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( MainFrame::handleAbout ) );

    // ---------------------------------------------------

    // tool bar button actions
    Connect( SWID_LOGPLAYER_GOTO_FIRST, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickGoToFirst ) );
    Connect( SWID_LOGPLAYER_GOTO_PREV_SCORE, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickGoToPrevScore ) );
    // SWID_LOGPLAYER_PLAY_BACK,
    Connect( SWID_LOGPLAYER_STEP_BACK, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickStepBack ) );
    Connect( SWID_LOGPLAYER_PLAY_OR_STOP,wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickPlayOrStop ) );
    // SWID_LOGPLAYER_STOP
    // SWID_LOGPLAYER_PLAY_FORWARD
    Connect( SWID_LOGPLAYER_STEP_FORWARD, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickStepForward ) );
    Connect( SWID_LOGPLAYER_GOTO_NEXT_SCORE, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickGoToNextScore ) );
    Connect( SWID_LOGPLAYER_GOTO_LAST, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickGoToLast ) );
    Connect( SWID_LOGPLAYER_DECELERATE, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickDecelerate ) );
    Connect( SWID_LOGPLAYER_ACCELERATE, wxEVT_COMMAND_TOOL_CLICKED,
             wxCommandEventHandler( MainFrame::clickAccelerate ) );
}

/*-------------------------------------------------------------------*/
/*!
  create & realize menu & menu bar.
  called from constructor.
*/
void
MainFrame::createMenuBar()
{
    // create menu
    wxMenuBar * menubar = new wxMenuBar(
#if 0
#ifdef __WXGTK__
                                        wxMB_DOCKABLE
#endif
#endif
                                        );

    // attach each menu tree
    menubar->Append( createMenuFile(), _( "&File" ) );
    menubar->Append( createMenuMonitor(), _( "&Monitor" ) );
    //menubar->Append( menu_logplayer, _( "&Logplayer" ) );
    menubar->Append( createMenuView(), _( "&View" ) );
    //menubar->Append( createMenuDialog(), _( "D&ialog" ) );
    //menubar->Append( createMenuDebug(), _( "&Debug" ) );
    menubar->Append( createMenuTool(), _( "&Tool" ) );
    menubar->Append( createMenuHelp(), _( "&Help" ) );

    // append menubar to MainFrame.
    this->SetMenuBar( menubar );
}

/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuFile()
{
    wxMenu * menu = new wxMenu;
    menu->Append( SWID_SHOW_OPEN_RCG_DIALOG,
                  _( "&Open rcg file...\tctrl-o" ),
                  _( "Open RoboCup Game Log file" ) );
    menu->Append( SWID_SHOW_SAVE_RCG_DIALOG,
                  _( "Save rcg file as..." ),
                  _( "Save RoboCup Game Log file" ) );
    menu->AppendSeparator();
    menu->Append( SWID_SHOW_OPEN_DEBUG_VIEW_DIR_DIALOG,
                  _( "Open debug view" ),
                  _( "Open the directory where debug view logs exist" ) );
    menu->Append( SWID_SHOW_SAVE_DEBUG_VIEW_DIR_DIALOG,
                  _( "Save debug view" ),
                  _( "Save debug view logs to the directory..." ) );
    menu->AppendSeparator();
    menu->Append( SWID_QUIT,
                  _( "&Quit\tctrl-q" ),
                  _( "Quit application" ) );

    return menu;
}


/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuMonitor()
{
    wxMenu * menu = new wxMenu;
    menu->Append( SWID_MONITOR_KICKOFF,
                  _( "&KickOff\tctrl-k" ),
                  _( "Start the game" ) );
    menu->Enable( SWID_MONITOR_KICKOFF, false );
    menu->Append( SWID_REQUEST_LIVE_MODE,
                  _( "&Live Mode\tctrl-l" ),
                  _( "Live mode" ) );
    menu->Enable( SWID_REQUEST_LIVE_MODE, false );
    menu->AppendSeparator();
    menu->Append( SWID_MONITOR_CONNECT,
                  _( "&Connect\tctrl-c" ),
                  _( "Connect to the rcssserver on localhost" ) );
    menu->Enable( SWID_MONITOR_CONNECT, true );
    menu->Append( SWID_SHOW_CONNECT_DIALOG,
                  _( "Connect &to ..." ),
                  _( "Connect to the rcssserver on other host" ) );
    menu->Enable( SWID_SHOW_CONNECT_DIALOG, true );
    menu->Append( SWID_MONITOR_DISCONNECT,
                  _( "&Disconnect" ),
                  _( "Disonnect from rcssserver" ) );
    menu->Enable( SWID_MONITOR_DISCONNECT, false );
    menu->AppendSeparator();
    menu->Append( SWID_REQUEST_KILL_SERVER,
                  _( "&Kill server" ),
                  _( "Kill the rcssserver process" ) );
    menu->Enable( SWID_REQUEST_KILL_SERVER, false );
    menu->Append( SWID_REQUEST_RESTART_SERVER,
                  _( "(Re)&start server" ),
                  _( "(Re)start rcssserver" ) );
    return menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuView()
{
    wxMenu * menu = new wxMenu;

    ////////////////////////////////////////////////////////
#if 0
    menu->Append( SWID_TOGGLE_MENUBAR,
                  _( "&Menu Bar\tctrl-m" ),
                  _( "Show/Hide Menu Bar" ) );
#endif
    menu->Append( SWID_TOGGLE_TOOLBAR,
                  _( "&Tool Bar" ),
                  _( "Show/Hide Tool Bar" ) );
    menu->Append( SWID_TOGGLE_STATUSBAR,
                  _( "&Status Bar" ),
                  _( "Show/Hide Status Bar" ) );
    menu->AppendSeparator();
    menu->Append( SWID_TOGGLE_FULLSCREEN,
                  _( "&Full Screen\talt-enter" ),
                  _( "Toggle Full Screen" ) );
    ////////////////////////////////////////////////////////
    menu->AppendSeparator();

    menu->Append( SWID_SHOW_PLAYER_TYPE_DIALOG,
                  _( "&Player Type List\tctrl-t" ),
                  _( "Show player type parameters dialog" ) );
    menu->Append( SWID_SHOW_DETAIL_DIALOG,
                  _( "&Object Detail\tctrl-i" ),
                  _( "Show detail information dialog" ) );
    ////////////////////////////////////////////////////////
    menu->AppendSeparator();

    menu->Append( SWID_SHOW_VIEW_CONFIG_DIALOG,
                  _( "&Preference\tctrl-v" ),
                  _( "Show view preference dialog" ) );
    return menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuTool()
{
    wxMenu * menu = new wxMenu;

    menu->Append( SWID_SHOW_DEBUG_MESSAGE_FRAME,
                  _( "Debug &Message\tctrl-d" ),
                  _( "Show debug message window" ) );

    menu->AppendSeparator();


    menu->Append( SWID_DEBUG_SERVER_START,
                  _( "Start Debug Server" ),
                  _( "Start to wait debug connection." ) );
    menu->Enable( SWID_DEBUG_SERVER_START, false );
    menu->Append( SWID_DEBUG_SERVER_STOP,
                  _( "Stop Debug Server" ),
                  _( "Disconnect debug connection." ) );
    menu->Enable( SWID_DEBUG_SERVER_STOP, false );

    menu->AppendSeparator();

    menu->Append( SWID_SHOW_IMAGE_SAVE_DIALOG,
                  _( "Save &Image" ),
                  _( "Save game log data as image files" ) );

    return menu;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxMenu *
MainFrame::createMenuHelp()
{
    wxMenu * menu = new wxMenu;

    menu->Append( SWID_SHOW_ABOUT_DIALOG,
                  _( "&About" ),
                  _( "About soccerwindow2" ) );

    return menu;
}

/*-------------------------------------------------------------------*/
/*!
  create main toolbar.
*/
wxToolBar *
MainFrame::OnCreateToolBar( long style,
                            wxWindowID id,
                            const wxString & name )
{
    M_log_player_tool_bar = new LogPlayerToolBar( this, style, M_main_data );

    M_log_player_tool_bar
        ->emit_cycleChanged.connect( boost::bind( &LogPlayer::goToCycle,
                                                  M_log_player,
                                                  _1 ) );

    M_log_player_tool_bar
        ->emit_indexChanged.connect( boost::bind( &LogPlayer::goToIndex,
                                                  M_log_player,
                                                  _1 ) );

    emit_viewUpdated.connect( boost::bind( &LogPlayerToolBar::updateSlider,
                                           M_log_player_tool_bar ) );

    return M_log_player_tool_bar;
}

/*-------------------------------------------------------------------*/
/*!
  create & realize toolbar.
  called from constructor.
*/
void
MainFrame::createToolBar()
{
    const long style = ( wxTB_FLAT
                         //#ifdef __WXGTK__
                         //| wxTB_DOCKABLE
                         //#endif
                         );
    this->CreateToolBar( style );

    //wxToolBar * tbar = new MonitorToolBar( this,
    //                                       wxTB_FLAT
    //                                       | wxVERTICAL );
    //tbar->SetSize(-1, this->GetClientSize().y );
    //tbar->Move( 0, 0 );
    //tbar->Realize();
}

/*-------------------------------------------------------------------*/
/*!
  create & realize status bar.
  called from constructor.
*/
void
MainFrame::createStatusBar()
{
    // already exist
    if ( this->GetStatusBar() )
    {
        return;
    }

    const int chw = this->GetCharWidth();
    const int widths[] = { -1, 20 * chw };
    this->CreateStatusBar( WXSIZEOF( widths ),
                           wxST_SIZEGRIP,
                           SWID_STATUSBAR );
    this->SetStatusText( wxT( "Ready" ), 0 );
    this->GetStatusBar()->SetStatusWidths( WXSIZEOF( widths ), widths );
}

/*-------------------------------------------------------------------*/
/*!
  \brief create all windows that has this mainframe as a parent
*/
void
MainFrame::createWindows()
{
    M_field_canvas = new FieldCanvas( this, M_main_data );
    M_field_canvas->SetSize( this->GetClientSize() );
    //M_field_canvas->SetSize( 20, 0, this->GetClientSize().x - 20, this->GetClientSize().y );

    M_field_canvas->SetFocus();

    M_field_canvas
        ->emit_dropBall.connect( boost::bind( &MainFrame::dropBall,
                                              this,
                                              _1 ) );
    M_field_canvas
        ->emit_freeKickLeft.connect( boost::bind( &MainFrame::freeKickLeft,
                                                  this,
                                                  _1 ) );
    M_field_canvas
        ->emit_freeKickRight.connect( boost::bind( &MainFrame::freeKickRight,
                                                   this,
                                                   _1 ) );
    M_field_canvas
        ->emit_dropBallThere.connect( boost::bind( &MainFrame::dropBallThere,
                                                   this ) );
    M_field_canvas
        ->emit_mouseMoved.connect( boost::bind( &MainFrame::updatePositionLabel,
                                                this,
                                                _1 ) );
    M_field_canvas
        ->emit_openRCG.connect( boost::bind( &MainFrame::openRCG,
                                             this ) );
    M_field_canvas
        ->emit_connectMonitor.connect( boost::bind( &MainFrame::connectMonitor,
                                                    this ) );
    M_field_canvas
        ->emit_restartServer.connect( boost::bind( &MainFrame::restartServer,
                                                   this ) );
    M_field_canvas
        ->emit_killServer.connect( boost::bind( &MainFrame::killServer,
                                                this ) );
    M_field_canvas
        ->emit_keyPressed.connect( boost::bind( &MainFrame::handleShortcutKey,
                                                this,
                                                _1, _2, _3, _4 ) );


    emit_viewUpdated.connect( boost::bind( &FieldCanvas::updateView,
                                           M_field_canvas ) );

    // --------------------------------------------------------------

    M_view_config_dialog = new ViewConfigDialog( this,
                                                 M_main_data,
                                                 M_main_data.getViewConfig() );
    M_view_config_dialog->Hide();

    M_view_config_dialog
        ->emit_configured.connect( emit_viewUpdated );

    M_view_config_dialog
        ->emit_canvasResized.connect( boost::bind( &MainFrame::resizeCanvas,
                                                   this,
                                                   _1 ) );

    // --------------------------------------------------------------

    // ViewConfigDialog -> FieldCanvas
    M_view_config_dialog
        ->emit_configured.connect( boost::bind( &FieldCanvas::setRedrawAllFlag,
                                                M_field_canvas ) );

    // FieldCanvas -> ViewConfigDialog
    M_field_canvas
        ->emit_focusChanged.connect( boost::bind( &ViewConfigDialog::setFocusPoint,
                                                  M_view_config_dialog,
                                                  _1 ) );

    // FieldCanvas -> LogPlayer
    M_field_canvas
        ->emit_wheelDown.connect( boost::bind( &LogPlayer::stepBack,
                                               M_log_player ) );
    M_field_canvas
        ->emit_wheelUp.connect( boost::bind( &LogPlayer::stepForward,
                                             M_log_player ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::loadConfig()
{
    //std::cerr << "MainFrame::loadConfig" << std::endl;
    wxFileConfig & config_file = SoccerWindowApp::get_config_file();
    wxString old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/MainFrame" ) );

    config_file.SetPath( old_path );

    ///////////////////////////////////////////////////////////////

    old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/App" ) );

    wxString tmp_str;

    if ( AppConfig::instance().gameLogDir().empty() )
    {
        config_file.Read( wxT( "game-log-dir" ), &tmp_str, wxT( "" ) );
#ifdef UNICODE
        AppConfig::instance().setGameLogDir( (const char*)tmp_str.mbc_str() );
#else
        AppConfig::instance().setGameLogDir( (const char*)tmp_str.c_str() );
#endif
    }

    if ( AppConfig::instance().debugLogDir().empty() )
    {
        config_file.Read( wxT( "debug-log-dir" ), &tmp_str, wxT( "" ) );
#ifdef UNICODE
        AppConfig::instance().setDebugLogDir( (const char*)tmp_str.mbc_str() );
#else
        AppConfig::instance().setDebugLogDir( (const char*)tmp_str.c_str() );
#endif
    }

    config_file.SetPath( old_path );
}

/*-------------------------------------------------------------------*/
/*!
  save configuration to file.
  called from destructor.
*/
void
MainFrame::saveConfig()
{
    //std::cerr << "MainFrame::saveConfig" << std::endl;
    wxFileConfig & config_file = SoccerWindowApp::get_config_file();
    wxString old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/MainFrame" ) );

    config_file.SetPath( old_path );

    ///////////////////////////////////////////////////////////////

    old_path = config_file.GetPath();
    config_file.SetPath( wxT( "/App" ) );

    config_file.Write( wxT( "server-command" ),
                       wxString( AppConfig::instance().serverPath().c_str(),
                                 *wxConvCurrent ) );

    config_file.Write( wxT( "game-log-dir" ),
                       wxString( AppConfig::instance().gameLogDir().c_str(),
                                 *wxConvCurrent ) );
    config_file.Write( wxT( "debug-log-dir" ),
                       wxString( AppConfig::instance().debugLogDir().c_str(),
                                 *wxConvCurrent ) );

    config_file.SetPath( old_path );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
MainFrame::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    //std::cerr << "quit main frame" << std::endl;
    this->Destroy();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
MainFrame::handleSizeEvent( wxSizeEvent & WXUNUSED( event ) )
{
    M_field_canvas->SetSize( this->GetClientSize() );

    //std::cerr << "sizeEvent width = " << this->GetClientSize().GetWidth()
    //<< " height = " << this->GetClientSize().GetHeight()
    //<< std::endl;
    M_main_data.getViewConfig().updateFieldSize( this->GetClientSize().GetWidth(),
                                                 this->GetClientSize().GetHeight() );
    if ( M_view_config_dialog
         && M_view_config_dialog->IsShown() )
    {
        M_view_config_dialog->updateFieldScale();
    }

    //wxLayoutAlgorithm layout;
    //layout.LayoutFrame( this );

    //this->Layout();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleOpenRCG( wxCommandEvent & WXUNUSED( event ) )
{
    openRCG();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleSaveRCG( wxCommandEvent & WXUNUSED( event ) )
{
    saveRCG();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleOpenDebugView( wxCommandEvent & WXUNUSED( event ) )
{
    openDebugView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleSaveDebugView( wxCommandEvent & WXUNUSED( event ) )
{
    saveDebugView();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleQuit( wxCommandEvent & WXUNUSED( event ) )
{
    this->Close();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleKickOff( wxCommandEvent & WXUNUSED( event ) )
{
    kickOff();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleSetLiveMode( wxCommandEvent & WXUNUSED( event ) )
{
    setLiveMode();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleConnectMonitor( wxCommandEvent & WXUNUSED( event ) )
{
    connectMonitor();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleConnectMonitorTo( wxCommandEvent & WXUNUSED( event ) )
{
    connectMonitorTo();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleDisconnectMonitor( wxCommandEvent & WXUNUSED( event ) )
{
    disconnectMonitor();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleKillServer( wxCommandEvent & WXUNUSED( event ) )
{
    killServer();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleRestartServer( wxCommandEvent & WXUNUSED( event ) )
{
    restartServer();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleToggleToolBar( wxCommandEvent & WXUNUSED( event ) )
{
    toggleToolBar();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleToggleStatusBar( wxCommandEvent & WXUNUSED( event ) )
{
    toggleStatusBar();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleToggleFullScreen( wxCommandEvent & WXUNUSED( event ) )
{
    toggleFullScreen();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleShowPlayerTypeDialog( wxCommandEvent & WXUNUSED( event ) )
{
    showPlayerTypeDialog();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleShowDetailDialog( wxCommandEvent & WXUNUSED( event ) )
{
    showDetailDialog();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleShowViewConfigDialog( wxCommandEvent & WXUNUSED( event ) )
{
    showViewConfigDialog();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleShowDebugMessageFrame( wxCommandEvent & WXUNUSED( event ) )
{
    showDebugMessageFrame();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleStartDebugServer( wxCommandEvent & WXUNUSED( event ) )
{
    startDebugServer();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleStopDebugServer( wxCommandEvent & WXUNUSED( event ) )
{
    stopDebugServer();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleShowImageSaveDialog( wxCommandEvent & WXUNUSED( event ) )
{
    showImageSaveDialog();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::handleAbout( wxCommandEvent & WXUNUSED( event ) )
{
    about();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickGoToFirst( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->goToFirst();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickGoToPrevScore( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->goToPrevScore();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickStepBack( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->stepBack();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickPlayOrStop( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->playOrStop();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickStepForward( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->stepForward();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickGoToNextScore( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->goToNextScore();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickGoToLast( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->goToLast();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickDecelerate( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->decelerate();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::clickAccelerate( wxCommandEvent & WXUNUSED( event ) )
{
    M_log_player->accelerate();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::openRCG()
{
#ifdef HAVE_LIBRCSC_GZ
    wxString wild_card( wxT( "Game Log files (*.rcg;*.rcg.gz)|*.rcg;*.rcg.gz"
                             "|All files (*.*)|*.*" ) );
#else
    wxString wild_card( wxT( "Game Log files (*.rcg)|*.rcg"
                             "|All files (*.*)|*.*" ) );
#endif

    wxString default_dir( AppConfig::instance().gameLogDir().c_str(),
                          *wxConvCurrent );

#if 1
    wxFileDialog file_dialog( this, // parent window
                              wxT( "Open" ), // dialog name shown on titlebar
                              default_dir, // default dir
                              wxT( "" ), // default file
                              wild_card,
                              wxOPEN
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return;
    }

    wxString file_path = file_dialog.GetPath();
#else
    wxString file_path
        = ::wxFileSelector( wxT( "Open" ), // message
                            default_dir, // default dir
                            wxT( "" ), // default file name,
                            wxT( "" ), // default extention
                            wild_card, // wild card
                            wxOPEN, // flag,
                            this // parent window
                            );
#endif
    if ( file_path.IsEmpty() )
    {
        return;
    }

    loadRCG( file_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::loadRCG( const wxString & file_path )
{
    disconnectMonitor();
    M_log_player->stop();

    // load rcg data from file
    std::string file_path_std;
#ifdef UNICODE
    file_path_std = (const char*)file_path.mbc_str();
#else
    file_path_std = (const char*)file_path.c_str();
#endif

    if ( ! M_main_data.loadRCG( file_path_std ) )
    {
        // failed...
        wxString err_msg = wxT( "Failed to read [" );
        err_msg += file_path;
        err_msg += wxT( "]!!!" );
        ::wxMessageBox( err_msg,
                        wxT( "RCG file read error!" ), // title
                        wxOK | wxICON_ERROR ); // style
        return;
    }

    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        wxString err_msg = wxT( "Empty Log [" );
        err_msg += file_path;
        err_msg += wxT( "]!!!" );
        ::wxMessageBox( err_msg,
                        wxT( "RCG file is empty!" ), // title
                        wxOK | wxICON_ERROR ); // style
        return;
    }

    if ( ::wxIsAbsolutePath( file_path ) )
    {
        std::string absolute_path;
#ifdef UNICODE
        absolute_path = (const char*)::wxPathOnly( file_path ).mbc_str();
#else
        absolute_path = (const char*)::wxPathOnly( file_path ).c_str();
#endif
        AppConfig::instance().setGameLogDir( absolute_path );
    }

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->updateData();
    }

    if ( M_debug_message )
    {
        M_debug_message->clearAll();
    }

    if ( M_view_config_dialog )
    {
        M_view_config_dialog->unzoom();
    }

    emit_viewUpdated();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::saveRCG()
{
    // check data size
    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        ::wxMessageBox( wxT( "No Monitor View Data!" ),
                        wxT( "Error" ), // title
                        wxOK | wxICON_ERROR ); // style
        return;
    }

    ///////////////////////////////////////////////////////////
    wxString default_file_name;
    {
        MonitorViewConstPtr latest = M_main_data.viewHolder().latestViewData();

        if ( latest )
        {
            wxDateTime now = wxDateTime::Now();
            default_file_name = now.Format( wxT( "%Y%m%d%H%M-" ),
                                            wxDateTime::Local );

            wxString left_team( latest->leftTeam().name().c_str(),
                                *wxConvCurrent );
            wxString left_score;
            left_score.sprintf( wxT( "_%d" ), latest->leftTeam().score() );
            wxString right_team( latest->rightTeam().name().c_str(),
                                 *wxConvCurrent );
            wxString right_score;
            right_score.sprintf( wxT( "_%d" ), latest->rightTeam().score() );

            default_file_name += left_team;
            default_file_name += left_score;
            default_file_name += wxT( "-" );
            default_file_name += right_team;
            default_file_name += right_score;
            default_file_name += wxT( ".rcg" );
        }
    }


    ///////////////////////////////////////////////////////////

    wxString wild_card( wxT( "Game Log files (*.rcg)|*.rcg"
                             "|All files (*.*)|*.*" ) );
    wxString default_dir( AppConfig::instance().gameLogDir().c_str(),
                          *wxConvCurrent );
#if 1
    wxFileDialog file_dialog( this, // parent window
                              wxT( "Save" ), // dialog name shown on titlebar
                              default_dir, // default dir
                              default_file_name, // default file
                              wild_card,
                              wxSAVE
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return;
    }

    wxString filepath = file_dialog.GetPath();
#else
    wxString filepath
        = ::wxFileSelector( wxT( "Save" ), // message
                            default_dir, // default dir
                            default_file_name, // default file name,
                            wxT( ".rcg" ), // default extention
                            wild_card, // wild card
                            wxSAVE, // flag,
                            this // parent window
                            );
    if ( filepath.IsEmpty() )
    {
        return;
    }
#endif

    if ( ::wxIsAbsolutePath( filepath ) )
    {
        std::string dir_str;
#ifdef UNICODE
        dir_str = (const char*)::wxPathOnly( filepath ).mbc_str();
#else
        dir_str = (const char*)::wxPathOnly( filepath ).c_str();
#endif
        AppConfig::instance().setGameLogDir( dir_str );
    }

    // set a file path string with proper extention.
    std::string file_path_std;
#ifdef UNICODE
    file_path_std = (const char*)filepath.mbc_str();
#else
    file_path_std = (const char*)filepath.c_str();
#endif

    bool is_gzip = false;
    if ( file_path_std.length() > 3
         && file_path_std.compare( filepath.length() - 3, 3, ".gz" ) == 0 )
    {
#ifdef HAVE_LIBRCSC_GZ
        if ( file_path_std.length() <= 7
             || file_path_std.compare( filepath.length() - 4, 4, ".rcg.gz" ) != 0 )
        {
            file_path_std == ".rcg.gz";
        }
        is_gzip = true;
#else
        // erase '.gz'
        file_path_std.erase( filepath.length() - 3 );
#endif
    }

    if ( ! is_gzip )
    {
        if ( file_path_std.length() <= 4
             || file_path_std.compare( filepath.length() - 4, 4, ".rcg" ) != 0 )
        {
            file_path_std += ".rcg";
        }
    }

    M_main_data.saveRCG( file_path_std );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::openDebugView()
{
    // check data size
    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        ::wxMessageBox( wxT( "No Monitor View Data!" ),
                        wxT( "Error" ), // title
                        wxOK | wxICON_ERROR ); // style
        return;
    }

    if ( AppConfig::instance().monitorClientMode() )
    {
        ::wxMessageBox( wxT( "Monitor client is running!" ),
                        wxT( "Error" ), // title
                        wxOK | wxICON_ERROR ); // style
        return;
    }

    // realize directory selector dialog

    wxString dir_str( wxT( "" ) );
    if ( AppConfig::instance().debugLogDir().empty() )
    {
        dir_str = ::wxGetCwd(); // current working directory
    }
    else
    {
        dir_str = wxString( AppConfig::instance().debugLogDir().c_str(),
                            *wxConvCurrent );
    }

    wxDirDialog dlg( this, // parent parent
                     wxT( "Choose a debug view log directory" ),
                     dir_str,
                     wxDEFAULT_DIALOG_STYLE );

    if ( dlg.ShowModal() != wxID_OK )
    {
        return;
    }

#ifdef UNICODE
    std::string dir_path = (const char*)dlg.GetPath().mbc_str();
#else
    std::string dir_path = (const char*)dlg.GetPath().c_str();
#endif

    M_main_data.getViewHolder().loadDebugView( dir_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::saveDebugView()
{
    // check data size
    if ( M_main_data.viewHolder().monitorViewCont().empty() )
    {
        ::wxMessageBox( wxT( "No Monitor View Data!" ),
                        wxT( "Error" ), // title
                        wxOK | wxICON_ERROR ); // style
        return;
    }

    if ( AppConfig::instance().monitorClientMode() )
    {
        ::wxMessageBox( wxT( "Monitor client is running!" ),
                        wxT( "Error" ), // title
                        wxOK | wxICON_ERROR ); // style
        return;
    }

    // realize directory selector dialog

    wxString dir_str = ::wxGetCwd(); // current working directory

    wxDirDialog dlg( this, // parent parent
                     wxT( "Choose a directry to save the debug view logs" ),
                     dir_str,
                     wxDEFAULT_DIALOG_STYLE );

    if ( dlg.ShowModal() != wxID_OK )
    {
        return;
    }

#ifdef UNICODE
    std::string dir_path = (const char*)dlg.GetPath().mbc_str();
#else
    std::string dir_path = (const char*)dlg.GetPath().c_str();
#endif

    M_main_data.viewHolder().saveDebugView( dir_path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::kickOff()
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        M_monitor_client->sendKickOff();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::setLiveMode()
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        M_log_player->setLiveMode();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::connectMonitor()
{
    connectMonitorTo( "localhost" );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::connectMonitorTo()
{
    wxString hostname = wxGetTextFromUser( wxT( "Input server host" ),// message
                                           wxT( "Input text" ), // caption
                                           wxT( "" ), // default value
                                           this ); // parent
    if ( hostname.Length() > 0 )
    {
#ifdef UNICODE
        std::cerr << "host = " << (const char*)hostname.mbc_str()
                  << std::endl;
        connectMonitorTo( (const char*)(hostname.mbc_str()) );
#else
        std::cerr << "host = " << (const char*)hostname.c_str()
                  << std::endl;
        connectMonitorTo( (const char*)(hostname.c_str()) );
#endif
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::connectMonitorTo( const char * hostname )
{
    if ( std::strlen( hostname ) == 0 )
    {
        std::cerr << "Empty host name! Connection failed!" << std::endl;
        return;
    }

    std::cerr << "Connect to rcssserver on [" << hostname << "]" << std::endl;

    M_monitor_client
        = boost::shared_ptr< MonitorClient >
        ( new MonitorClient( AppConfig::instance().clientVersion(),
                             hostname,
                             AppConfig::instance().port(),
                             this,
                             M_main_data.getViewHolder() ) );
    if ( ! M_monitor_client->isConnected() )
    {
        std::cerr << "Conenction failed." << std::endl;
        M_monitor_client.reset();
        return;
    }

    // reset all data
    M_main_data.clear();

    if ( M_player_type_dialog )
    {
        M_player_type_dialog->Hide();
    }

    if ( M_debug_message )
    {
        M_debug_message->clearAll();
    }

    if ( M_view_config_dialog )
    {
        M_view_config_dialog->unzoom();
    }

    if ( M_debug_server )
    {
        std::cerr << "============debug server is runnning" << std::endl;
        M_debug_server.reset();
        startDebugServer();
        std::cerr << "=========== restart debug server" << std::endl;
    }

    AppConfig::instance().setMonitorClientMode( true );


    wxMenuBar * menubar = this->GetMenuBar();
    if ( menubar )
    {
        menubar->Enable( SWID_SHOW_SAVE_RCG_DIALOG, false );

        menubar->Enable( SWID_MONITOR_KICKOFF, true );
        menubar->Enable( SWID_REQUEST_LIVE_MODE, true );
        menubar->Enable( SWID_MONITOR_CONNECT, false );
        menubar->Enable( SWID_SHOW_CONNECT_DIALOG, false );
        menubar->Enable( SWID_MONITOR_DISCONNECT, true );
        menubar->Enable( SWID_REQUEST_KILL_SERVER, true );

        menubar->Enable( SWID_DEBUG_SERVER_START, true );
        menubar->Enable( SWID_DEBUG_SERVER_STOP, true );
        menubar->Enable( SWID_SHOW_IMAGE_SAVE_DIALOG, false );
    }


    M_log_player->setLiveMode();

    M_monitor_client->sendDispInit();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::disconnectMonitor()
{
    if ( M_monitor_client )
    {
        std::cerr << "Disconnect" << std::endl;
        M_monitor_client.reset(); // delete current connection
    }

    if ( M_debug_server )
    {
        M_debug_server.reset();
    }

    AppConfig::instance().setMonitorClientMode( false );

    wxMenuBar * menubar = this->GetMenuBar();
    if ( menubar )
    {
        menubar->Enable( SWID_SHOW_SAVE_RCG_DIALOG, true );

        menubar->Enable( SWID_MONITOR_KICKOFF, false );
        menubar->Enable( SWID_REQUEST_LIVE_MODE, false );
        menubar->Enable( SWID_MONITOR_CONNECT, true );
        menubar->Enable( SWID_SHOW_CONNECT_DIALOG, true );
        menubar->Enable( SWID_MONITOR_DISCONNECT, false );
        menubar->Enable( SWID_REQUEST_KILL_SERVER, false );

        menubar->Enable( SWID_DEBUG_SERVER_START, false );
        menubar->Enable( SWID_DEBUG_SERVER_STOP, false );
        menubar->Enable( SWID_SHOW_IMAGE_SAVE_DIALOG, true );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::killServer()
{
    disconnectMonitor();

    AppConfig::instance().setKillServer( false );

    if ( AppConfig::instance().serverPID() != 0 )
    {
        AppConfig::instance().setServerPID( 0 );
        ::kill( pid_t( AppConfig::instance().serverPID() ), SIGINT );
    }
    else
    {
        std::system( "killall -s INT rcssserver" );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::restartServer()
{
    ::wxBeginBusyCursor();

    if ( M_monitor_client )
    {
        killServer();
        M_monitor_client.reset();

        ::wxSleep( 1 ); // sleep 1 sec
    }

    AppConfig::instance().setServerPID( 0 );
    AppConfig::instance().setKillServer( true );

    std::string rcssserver = AppConfig::instance().serverPath();

    if ( rcssserver.empty() )
    {
        ::wxEndBusyCursor();
        ::wxMessageBox( wxT( "Empty command string to invoke the rcssserver." ),
                        wxT( "Error" ),   // title
                        wxOK | wxICON_ERROR, // style
                        this ); // parent frame
        return;
    }

    rcssserver += " > /dev/null 2>&1 &";

    if ( std::system( rcssserver.c_str() ) == -1 )
    {
        ::wxEndBusyCursor();
        ::wxMessageBox( wxT( "Failed to start rcssserver." ),
                        wxT( "Error" ),   // title
                        wxOK | wxICON_ERROR, // style
                        this ); // parent frame
        return;
    }

    ::wxSleep( 1 ); // sleep 1 sec

    connectMonitorTo( "localhost" );

    ::wxEndBusyCursor();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleMenuBar()
{
    if ( this->GetMenuBar() )
    {
        if ( this->GetMenuBar()->IsShown() )
        {
            this->GetMenuBar()->Hide();
        }
        else
        {
            this->GetMenuBar()->Show();
        }
        this->Fit();
    }
    else
    {
        createMenuBar();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleToolBar()
{
    if ( this->GetToolBar() )
    {
        if ( this->GetToolBar()->IsShown() )
        {
            this->GetToolBar()->Hide();
        }
        else
        {
            this->GetToolBar()->Show();
        }
        this->Fit();
    }
    else
    {
        createToolBar();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleStatusBar()
{
    if ( this->GetStatusBar() )
    {
        if ( this->GetStatusBar()->IsShown() )
        {
            this->GetStatusBar()->Hide();
        }
        else
        {
            this->GetStatusBar()->Show();
            this->SetStatusText( wxT( "Ready" ), 0 );
        }
        this->Fit();
    }
    else
    {
        createStatusBar();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::toggleFullScreen()
{
    this->ShowFullScreen( ! this->IsFullScreen() );
}


/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::showPlayerTypeDialog()
{
    if ( M_player_type_dialog )
    {
        M_player_type_dialog->Show( ! M_player_type_dialog->IsShown() );
    }
    else
    {
        M_player_type_dialog = new PlayerTypeDialog( this, M_main_data );
        M_player_type_dialog->Show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::showDetailDialog()
{
    if ( M_detail_dialog )
    {
        // toggle show/hide
        M_detail_dialog->Show( ! M_detail_dialog->IsShown() );
    }
    else
    {
        M_detail_dialog = new DetailDialog( this, M_main_data );
        M_detail_dialog->Show();

        emit_viewUpdated.connect( boost::bind( &DetailDialog::updateLabels,
                                               M_detail_dialog ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::showViewConfigDialog()
{
    // toggle show/hide
    M_view_config_dialog->Show( ! M_view_config_dialog->IsShown() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::showDebugMessageFrame()
{
    if ( M_debug_message )
    {
        M_debug_message->Show( ! M_debug_message->IsShown() );
    }
    else
    {
        M_debug_message = new DebugMessageFrame( this, M_main_data );
        M_debug_message
            ->emit_configured.connect( emit_viewUpdated );

        M_debug_message->Show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::startDebugServer()
{
    if ( M_debug_server )
    {
        // already created
        return;
    }

    std::cerr << "Start Debug Server" << std::endl;

    int port = static_cast< int >( AppConfig::instance().debugServerPort() );

    M_debug_server
        = boost::shared_ptr< DebugServer >
        ( new DebugServer( port, M_main_data.getViewHolder() ) );
    if ( ! M_debug_server->isConnected() )
    {
        std::cerr << "failed to create Debug Server" << std::endl;
        M_debug_server.reset();
        return;
    }

    M_debug_server->start();


    wxMenuBar * menubar = this->GetMenuBar();
    if ( menubar )
    {
        menubar->Enable( SWID_DEBUG_SERVER_START, false );
        menubar->Enable( SWID_DEBUG_SERVER_STOP, true );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::stopDebugServer()
{
    if ( M_debug_server )
    {
        std::cerr << "Stop Debug Server" << std::endl;
        M_debug_server.reset();
    }

    wxMenuBar * menubar = this->GetMenuBar();
    if ( menubar )
    {
        if ( AppConfig::instance().monitorClientMode() )
        {
            menubar->Enable( SWID_DEBUG_SERVER_START, true );
        }
        else
        {
            menubar->Enable( SWID_DEBUG_SERVER_START, false );
        }
        menubar->Enable( SWID_DEBUG_SERVER_STOP, false );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::showImageSaveDialog()
{
    ImageSaveDialog dlg( this, M_field_canvas, M_main_data );

    dlg.ShowModal();
#if 0
    if ( M_image_save_dialog )
    {
        // toggle show/hide
        M_image_save_dialog->Show( ! M_image_save_dialog->IsShown() );
    }
    else
    {
        M_image_save_dialog = new ImageSaveDialog( this,
                                                   M_field_canvas,
                                                   M_main_data );
        M_image_save_dialog->Show();
    }
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::about()
{
    wxString msg( PACKAGE,
                  *wxConvCurrent );
    msg += wxT( "-" );
    msg += wxT( VERSION );
    msg += wxT( "\n\n"
                "soccerwindow2 is a viewer applicaton for\n"
                "the RoboCup Soccer Simulator.\n"
                "  http://sserver.sourceforge.net/\n"
                "\n"
                "soccerwindow2 Development Site:\n"
                "  http://rctools.sourceforge.jp/" );
    ::wxMessageBox( msg,
                    wxT( "About soccerwindow2" ),   // title
                    wxOK | wxICON_INFORMATION, // style
                    this ); // parent frame
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MainFrame::handleShortcutKey( const int key_code,
                              const bool ctrl_down,
                              const bool alt_down,
                              const bool shift_down )
{
    // if Ctrl is down, the key code value of char event of 'a' becomes 1.
    // case 1:

    switch ( key_code ) {
    case wxT( '1' ):
    case wxT( '2' ):
    case wxT( '3' ):
    case wxT( '4' ):
    case wxT( '5' ):
    case wxT( '6' ):
    case wxT( '7' ):
    case wxT( '8' ):
    case wxT( '9' ):
        {
            int number = key_code - wxT( '0' );
            if ( ctrl_down ) number += 11;
            M_view_config_dialog->selectPlayerNumber( number );
        }
        break;
    case wxT( '0' ):
        if ( ctrl_down )
        {
            M_view_config_dialog->selectPlayerNumber( 11 + 10 );
        }
        else
        {
            M_view_config_dialog->selectPlayerNumber( 10 );
        }
        break;
    case wxT( '-' ):
        if ( ctrl_down )
        {
            M_view_config_dialog->selectPlayerNumber( 11 + 11 );
        }
        else
        {
            M_view_config_dialog->selectPlayerNumber( 11 );
        }
        break;
    case wxT( 'a' ):
        M_view_config_dialog->toggleSelectAutoAll();
        break;
    case wxT( 'b' ):
        if ( ctrl_down )
        {
            M_view_config_dialog->toggleShowBall();
        }
        else
        {
            M_view_config_dialog->toggleFocusBall();
        }
        break;
    case wxT( 'c' ):
        if ( ctrl_down )
        {
            return false;
        }
        else
        {
            M_view_config_dialog->toggleShowControlArea();
        }
        break;
    case wxT( 'e' ):
        M_view_config_dialog->toggleEnlarge();
        break;
    case wxT( 'f' ):
        M_view_config_dialog->toggleShowFlags();
        break;
    case wxT( 'g' ):
        if ( ctrl_down )
        {
            M_log_player->goToPrevScore();
        }
        else
        {
            M_log_player->goToNextScore();
        }
        break;
    case wxT( 'i' ):
        if ( ctrl_down )
        {
            return false;
        }
        else
        {
            M_view_config_dialog->unzoom();
        }
        break;
    case wxT( 'h' ):
        M_view_config_dialog->toggleShowHeteroNumber();
        break;
    case wxT( 'k' ):
        if ( ctrl_down )
        {
            return false;
        }
        else
        {
            M_view_config_dialog->toggleKeepawayMode();
        }
        break;
    case wxT( 'l' ):
        if ( ctrl_down )
        {
            return false;
        }
        else
        {
            M_view_config_dialog->toggleSelectAutoLeft();
        }
        break;
    case wxT( 'n' ):
        M_view_config_dialog->toggleShowPlayerNumber();
        break;
    case wxT( 'o' ):
        if ( ctrl_down )
        {
            return false;
        }
        else
        {
            M_view_config_dialog->toggleShowOffsideLine();
        }
        break;
    case wxT( 'p' ):
        if ( ctrl_down )
        {
            M_view_config_dialog->toggleShowPlayers();
        }
        else
        {
            M_view_config_dialog->toggleFocusPlayer();
        }
        break;
    case wxT( 'r' ):
        M_view_config_dialog->toggleSelectAutoRight();
        break;
    case wxT( 's' ):
        M_view_config_dialog->toggleShowStamina();
        break;
    case wxT( 't' ):
        M_view_config_dialog->toggleShowScoreBoard();
        break;
    case wxT( 'u' ):
        M_view_config_dialog->setUnselect();
        break;
    case wxT( 'v' ):
        if ( ctrl_down )
        {
            return false;
        }
        else
        {
            M_view_config_dialog->toggleShowViewCone();
        }
        break;
        //case 1 + ('z' - 'a'): // Ctrl + char event of 'z'
    case wxT( 'x' ):
        M_view_config_dialog->zoomOut();
        break;
    case wxT( 'z' ):
        M_view_config_dialog->zoomIn();
        break;
    case WXK_LEFT:
        if ( ctrl_down )
        {
            M_log_player->decelerate();
        }
        else
        {
            M_log_player->stepBack();
        }
        break;
    case WXK_RIGHT:
        if ( ctrl_down )
        {
            M_log_player->accelerate();
        }
        else
        {
            M_log_player->stepForward();
        }
        break;
    case WXK_UP:
        M_log_player->playForward();
        break;
    case WXK_DOWN:
        M_log_player->stop();
        break;
    case WXK_HOME:
        M_log_player->goToFirst();
        break;
    case WXK_END:
        M_log_player->goToLast();
        break;
    case WXK_SPACE:
        M_log_player->playOrStop();
        break;
    default:
        return false;
        break;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::resizeCanvas( const wxSize & size )
{
    this->SetClientSize( size );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::receiveMonitorPacket()
{
    if ( M_log_player->isLiveMode() )
    {
        M_log_player->showLive();
    }
    else
    {
        std::cerr << "receive monitor packet  no live" << std::endl;
        M_log_player_tool_bar->updateSlider();
    }


    if ( AppConfig::instance().autoQuitMode() )
    {
        static long s_game_end_time = 0;

        if ( M_main_data.viewHolder().lastPlayMode() != rcsc::PM_TimeOver )
        {
            s_game_end_time = 0;
        }
        else
        {
            if ( s_game_end_time == 0 )
            {
                s_game_end_time = ::wxGetLocalTime();
            }
            else
            {
                if ( ::wxGetLocalTime() - s_game_end_time
                     >= AppConfig::instance().waitSeconds() )
                {
                    std::cerr << "Elapsed " << AppConfig::instance().waitSeconds()
                              << " seconds after game end\n"
                              << "Exit..."
                              << std::endl;
                    ::wxExit();
                }
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::updatePositionLabel( const wxPoint & point )
{
    if ( GetStatusBar()
         && GetStatusBar()->IsShown() )
    {
        double x = M_main_data.viewConfig().getFieldX( point.x );
        double y = M_main_data.viewConfig().getFieldY( point.y );

        wxString pos_str;
        pos_str.Printf( wxT( "(%.2f %.2f)" ), x, y );

        GetStatusBar()->SetStatusText( pos_str, 1 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::dropBallThere()
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        MonitorViewConstPtr view = M_main_data.viewHolder().latestViewData();
        if ( view )
        {
            std::cerr << "drop ball at current position "
                      << std::endl;
            M_monitor_client->sendDropBall( view->ball().x(),
                                            view->ball().y() );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::dropBall( const wxPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().getFieldX( point.x );
        double y = M_main_data.viewConfig().getFieldY( point.y );

        std::cerr << "drop ball to ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendDropBall( x, y );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::freeKickLeft( const wxPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().getFieldX( point.x );
        double y = M_main_data.viewConfig().getFieldY( point.y );

        std::cerr << "free kick left at ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendFreeKickLeft( x, y );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainFrame::freeKickRight( const wxPoint & point )
{
    if ( M_monitor_client
         && M_monitor_client->isConnected() )
    {
        double x = M_main_data.viewConfig().getFieldX( point.x );
        double y = M_main_data.viewConfig().getFieldY( point.y );

        std::cerr << "free kick right at ("
                  << x << ", " << y << ")"
                  << std::endl;
        M_monitor_client->sendFreeKickRight( x, y );
    }
}
