// -*-c++-*-

/*!
  \file view_config_dialog.cpp
  \brief View Config Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// For compilers that support precompilation, includes "wx/wx.h".
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/spinctrl.h>
#include <wx/radiobut.h>
#endif

#include "view_config_dialog.h"

// model
#include "id.h"
#include "main_data.h"
#include "view_config.h"

#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::ViewConfigDialog( wxWindow * parent,
                                    const MainData & main_data,
                                    ViewConfig & view_config )
    : wxDialog( parent,
                -1,
                wxT( "View Preference"),
                wxDefaultPosition,
                wxDefaultSize,
                wxCAPTION | wxSYSTEM_MENU | wxCLOSE_BOX )
    , M_main_data( main_data )
    , M_view_config( view_config )
{
    createControls();
    connectEvents();

    wxRect cli_rect = ::wxGetClientDisplayRect();
#if 0
    std::cerr << "client display.  "
              << " x = " << cli_rect.GetX()
              << " y = " << cli_rect.GetY()
              << " width = " << cli_rect.GetWidth()
              << " height = " << cli_rect.GetHeight()
              << std::endl;
    std::cerr << "my size = "
              << this->GetSize().x << " "
              << this->GetSize().y  << "  virtual = "
              << this->GetVirtualSize().x  << " "
              << this->GetVirtualSize().y  << std::endl;
#endif
    this->SetSize( 0, cli_rect.GetHeight() - this->GetSize().y - 32,
                   -1, -1 );
}

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::~ViewConfigDialog()
{
    //std::cerr << "delete ViewConfigDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ViewConfigDialog::Show( bool show )
{
    if ( show )
    {
        updateAll();
    }

    return wxDialog::Show( show );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::createControls()
{
    wxBoxSizer * main_sizer = new wxBoxSizer( wxVERTICAL );

    // zoom options
    main_sizer->Add( createZoomControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // canvas size
    main_sizer->Add( createCanvasSizeControls( this ),
                     0, wxLEFT | wxRIGHT, 4 );
    // detail info options
    main_sizer->Add( createPlayersDetailControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // show / hide info
    main_sizer->Add( createShowControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // show / hide computational geometry
    main_sizer->Add( createCompGeomControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // focus selector
    main_sizer->Add( createFocusControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // player selector
    main_sizer->Add( createPlayerSelectionControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // grass type
    main_sizer->Add( createGrassTypeControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // trace cycle span
    main_sizer->Add( createTraceControls(),
                     0, wxLEFT | wxRIGHT, 4 );
    // future cycle info
    main_sizer->Add( createInertiaMoveControls(),
                     0, wxLEFT | wxRIGHT, 4 );

    main_sizer->Add( 0, 2 ); // bottom space

    main_sizer->SetSizeHints( this );
    this->SetAutoLayout( true );
    this->SetSizer( main_sizer );
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createZoomControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Zoom" ) );
    wxStaticBoxSizer * top_sizer = new wxStaticBoxSizer( sbox, wxVERTICAL );

    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );

        sizer->Add( new wxButton( this, SWID_CANVAS_ZOOMIN,
                                  wxT( "+" ),
                                  wxDefaultPosition,
                                  wxSize( 64, -1 ) ),
                    0, 0, 0 );
        sizer->Add( new wxButton( this, SWID_CANVAS_ZOOMOUT,
                                  wxT( "-" ),
                                  wxDefaultPosition,
                                  wxSize( 64, -1 ) ),
                    0, 0, 0 );
        M_enlarge_cb = new wxCheckBox( this,
                                       SWID_CANVAS_ENLARGE,
                                       wxT( "Enlarge Objects" ) );
        M_enlarge_cb->SetValue( M_view_config.isEnlarged() );
        sizer->Add( M_enlarge_cb, 0, wxLEFT | wxRIGHT, 4 );
        sizer->Add( new wxButton( this, SWID_CANVAS_UNZOOM,
                                  wxT( "Fit" ),
                                  wxDefaultPosition,
                                  wxSize( 64, -1 ) ),
                    0, wxLEFT, 10 );

        top_sizer->Add( sizer, 0, 0, 0 );
    }

    {
        wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );

        sizer->Add( new wxStaticText( this, -1, wxT( "Scale" ) ),
                    0, wxLEFT | wxRIGHT | wxALIGN_CENTER_VERTICAL,
                    1 );

        M_scale_slider = new wxSlider( this,
                                       SWID_FIELD_SCALE_SLIDER,
                                       80, 10, 4000,  // value, min value, max value
                                       wxDefaultPosition,
                                       wxSize( 240, -1 ),
                                       wxSL_HORIZONTAL
                                       | wxSL_AUTOTICKS );
        M_scale_slider->SetLineSize( 1 );
        M_scale_slider->SetPageSize( 1 ); // set the minimal increment/decrement step
        sizer->Add( M_scale_slider, 0, 0, 0 );

        M_scale_text = new wxStaticText( this, -1, wxT( "0000.00" ),
                                         wxDefaultPosition,
                                         wxSize( 48, -1 ),
                                         wxALIGN_LEFT | wxST_NO_AUTORESIZE );

        sizer->Add( M_scale_text, 0, wxLEFT | wxRIGHT | wxALIGN_CENTER_VERTICAL,
                    1 );

        top_sizer->Add( sizer, 0, 0, 0 );
    }

    return top_sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createCanvasSizeControls( wxWindow * parent )
{
    wxStaticBox * sbox = new wxStaticBox( parent, -1,
                                          wxT( "Canvas Size" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    sizer->Add( new wxStaticText( parent, -1, wxT( " Width:" ) ),
                0, wxALIGN_CENTER_VERTICAL, 0 );

    wxString size_str;

    size_str.sprintf( wxT( "%d" ), parent->GetClientSize().GetWidth() );
    M_canvas_width_text = new wxTextCtrl( parent, -1, size_str );
    sizer->Add( M_canvas_width_text,
                0, wxALIGN_CENTER_VERTICAL, 0 );

    sizer->Add( new wxStaticText( parent, -1, wxT( " Height:" ) ),
                0, wxALIGN_CENTER_VERTICAL, 0 );

    size_str.sprintf( wxT( "%d" ), parent->GetClientSize().GetHeight() );
    M_canvas_height_text = new wxTextCtrl( parent, -1, size_str );
    sizer->Add( M_canvas_height_text,
                0, wxALIGN_CENTER_VERTICAL, 0 );

    //M_set_canvas_size_btn = new wxButton( parent, SWID_CANVAS_LOAD_SIZE_BUTTON,
    //wxT( "load" ) );
    //sizer->Add( M_set_canvas_size_btn,
    //0, wxALIGN_CENTER_VERTICAL | wxLEFT, 6 );

    M_set_canvas_size_btn = new wxButton( parent, SWID_CANVAS_SET_SIZE_BUTTON,
                                          wxT( "apply" ) );
    sizer->Add( M_set_canvas_size_btn,
                0, wxALIGN_CENTER_VERTICAL, 0 );

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createPlayersDetailControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Players\' Detail" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_player_number_cb = new wxCheckBox( this,
                                             SWID_CANVAS_DRAW_PLAYER_NUMBER,
                                             wxT( "Unum") );
        M_player_number_cb->SetValue( M_view_config.isShownPlayerNumber() );
        sizer->Add( M_player_number_cb );
    }
    {
        M_hetero_number_cb = new wxCheckBox( this,
                                             SWID_CANVAS_DRAW_HETERO_NUMBER,
                                             wxT( "Type" ) );
        M_hetero_number_cb->SetValue( M_view_config.isShownHeteroNumber() );
        sizer->Add( M_hetero_number_cb );
    }
    {
        M_stamina_cb = new wxCheckBox( this,
                                       SWID_CANVAS_DRAW_STAMINA,
                                       wxT( "Stamina" ) );
        M_stamina_cb->SetValue( M_view_config.isShownStamina() );
        sizer->Add( M_stamina_cb );
    }
    {
        M_view_cone_cb = new wxCheckBox( this,
                                         SWID_CANVAS_DRAW_VIEW_CONE,
                                         wxT( "View Cone" ) );
        M_view_cone_cb->SetValue( M_view_config.isShownViewCone() );
        sizer->Add( M_view_cone_cb );
    }
    {
        M_control_area_cb = new wxCheckBox( this,
                                            SWID_CANVAS_DRAW_CONTROL_AREA,
                                            wxT( "Control Area" ) );
        M_control_area_cb->SetValue( M_view_config.isShownControlArea() );
        sizer->Add( M_control_area_cb );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createShowControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Show" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_show_score_board_cb = new wxCheckBox( this,
                                                SWID_CANVAS_DRAW_SCORE_BOARD,
                                                wxT( "Score Board" ) );
        M_show_score_board_cb->SetValue( M_view_config.isShownScoreBoard() );
        sizer->Add( M_show_score_board_cb );
    }
    {
        M_show_ball_cb = new wxCheckBox( this,
                                         SWID_CANVAS_DRAW_BALL,
                                         wxT( "Ball" ) );
        M_show_ball_cb->SetValue( M_view_config.isShownBall() );
        sizer->Add( M_show_ball_cb );
    }
    {
        M_show_players_cb = new wxCheckBox( this,
                                           SWID_CANVAS_DRAW_PLAYER,
                                           wxT( "Player" ) );
        M_show_players_cb->SetValue( M_view_config.isShownPlayers() );
        sizer->Add( M_show_players_cb );
    }
    {
        M_show_flags_cb = new wxCheckBox( this,
                                          SWID_CANVAS_DRAW_FLAGS,
                                          wxT( "Flag" ) );
        M_show_flags_cb->SetValue( M_view_config.isShownFlags() );
        sizer->Add( M_show_flags_cb );
    }
    {
        M_show_offside_line_cb = new wxCheckBox( this,
                                                 SWID_CANVAS_DRAW_OFFSIDE_LINE,
                                                 wxT( "Offside Line" ) );
        M_show_offside_line_cb->SetValue( M_view_config.isShownOffsideLine() );
        sizer->Add( M_show_offside_line_cb );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createCompGeomControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Compututational Geometry" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_show_voronoi_cb = new wxCheckBox( this,
                                            SWID_CANVAS_DRAW_VORONOI_DIAGRAM,
                                            wxT( "VoronoiDiagram" ) );
        M_show_voronoi_cb->SetValue( M_view_config.isShownVoronoiDiagram() );
        sizer->Add( M_show_voronoi_cb, 0, wxALIGN_CENTER_VERTICAL  );
    }
    {
        M_show_delaunay_cb = new wxCheckBox( this,
                                             SWID_CANVAS_DRAW_DELAUNAY_TRIANGLE,
                                             wxT( "DelaunayTrianglation" ) );
        M_show_delaunay_cb->SetValue( M_view_config.isShownDelaunayTrianglation() );
        sizer->Add( M_show_delaunay_cb, 0, wxALIGN_CENTER_VERTICAL  );
    }
    {
        wxString choices[] = { wxT( "All" ), wxT( "Left Team" ), wxT( "Right Team" ) };
        M_voronoi_choice = new wxChoice( this,
                                         SWID_VORONOI_CHOICE,
                                         wxDefaultPosition,
                                         wxDefaultSize,
                                         WXSIZEOF( choices ),
                                         choices );
        sizer->Add( M_voronoi_choice, 0, wxLEFT, 5 );
        M_voronoi_choice->SetSelection( 0 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createFocusControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Focus" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_focus_ball_rb = new wxRadioButton( this,
                                             SWID_CANVAS_FOCUS_BALL,
                                             wxT( "Ball" ),
                                             wxDefaultPosition,
                                             wxDefaultSize,
                                             wxRB_GROUP );
        M_focus_ball_rb->SetValue( false );
        sizer->Add( M_focus_ball_rb );
    }
    {
        M_focus_player_rb = new wxRadioButton( this,
                                               SWID_CANVAS_FOCUS_PLAYER,
                                               wxT( "Selected Player" ) );
        M_focus_player_rb->SetValue( false );
        sizer->Add( M_focus_player_rb );
    }
    {
        M_focus_fix_rb = new wxRadioButton( this,
                                            SWID_CANVAS_FOCUS_FIX,
                                            wxT( "Fix" ) );
        M_focus_fix_rb->SetValue( true );
        sizer->Add( M_focus_fix_rb );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createPlayerSelectionControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Player Selection" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxVERTICAL );
    // selection type
    {
        wxBoxSizer * bsizer = new wxBoxSizer( wxHORIZONTAL );
        {
            M_select_all_rb = new wxRadioButton( this,
                                                 SWID_CANVAS_SELECT_AUTO_ALL,
                                                 wxT( "Auto" ),
                                                 wxDefaultPosition,
                                                 wxDefaultSize,
                                                 wxRB_GROUP );
            M_select_all_rb->SetValue( false );
            bsizer->Add( M_select_all_rb );
        }
        {
            M_select_left_rb = new wxRadioButton( this,
                                                  SWID_CANVAS_SELECT_AUTO_LEFT,
                                                  wxT( "Auto left" ) );
            M_select_left_rb->SetValue( false );
            bsizer->Add( M_select_left_rb );
        }
        {
            M_select_right_rb = new wxRadioButton( this,
                                                   SWID_CANVAS_SELECT_AUTO_RIGHT,
                                                   wxT( "Auto right" ) );
            M_select_right_rb->SetValue( false );
            bsizer->Add( M_select_right_rb );
        }
        {
            M_select_fix_rb = new wxRadioButton( this,
                                                 SWID_CANVAS_SELECT_FIX,
                                                 wxT( "Fix" ) );
            M_select_fix_rb->SetValue( false );
            bsizer->Add( M_select_fix_rb );
        }
        {
            M_unselect_rb = new wxRadioButton( this,
                                               SWID_CANVAS_UNSELECT_PLAYER,
                                               wxT( "Unselect" ) );
            M_unselect_rb->SetValue( true );
            bsizer->Add( M_unselect_rb );
        }
        sizer->Add( bsizer, 0, wxLEFT | wxRIGHT, 4 );
    }

    {
        wxString choices[] = { wxT( "No Selection" ),
                               wxT( "Left number: 1" ),
                               wxT( "Left number: 2" ),
                               wxT( "Left number: 3" ),
                               wxT( "Left number: 4" ),
                               wxT( "Left number: 5" ),
                               wxT( "Left number: 6" ),
                               wxT( "Left number: 7" ),
                               wxT( "Left number: 8" ),
                               wxT( "Left number: 9" ),
                               wxT( "Left number: 10" ),
                               wxT( "Left number: 11" ),
                               wxT( "Right number: 1" ),
                               wxT( "Right number: 2" ),
                               wxT( "Right number: 3" ),
                               wxT( "Right number: 4" ),
                               wxT( "Right number: 5" ),
                               wxT( "Right number: 6" ),
                               wxT( "Right number: 7" ),
                               wxT( "Right number: 8" ),
                               wxT( "Right number: 9" ),
                               wxT( "Right number: 10" ),
                               wxT( "Right number: 11" )
        };
        M_player_choice = new wxChoice( this,
                                        SWID_CONFIG_PLAYER_CHOICE,
                                        wxDefaultPosition,
                                        wxDefaultSize,
                                        WXSIZEOF( choices ),
                                        choices );
        M_player_choice->SetSelection( 0 );
        sizer->Add( M_player_choice, 0, wxLEFT );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createGrassTypeControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Grass Type" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        M_keepaway_mode_cb = new wxCheckBox( this,
                                             SWID_CANVAS_KEEPAWAY_MODE,
                                             wxT( "Keepaway" ) );
        M_keepaway_mode_cb->SetValue( M_view_config.keepawayMode() );
        sizer->Add( M_keepaway_mode_cb );
    }
    {
        M_grass_normal_rb = new wxRadioButton( this,
                                               SWID_CANVAS_GRASS_NORMAL,
                                               wxT( "mono" ),
                                               wxDefaultPosition,
                                               wxDefaultSize,
                                               wxRB_GROUP );
        M_grass_normal_rb->SetValue( M_view_config.grassType()
                                     == ViewConfig::GRASS_NORMAL );
        sizer->Add( M_grass_normal_rb );
    }
    {
        M_grass_line_rb = new wxRadioButton( this,
                                             SWID_CANVAS_GRASS_LINES,
                                             wxT( "lines" ) );
        M_grass_line_rb->SetValue( M_view_config.grassType()
                                   == ViewConfig::GRASS_LINES );
        sizer->Add( M_grass_line_rb );
    }
    {
        M_grass_checker_rb = new wxRadioButton( this,
                                                SWID_CANVAS_GRASS_CHECKER,
                                                wxT( "checker" ) );
        M_grass_checker_rb->SetValue( M_view_config.grassType()
                                      == ViewConfig::GRASS_CHECKER );
        sizer->Add( M_grass_checker_rb );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createTraceControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Trace" ) );
    wxStaticBoxSizer * main_sizer = new wxStaticBoxSizer( sbox, wxVERTICAL );

    int text_all_width, text_all_height;
    this->GetTextExtent( wxT( "All" ), &text_all_width, &text_all_height );

    wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );
    {
        wxStaticBox * bsbox = new wxStaticBox( this, -1,
                                               wxT( "Ball" ) );
        wxStaticBoxSizer * bsizer = new wxStaticBoxSizer( bsbox, wxHORIZONTAL );

        M_ball_trace_start = new wxTextCtrl( this, SWID_TEXTBOX_BALL_TRACE_START,
                                             wxT( "0" ),
                                             wxDefaultPosition, wxSize( 40, -1 ) );
        bsizer->Add( M_ball_trace_start,
                     0, 0, 0 );

        bsizer->Add( new wxStaticText( this, -1, wxT( "-" ) ),
                     0, wxLEFT | wxRIGHT, 1 );

        M_ball_trace_end = new wxTextCtrl( this, SWID_TEXTBOX_BALL_TRACE_END,
                                           wxT( "0" ),
                                           wxDefaultPosition, wxSize( 40, -1 ) );
        bsizer->Add( M_ball_trace_end,
                     0, 0, 0 );
        bsizer->Add( new wxButton( this, SWID_CANVAS_BALL_TRACE_SELECT_ALL,
                                   wxT( "All" ),
                                   wxDefaultPosition,
                                   wxSize( text_all_width + 12, -1 ) ),
                     0, wxALIGN_CENTER_VERTICAL | wxRIGHT, 2 );

        M_ball_trace_cb = new wxCheckBox( this,
                                          SWID_CANVAS_TOGGLE_BALL_AUTO_TRACE,
                                          wxT( "Auto" ) );
        M_ball_trace_cb->SetValue( false );
        bsizer->Add( M_ball_trace_cb, 1, 0, 0 );

        sizer->Add( bsizer,
                    0, wxLEFT | wxRIGHT, 2 );
    }
    {
        wxStaticBox * psbox = new wxStaticBox( this, -1,
                                               wxT( "Player" ) );
        wxStaticBoxSizer * psizer = new wxStaticBoxSizer( psbox, wxHORIZONTAL );

        M_player_trace_start = new wxTextCtrl( this, SWID_TEXTBOX_PLAYER_TRACE_START,
                                               wxT( "0" ),
                                               wxDefaultPosition, wxSize( 40, -1 ) );
        psizer->Add( M_player_trace_start,
                     0, 0, 0 );

        psizer->Add( new wxStaticText( this, -1, wxT( "-" ) ),
                     0, wxLEFT | wxRIGHT, 1 );

        M_player_trace_end = new wxTextCtrl( this, SWID_TEXTBOX_PLAYER_TRACE_END,
                                             wxT( "0" ),
                                             wxDefaultPosition, wxSize( 40, -1 ) );
        psizer->Add( M_player_trace_end,
                     0, 0, 0 );

        psizer->Add( new wxButton( this, SWID_CANVAS_PLAYER_TRACE_SELECT_ALL,
                                   wxT( "All" ),
                                   wxDefaultPosition,
                                   wxSize( text_all_width + 12, -1 ) ),
                     0, wxALIGN_CENTER_VERTICAL | wxRIGHT, 2 );

        M_player_trace_cb =  new wxCheckBox( this,
                                             SWID_CANVAS_TOGGLE_PLAYER_AUTO_TRACE,
                                             wxT( "Auto" ) );
        M_player_trace_cb->SetValue( false );
        psizer->Add( M_player_trace_cb,
                     0, wxALIGN_CENTER_VERTICAL | wxLEFT | wxRIGHT, 2 );

        sizer->Add( psizer,
                    0, wxLEFT | wxRIGHT, 2 );
    }

    main_sizer->Add( sizer );
    main_sizer->Add( 6, 0 );

    wxBoxSizer * sizer2 = new wxBoxSizer( wxHORIZONTAL );
    {
        wxStaticBox * trace_box = new wxStaticBox( this, -1, wxT( "Auto Trace Period" ) );
        wxStaticBoxSizer * tsizer = new wxStaticBoxSizer( trace_box, wxHORIZONTAL );

        tsizer->Add( new wxStaticText( this, -1, wxT( "Start:" ) ),
                     0, wxLEFT | wxRIGHT | wxALIGN_CENTER_VERTICAL, 1 );

        M_auto_trace_start = new wxTextCtrl( this, SWID_TEXTBOX_AUTO_TRACE_START,
                                             wxT( "-10" ),
                                             wxDefaultPosition, wxSize( 40, -1 ) );
        tsizer->Add( M_auto_trace_start,
                     0, 0, 0 );

        tsizer->Add( 2, 0 );

        tsizer->Add( new wxStaticText( this, -1, wxT( "Period:" ) ),
                     0, wxLEFT | wxRIGHT | wxALIGN_CENTER_VERTICAL, 1 );

        M_auto_trace_period = new wxSpinCtrl( this, SWID_TEXTBOX_AUTO_TRACE_PERIOD );
        M_auto_trace_period->SetValue( 20 );
        M_auto_trace_period->SetRange( 1, 12000 );
        tsizer->Add( M_auto_trace_period,
                     0, 0, 0 );

        sizer2->Add( tsizer );
    }
    sizer2->Add( 12, 0 );
    {
        sizer2->Add( new wxButton( this, SWID_CANVAS_TOGGLE_TRACE_SHAPE,
                                   wxT( "Line/Point" ) ),
                     0, wxALIGN_CENTER_VERTICAL | wxCENTER, 2 );
    }

    main_sizer->Add( sizer2 );

    return main_sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ViewConfigDialog::createInertiaMoveControls()
{
    wxStaticBox * sbox = new wxStaticBox( this, -1,
                                          wxT( "Inertia Move" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );
    {
        wxStaticBox * bsbox = new wxStaticBox( this, -1,
                                               wxT( "Ball" ) );
        wxStaticBoxSizer * bsizer = new wxStaticBoxSizer( bsbox, wxHORIZONTAL );

        M_ball_future = new wxSpinCtrl( this, SWID_TEXTBOX_BALL_FUTURE_CYCLE );
        M_ball_future->SetValue( 0 );
        M_ball_future->SetRange( 0, 50 );

        bsizer->Add( M_ball_future,
                     0, 0, 0 );

        sizer->Add( bsizer,
                    0, wxLEFT | wxRIGHT, 2 );
    }
    {
        wxStaticBox * psbox = new wxStaticBox( this, -1,
                                               wxT( "Player" ) );
        wxStaticBoxSizer * psizer = new wxStaticBoxSizer( psbox, wxHORIZONTAL );

        M_player_future = new wxSpinCtrl( this, SWID_TEXTBOX_PLAYER_FUTURE_CYCLE );
        M_player_future->SetValue( 0 );
        M_player_future->SetRange( 0, 10 );

        psizer->Add( M_player_future,
                     0, 0, 0 );

        sizer->Add( psizer,
                    0, wxLEFT | wxRIGHT, 2 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::connectEvents()
{
    // close frame
    Connect( wxID_ANY, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( ViewConfigDialog::handleCloseEvent ) );

    // zoom actions
    Connect( SWID_CANVAS_ZOOMIN, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickZoomIn ) );
    Connect( SWID_CANVAS_ZOOMOUT, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickZoomOut ) );
    Connect( SWID_CANVAS_UNZOOM, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickFit ) );
    Connect( SWID_CANVAS_ENLARGE, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickEnlarge ) );
    Connect( SWID_FIELD_SCALE_SLIDER, wxEVT_SCROLL_THUMBTRACK,
             wxScrollEventHandler( ViewConfigDialog::slideFieldScale ) );
    // canvas size actions
    Connect( SWID_CANVAS_SET_SIZE_BUTTON, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickSetCanvasSize ) );
    // show/hide actions
    Connect( SWID_CANVAS_DRAW_PLAYER_NUMBER, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowPlayerNumber ) );
    Connect( SWID_CANVAS_DRAW_HETERO_NUMBER, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowHeteroNumber ) );
    Connect( SWID_CANVAS_DRAW_STAMINA, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowStamina ) );
    Connect( SWID_CANVAS_DRAW_VIEW_CONE, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowViewCone ) );
    Connect( SWID_CANVAS_DRAW_CONTROL_AREA, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowControlArea ) );

    Connect( SWID_CANVAS_DRAW_SCORE_BOARD, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowScoreBoard ) );
    Connect( SWID_CANVAS_DRAW_BALL, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowBall ) );
    Connect( SWID_CANVAS_DRAW_PLAYER, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowPlayers ) );
    Connect( SWID_CANVAS_DRAW_FLAGS, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowFlags ) );
    Connect( SWID_CANVAS_DRAW_OFFSIDE_LINE, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowOffsideLine ) );

    // grass type
    Connect( SWID_CANVAS_GRASS_NORMAL, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickGrassNormal ) );
    Connect( SWID_CANVAS_GRASS_LINES, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickGrassLine ) );
    Connect( SWID_CANVAS_GRASS_CHECKER, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickGrassChecker ) );

    Connect( SWID_CANVAS_KEEPAWAY_MODE, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickKeepawayMode ) );

    // computational geometry
    Connect( SWID_CANVAS_DRAW_VORONOI_DIAGRAM, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowVoronoiDiagram ) );
    Connect( SWID_CANVAS_DRAW_DELAUNAY_TRIANGLE, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickShowDelaunayTrianglation ) );
    Connect( SWID_VORONOI_CHOICE, wxEVT_COMMAND_CHOICE_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::selectCompGeomSide ) );

    // focus type
    Connect( SWID_CANVAS_FOCUS_BALL, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickFocusBall ) );
    Connect( SWID_CANVAS_FOCUS_PLAYER, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickFocusPlayer ) );
    Connect( SWID_CANVAS_FOCUS_FIX, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickFocusFix ) );

    // player selection type
    Connect( SWID_CANVAS_SELECT_AUTO_ALL, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickSelectAutoAll ) );
    Connect( SWID_CANVAS_SELECT_AUTO_LEFT, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickSelectAutoLeft ) );
    Connect( SWID_CANVAS_SELECT_AUTO_RIGHT, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickSelectAutoRight ) );
    Connect( SWID_CANVAS_SELECT_FIX, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickSelectFix ) );
    Connect( SWID_CANVAS_UNSELECT_PLAYER, wxEVT_COMMAND_RADIOBUTTON_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::clickUnselect ) );

    // selectPlayer
    Connect( SWID_CONFIG_PLAYER_CHOICE, wxEVT_COMMAND_CHOICE_SELECTED,
             wxCommandEventHandler( ViewConfigDialog::selectPlayer ) );

    // trace control actions
    // ball trace
    Connect( SWID_TEXTBOX_BALL_TRACE_START, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::editBallTraceStart ) );
    Connect( SWID_TEXTBOX_BALL_TRACE_END, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::editBallTraceEnd ) );
    Connect( SWID_CANVAS_BALL_TRACE_SELECT_ALL, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickBallTraceAll ) );
    Connect( SWID_CANVAS_TOGGLE_BALL_AUTO_TRACE, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickAutoBallTrace ) );
    // player trace
    Connect( SWID_TEXTBOX_PLAYER_TRACE_START, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::editPlayerTraceStart ) );
    Connect( SWID_TEXTBOX_PLAYER_TRACE_END, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::editPlayerTraceEnd ) );
    Connect( SWID_CANVAS_PLAYER_TRACE_SELECT_ALL, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickPlayerTraceAll ) );
    Connect( SWID_CANVAS_TOGGLE_PLAYER_AUTO_TRACE, wxEVT_COMMAND_CHECKBOX_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickAutoPlayerTrace ) );
    // trace period
    Connect( SWID_TEXTBOX_AUTO_TRACE_START, wxEVT_COMMAND_TEXT_UPDATED,
             wxTextEventHandler( ViewConfigDialog::editAutoTraceStart ) );
    Connect( SWID_TEXTBOX_AUTO_TRACE_PERIOD, wxEVT_COMMAND_SPINCTRL_UPDATED,
             wxSpinEventHandler( ViewConfigDialog::changeAutoTracePeriod ) );
    // trace line type
    Connect( SWID_CANVAS_TOGGLE_TRACE_SHAPE, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ViewConfigDialog::clickLinePointButton ) );

    // inertia trace actions
    Connect( SWID_TEXTBOX_BALL_FUTURE_CYCLE, wxEVT_COMMAND_SPINCTRL_UPDATED,
             wxSpinEventHandler( ViewConfigDialog::changeBallFutureCycle ) );
    Connect( SWID_TEXTBOX_PLAYER_FUTURE_CYCLE, wxEVT_COMMAND_SPINCTRL_UPDATED,
             wxSpinEventHandler( ViewConfigDialog::changePlayerFutureCycle ) );
    //Connect( SWID_TEXTBOX_BALL_FUTURE_CYCLE, wxEVT_COMMAND_TEXT_UPDATED,
    //         wxTextEventHandler( ViewConfigDialog::handleBallFutureCycle ) );
    //Connect( SWID_TEXTBOX_PLAYER_FUTURE_CYCLE, wxEVT_COMMAND_TEXT_UPDATED,
    //         wxTextEventHandler( ViewConfigDialog::handlePlayerFutureCycle ) );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    this->Hide();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::updateAll()
{
    wxString buf;

    M_enlarge_cb->SetValue( M_view_config.isEnlarged() );

    updateFieldScale();

    buf.Printf( wxT( "%d" ), M_view_config.canvasWidth() );
    M_canvas_width_text->SetValue( buf );

    buf.Printf( wxT( "%d" ), M_view_config.canvasHeight() );
    M_canvas_height_text->SetValue( buf );

    M_player_number_cb->SetValue( M_view_config.isShownPlayerNumber() );
    M_hetero_number_cb->SetValue( M_view_config.isShownHeteroNumber() );
    M_stamina_cb->SetValue( M_view_config.isShownStamina() );
    M_view_cone_cb->SetValue( M_view_config.isShownViewCone() );
    M_control_area_cb->SetValue( M_view_config.isShownControlArea() );

    M_show_score_board_cb->SetValue( M_view_config.isShownScoreBoard() );
    M_show_ball_cb->SetValue( M_view_config.isShownBall() );
    M_show_players_cb->SetValue( M_view_config.isShownPlayers() );
    M_show_flags_cb->SetValue( M_view_config.isShownFlags() );
    M_show_offside_line_cb->SetValue( M_view_config.isShownOffsideLine() );

    switch ( M_view_config.grassType() ) {
    case ViewConfig::GRASS_NORMAL:
        M_grass_normal_rb->SetValue( true );
        break;
    case ViewConfig::GRASS_LINES:
        M_grass_line_rb->SetValue( true );
        break;
    case ViewConfig::GRASS_CHECKER:
        M_grass_checker_rb->SetValue( true );
        break;
    default:
        M_grass_normal_rb->SetValue( true );
        break;
    }

    M_show_voronoi_cb->SetValue( M_view_config.isShownVoronoiDiagram() );
    M_show_delaunay_cb->SetValue( M_view_config.isShownDelaunayTrianglation() );

    switch ( M_view_config.voronoiTarget() ) {
    case rcsc::NEUTRAL:
        M_voronoi_choice->SetSelection( 0 );
        break;
    case rcsc::LEFT:
        M_voronoi_choice->SetSelection( 1 );
        break;
    case rcsc::RIGHT:
        M_voronoi_choice->SetSelection( 2 );
        break;
    default:
        M_voronoi_choice->SetSelection( 0 );
        break;
    }

    switch( M_view_config.focusType() ) {
    case ViewConfig::FOCUS_BALL:
        M_focus_ball_rb->SetValue( true );
        break;
    case ViewConfig::FOCUS_PLAYER:
        M_focus_player_rb->SetValue( true );
        break;
    case ViewConfig::FOCUS_POINT:
        M_focus_fix_rb->SetValue( true );
        break;
    default:
        M_focus_fix_rb->SetValue( true );
        break;
    }

    switch ( M_view_config.playerSelectType() ) {
    case ViewConfig::SELECT_AUTO_ALL:
        M_select_all_rb->SetValue( true );
        break;
    case ViewConfig::SELECT_AUTO_LEFT:
        M_select_left_rb->SetValue( true );
        break;
    case ViewConfig::SELECT_AUTO_RIGHT:
        M_select_right_rb->SetValue( true );
        break;
    case ViewConfig::SELECT_UNSELECT:
        M_unselect_rb->SetValue( true );
        break;
    default:
        M_unselect_rb->SetValue( true );
        break;
    }

    buf.Printf( wxT( "%d" ), M_view_config.ballTraceStart() );
    M_ball_trace_start->SetValue( buf );

    buf.Printf( wxT( "%d" ), M_view_config.ballTraceEnd() );
    M_ball_trace_end->SetValue( buf );

    M_ball_trace_cb->SetValue( M_view_config.isBallAutoTrace() );

    buf.Printf( wxT( "%d" ), M_view_config.playerTraceStart() );
    M_player_trace_start->SetValue( buf );

    buf.Printf( wxT( "%d" ), M_view_config.playerTraceEnd() );
    M_player_trace_end->SetValue( buf );

    M_player_trace_cb->SetValue( M_view_config.isPlayerAutoTrace() );

    buf.Printf( wxT( "%d" ), M_view_config.autoTraceStart() );
    M_auto_trace_start->SetValue( buf );
    M_auto_trace_period->SetValue( M_view_config.autoTracePeriod() );;

    M_ball_future->SetValue( M_view_config.ballFutureCycle() );
    M_player_future->SetValue( M_view_config.playerFutureCycle() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::updateFieldScale()
{
    M_scale_slider->SetValue( static_cast< int >
                              ( M_view_config.fieldScale() * 10.0 ) );
    wxString buf;
    buf.Printf( wxT( "%.1f" ), M_view_config.fieldScale() );
    M_scale_text->SetLabel( buf );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickZoomIn( wxCommandEvent & WXUNUSED( event ) )
{
    zoomIn();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::zoomIn()
{
    M_view_config.zoomIn();

    updateFieldScale();

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickZoomOut( wxCommandEvent & WXUNUSED( event ) )
{
    zoomOut();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::zoomOut()
{
    M_view_config.zoomOut();

    updateFieldScale();

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFit( wxCommandEvent & WXUNUSED( event ) )
{
    unzoom();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::unzoom()
{
    M_view_config.unzoom();
    updateFieldScale();

    M_focus_fix_rb->SetValue( true );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickEnlarge( wxCommandEvent & event )
{
    if ( M_view_config.isEnlarged() != event.IsChecked() )
    {
        M_view_config.toggleEnlarge();

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleEnlarge()
{
    M_view_config.toggleEnlarge();
    M_enlarge_cb->SetValue( M_view_config.isEnlarged() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::slideFieldScale( wxScrollEvent & WXUNUSED( event ) )
{
    double scale = M_scale_slider->GetValue() * 0.1;

    wxString label;
    label.Printf(  wxT( "%5.1f" ), scale );
    M_scale_text->SetLabel( label );

    M_view_config.setFieldScale( scale );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSetCanvasSize( wxCommandEvent & WXUNUSED( event ) )
{
    std::cerr << "clickSetCanvasSize" << std::endl;
    applyCanvasSize();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::applyCanvasSize()
{
    long width = 0;
    long height = 0;

    if ( M_canvas_width_text->GetValue().ToLong( &width )
         && M_canvas_height_text->GetValue().ToLong( &height )
         && width > 0
         && height > 0 )
    {
        emit_canvasResized( wxSize( static_cast< int >( width ),
                                    static_cast< int >( height ) ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowPlayerNumber( wxCommandEvent & event )
{
    if ( M_view_config.isShownPlayerNumber() != event.IsChecked() )
    {
        M_view_config.toggleShowPlayerNumber();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowPlayerNumber()
{
    M_view_config.toggleShowPlayerNumber();
    M_player_number_cb->SetValue( M_view_config.isShownPlayerNumber() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowHeteroNumber( wxCommandEvent & event )
{
    if ( M_view_config.isShownHeteroNumber() != event.IsChecked() )
    {
        M_view_config.toggleShowHeteroNumber();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowHeteroNumber()
{
    M_view_config.toggleShowHeteroNumber();
    M_hetero_number_cb->SetValue( M_view_config.isShownHeteroNumber() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowStamina( wxCommandEvent & event )
{
    if ( M_view_config.isShownStamina() != event.IsChecked() )
    {
        M_view_config.toggleShowStamina();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowStamina()
{
    M_view_config.toggleShowStamina();
    M_stamina_cb->SetValue( M_view_config.isShownStamina() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowViewCone( wxCommandEvent & event )
{
    if ( M_view_config.isShownViewCone() != event.IsChecked() )
    {
        M_view_config.toggleShowViewCone();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowViewCone()
{
    M_view_config.toggleShowViewCone();
    M_view_cone_cb->SetValue( M_view_config.isShownViewCone() );

    emit_configured();
}


/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowControlArea( wxCommandEvent & event )
{
    if ( M_view_config.isShownControlArea() != event.IsChecked() )
    {
        M_view_config.toggleShowControlArea();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowControlArea()
{
    M_view_config.toggleShowControlArea();
    M_control_area_cb->SetValue( M_view_config.isShownControlArea() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowScoreBoard( wxCommandEvent & event )
{
    if ( M_view_config.isShownScoreBoard() != event.IsChecked() )
    {
        M_view_config.toggleShowScoreBoard();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowScoreBoard()
{
    M_view_config.toggleShowScoreBoard();
    M_show_score_board_cb->SetValue( M_view_config.isShownScoreBoard() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowBall( wxCommandEvent & event )
{
    if ( M_view_config.isShownBall() != event.IsChecked() )
    {
        M_view_config.toggleShowBall();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowBall()
{
    M_view_config.toggleShowBall();
    M_show_ball_cb->SetValue( M_view_config.isShownBall() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowPlayers( wxCommandEvent & event )
{
    if ( M_view_config.isShownPlayers() != event.IsChecked() )
    {
        M_view_config.toggleShowPlayers();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowPlayers()
{
    M_view_config.toggleShowPlayers();
    M_show_players_cb->SetValue( M_view_config.isShownPlayers() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowFlags( wxCommandEvent & event )
{
    if ( M_view_config.isShownFlags() != event.IsChecked() )
    {
        M_view_config.toggleShowFlags();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowFlags()
{
    M_view_config.toggleShowFlags();
    M_show_flags_cb->SetValue( M_view_config.isShownFlags() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowOffsideLine( wxCommandEvent & event )
{
    if ( M_view_config.isShownOffsideLine() != event.IsChecked() )
    {
        M_view_config.toggleShowOffsideLine();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowOffsideLine()
{
    M_view_config.toggleShowOffsideLine();
    M_show_offside_line_cb->SetValue( M_view_config.isShownOffsideLine() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassNormal( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_NORMAL )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_NORMAL );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassLine( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_LINES )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_LINES );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassChecker( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_CHECKER )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_CHECKER );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickKeepawayMode( wxCommandEvent & event )
{
    if ( M_view_config.keepawayMode() != event.IsChecked() )
    {
        M_view_config.toggleKeepawayMode();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleKeepawayMode()
{
    M_view_config.toggleKeepawayMode();
    M_keepaway_mode_cb->SetValue( M_view_config.keepawayMode() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowVoronoiDiagram( wxCommandEvent & event )
{
    if ( M_view_config.isShownVoronoiDiagram() != event.IsChecked() )
    {
        M_view_config.toggleShowVoronoiDiagram();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowDelaunayTrianglation( wxCommandEvent & event )
{
    if ( M_view_config.isShownDelaunayTrianglation() != event.IsChecked() )
    {
        M_view_config.toggleShowDelaunayTrianglation();
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectCompGeomSide( wxCommandEvent & WXUNUSED( event ) )
{
    switch ( M_voronoi_choice->GetSelection() ) {
    case 0:
        if ( M_view_config.voronoiTarget() != rcsc::NEUTRAL )
        {
            M_view_config.setVoronoiTarget( rcsc::NEUTRAL );
            emit_configured();
        }
        break;
    case 1:
        if ( M_view_config.voronoiTarget() != rcsc::LEFT )
        {
            M_view_config.setVoronoiTarget( rcsc::LEFT );
            emit_configured();
        }
        break;
    case 2:
        if ( M_view_config.voronoiTarget() != rcsc::RIGHT )
        {
            M_view_config.setVoronoiTarget( rcsc::RIGHT );
            emit_configured();
        }
        break;
    default:
        std::cerr << "ViewConfigDialog::selectCompGeomSide() unsupported index = "
                  << M_voronoi_choice->GetSelection()
                  << std::endl;
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusBall( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_BALL )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_BALL );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleFocusBall()
{
    if ( M_view_config.focusType() == ViewConfig::FOCUS_BALL )
    {
        setFocusFix();
    }
    else
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_BALL );
        M_focus_ball_rb->SetValue( true );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusPlayer( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_PLAYER )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_PLAYER );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleFocusPlayer()
{
    if ( M_view_config.focusType() == ViewConfig::FOCUS_PLAYER )
    {
        setFocusFix();
    }
    else
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_PLAYER );
        M_focus_player_rb->SetValue( true );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusFix( wxCommandEvent & WXUNUSED( event ) )
{
    setFocusFix();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusFix()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_POINT )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_POINT );
        M_focus_fix_rb->SetValue( true );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusPoint( const wxPoint & point )
{
    M_view_config.setFocusPoint( point.x, point.y );
    M_focus_fix_rb->SetValue( true );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoAll( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_AUTO_ALL )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoAll()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_AUTO_ALL )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
        M_select_all_rb->SetValue( true );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoLeft( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_AUTO_LEFT )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_LEFT );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoLeft()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_AUTO_LEFT )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_LEFT );
        M_select_left_rb->SetValue( true );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoRight( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_AUTO_RIGHT )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoRight()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_AUTO_RIGHT )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        M_select_right_rb->SetValue( true );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectFix( wxCommandEvent & WXUNUSED( event ) )
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_FIX )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectFix()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_FIX )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_FIX );
        M_select_fix_rb->SetValue( true );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickUnselect( wxCommandEvent & WXUNUSED( event ) )
{
    setUnselect();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setUnselect()
{
    M_view_config.unselectPlayer();

    M_unselect_rb->SetValue( true );

    M_player_choice->SetSelection( 0 );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectPlayer( wxCommandEvent & event )
{
    selectPlayerNumber( event.GetSelection() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectPlayerNumber( int number )
{
    // left side:  number = [1,11]
    // right side:  number = [12,22]

    if ( number <= 0
         || 22 < number )
    {
        setUnselect();
        return;
    }

    M_view_config.setPlayerSelectType( ViewConfig::SELECT_FIX );
    if ( number <= 11 )
    {
        M_view_config.setSelectedNumber( rcsc::LEFT, number );
    }
    else
    {
        M_view_config.setSelectedNumber( rcsc::RIGHT, number - 11 );
    }

    M_select_fix_rb->SetValue( true );

    M_player_choice->SetSelection( number );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallTraceStart( wxCommandEvent & WXUNUSED( event ) )
{
    long value = 0;
    if ( M_ball_trace_start->GetValue().ToLong( &value ) )
    {
        std::cerr << "editBallTraceStart " << value << std::endl;
        M_view_config.setBallTraceStart( value );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallTraceEnd( wxCommandEvent & WXUNUSED( event ) )
{
    long value = 0;
    if ( M_ball_trace_end->GetValue().ToLong( &value ) )
    {
        std::cerr << "editBallTraceStart " << value << std::endl;
        M_view_config.setBallTraceEnd( value );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickBallTraceAll( wxCommandEvent & WXUNUSED( event ) )
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();
    if ( ! view.empty() )
    {
        wxString label;

        label.Printf( wxT( "%ld" ), view.front()->cycle() );
        M_ball_trace_start->SetValue( label );

        label.Printf( wxT( "%ld" ), view.back()->cycle() );
        M_ball_trace_end->SetValue( label );

        M_ball_trace_cb->SetValue( false );

        M_view_config.setBallTraceStart( view.front()->cycle() );
        M_view_config.setBallTraceEnd( view.back()->cycle() );
        if ( M_view_config.isBallAutoTrace() )
        {
            M_view_config.toggleBallAutoTrace();
        }

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAutoBallTrace( wxCommandEvent & event )
{
    if ( M_view_config.isBallAutoTrace() != event.IsChecked() )
    {
        M_view_config.toggleBallAutoTrace();

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerTraceStart( wxCommandEvent & WXUNUSED( event ) )
{
    long value = 0;
    if ( M_player_trace_start->GetValue().ToLong( &value ) )
    {
        std::cerr << "editPlayerTraceStart " << value << std::endl;
        M_view_config.setPlayerTraceStart( value );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerTraceEnd( wxCommandEvent & WXUNUSED( event ) )
{
    long value = 0;
    if ( M_player_trace_end->GetValue().ToLong( &value ) )
    {
        std::cerr << "editPlayerTraceEnd " << value << std::endl;
        M_view_config.setPlayerTraceEnd( value );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickPlayerTraceAll( wxCommandEvent & WXUNUSED( event ) )
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();
    if ( ! view.empty() )
    {
        wxString label;

        label.Printf( wxT( "%ld" ), view.front()->cycle() );
        M_player_trace_start->SetLabel( label );

        label.Printf( wxT( "%ld" ), view.back()->cycle() );
        M_player_trace_end->SetLabel( label );

        M_player_trace_cb->SetValue( false );

        M_view_config.setPlayerTraceStart( view.front()->cycle() );
        M_view_config.setPlayerTraceEnd( view.back()->cycle() );
        if ( M_view_config.isPlayerAutoTrace() )
        {
            M_view_config.togglePlayerAutoTrace();
        }

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAutoPlayerTrace( wxCommandEvent & event )
{
    if ( M_view_config.isPlayerAutoTrace() != event.IsChecked() )
    {
        M_view_config.togglePlayerAutoTrace();

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editAutoTraceStart( wxCommandEvent & WXUNUSED( event ) )
{
    long value = 0;
    if ( M_auto_trace_start->GetValue().ToLong( &value ) )
    {
        std::cerr << "editAutTraceStart " << value << std::endl;
        M_view_config.setAutoTraceStart( value );

        emit_configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changeAutoTracePeriod( wxSpinEvent & WXUNUSED( event ) )
{
    M_view_config.setAutoTracePeriod( M_auto_trace_period->GetValue() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickLinePointButton( wxCommandEvent & WXUNUSED( event ) )
{
    M_view_config.toggleLineTrace();

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changeBallFutureCycle( wxSpinEvent & WXUNUSED( event ) )
{
    M_view_config.setBallFutureCycle( M_ball_future->GetValue() );

    emit_configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changePlayerFutureCycle( wxSpinEvent & WXUNUSED( event ) )
{
    M_view_config.setPlayerFutureCycle( M_player_future->GetValue() );

    emit_configured();
}
