// -*-c++-*-

/*!
	\file feditor_canvas.cpp
	\brief formation editor canvas class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "fedit_canvas.h"

#include "main_window.h"
#include "fedit_data.h"
#include "fedit_config.h"

#include <rcsc/formation/formation_dt.h>

#include <rcsc/geom/triangle_2d.h>

#include <algorithm>
#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
FEditCanvas::FEditCanvas( MainWindow * parent )
    : QWidget( parent, 0, Qt::WNoAutoErase )
    , M_scale( 1.0 )
    , M_center( 0, 0 )
    , M_field_brush( QColor( 31, 160, 31 ), Qt::SolidPattern )
    , M_field_dark_brush( QColor( 15, 143, 15 ), Qt::SolidPattern )
    , M_line_pen( QColor( 255, 255, 255 ), 1, Qt::SolidLine )
    , M_triangle_pen( QColor( 255, 127, 0 ), 1, Qt::SolidLine )
    , M_triangle_font( "Sans Serif", 10 )
    , M_area_pen( QColor( 127, 127, 127 ), 1, Qt::SolidLine )
    , M_ball_pen( QColor( 255, 255, 255 ), 1, Qt::SolidLine )
    , M_ball_brush( QColor( 255, 255, 255 ), Qt::SolidPattern )
    , M_player_pen( QColor( 0, 0, 0 ), 1, Qt::SolidLine )
    , M_select_pen( Qt::white, 2, Qt::SolidLine )
    , M_left_team_brush( QColor( 255, 215, 0 ), Qt::SolidPattern )
    , M_right_team_brush( QColor( 240, 20, 20 ), Qt::SolidPattern )
    , M_mirror_brush( QColor( 0, 255, 95 ), Qt::SolidPattern )
    , M_player_font( "Sans Serif", 10 )
{
    assert( parent );

    // need for the MouseMoveEvent
    this->setMouseTracking( true );

    this->setFocusPolicy( QWidget::WheelFocus );

    updateSize();

    M_canvas_pixmap.resize( this->size() );
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditCanvas::~FEditCanvas()
{
    //std::cerr << "delete FEditCanvas" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::updateSize()
{
    M_center.setX( this->width() / 2 );
    M_center.setY( this->height() / 2 );

    const double pitch_l = ( FEditConfig::PITCH_LENGTH
                             + FEditConfig::PITCH_MARGIN * 2.0 );
    const double pitch_w = ( FEditConfig::PITCH_WIDTH
                             + FEditConfig::PITCH_MARGIN * 2.0 );

    M_scale = this->width() / pitch_l;
    if ( pitch_w * M_scale > this->height() )
    {
        M_scale = this->height() / pitch_w;
    }

    M_scale = std::max( 2.0, M_scale );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::paintEvent( QPaintEvent * event )
{
    QWidget::paintEvent( event );

    if ( M_canvas_pixmap.isNull()
         || M_canvas_pixmap.size() != this->size() )
    {
        M_canvas_pixmap.resize( this->size() );
        updateSize();
    }

    QPainter painter;
    painter.begin( &M_canvas_pixmap, this );

    drawField( painter );
    drawTrainingData( painter );
    drawPlayers( painter );
    drawBall( painter );

    painter.end();

    bitBlt( this, 0, 0,
            &M_canvas_pixmap, 0, 0, this->width(), this->height() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawField( QPainter & painter )
{
    //--------------------------------------
    // fill background
    painter.fillRect( painter.window(), M_field_brush );

    //--------------------------------------
    drawContainedArea( painter );

    // set screen coordinates of field
    const int left_x   = getScreenX( - FEditConfig::PITCH_LENGTH * 0.5 );
    const int right_x  = getScreenX( + FEditConfig::PITCH_LENGTH * 0.5 );
    const int top_y    = getScreenY( - FEditConfig::PITCH_WIDTH * 0.5 );
    const int bottom_y = getScreenY( + FEditConfig::PITCH_WIDTH * 0.5 );

    //--------------------------------------
    // draw lines
    painter.setPen( M_line_pen );
    painter.setBrush( Qt::NoBrush );

    // side lines & goal lines
    painter.drawLine( left_x, top_y, right_x, top_y );
    painter.drawLine( right_x, top_y, right_x, bottom_y );
    painter.drawLine( right_x, bottom_y, left_x, bottom_y );
    painter.drawLine( left_x, bottom_y, left_x, top_y );
    // center line
    painter.drawLine( M_center.x(), top_y, M_center.x(), bottom_y );
    // center circle
    const int center_radius = scale( FEditConfig::CENTER_CIRCLE_R );
    painter.drawEllipse( M_center.x() - center_radius,
                         M_center.y() - center_radius,
                         center_radius * 2,
                         center_radius * 2 );
    // draw penalty area box
    const int pen_top_y    = getScreenY( - FEditConfig::PENALTY_AREA_WIDTH / 2 );
    const int pen_bottom_y = getScreenY( + FEditConfig::PENALTY_AREA_WIDTH / 2 );
    // left penalty box
    int pen_x = getScreenX( -( FEditConfig::PITCH_LENGTH*0.5
                                - FEditConfig::PENALTY_AREA_LENGTH ) );
    painter.drawLine( left_x, pen_top_y, pen_x, pen_top_y );
    painter.drawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    painter.drawLine( pen_x, pen_bottom_y, left_x, pen_bottom_y );
    // right penalty box
    pen_x = getScreenX( +( FEditConfig::PITCH_LENGTH*0.5
                        - FEditConfig::PENALTY_AREA_LENGTH ) );
    painter.drawLine( right_x, pen_top_y, pen_x, pen_top_y );
    painter.drawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    painter.drawLine( pen_x, pen_bottom_y, right_x, pen_bottom_y );
    // draw goal area box
    const int goal_area_y_abs = scale( FEditConfig::GOAL_AREA_WIDTH*0.5 );
    const int goal_area_top_y = M_center.y() - goal_area_y_abs;
    const int goal_area_bottom_y = M_center.y() + goal_area_y_abs;
    // left goal area
    int goal_area_x = getScreenX( - FEditConfig::PITCH_LENGTH*0.5
                               + FEditConfig::GOAL_AREA_LENGTH );
    painter.drawLine( left_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    painter.drawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    painter.drawLine( goal_area_x, goal_area_bottom_y, left_x, goal_area_bottom_y );
    // right goal area
    goal_area_x = getScreenX( FEditConfig::PITCH_LENGTH*0.5
                           - FEditConfig::GOAL_AREA_LENGTH );
    painter.drawLine( right_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    painter.drawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    painter.drawLine( goal_area_x, goal_area_bottom_y, right_x, goal_area_bottom_y );

    //--------------------------------------
    // draw goal boxes
    painter.setPen( Qt::black );
    painter.setBrush( Qt::black );
    const int goal_top_y = getScreenY( - FEditConfig::GOAL_WIDTH*0.5 );
    const int goal_size_x = scale( FEditConfig::GOAL_DEPTH );
    const int goal_size_y = scale( FEditConfig::GOAL_WIDTH );
    // left goal
    painter.drawRect( getScreenX( - FEditConfig::PITCH_LENGTH*0.5
                                  - FEditConfig::GOAL_DEPTH ),
                      goal_top_y, 
                      goal_size_x, goal_size_y );
    // right goal
    painter.drawRect( getScreenX( FEditConfig::PITCH_LENGTH*0.5 ) + 1,
                      goal_top_y,
                      goal_size_x + 1, goal_size_y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawContainedArea( QPainter & painter )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return ;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    if ( f->methodName() != rcsc::FormationDT::name() )
    {
        return;
    }

    if ( ptr->triangulation().vertices().size() < 3 )
    {
        return;
    }

    // draw polygon
    const rcsc::DelaunayTriangulation::TrianglePtr tri
        = ptr->triangulation().findTriangleContains( ptr->ball() );
    if ( tri )
    {
        painter.setPen( Qt::NoPen );
        painter.setBrush( M_field_dark_brush );

        QPointArray vertices( 3 );
        vertices.setPoint( 0,
                           getScreenX( tri->vertex( 0 )->pos().x ),
                           getScreenY( tri->vertex( 0 )->pos().y ) );
        vertices.setPoint( 1,
                           getScreenX( tri->vertex( 1 )->pos().x ),
                           getScreenY( tri->vertex( 1 )->pos().y ) );
        vertices.setPoint( 2,
                           getScreenX( tri->vertex( 2 )->pos().x ),
                           getScreenY( tri->vertex( 2 )->pos().y ) );

        painter.drawConvexPolygon( vertices );

        // draw center point
        rcsc::Vector2D center
            = rcsc::Triangle2D::centroid( tri->vertex( 0 )->pos(),
                                          tri->vertex( 1 )->pos(),
                                          tri->vertex( 2 )->pos() );

        painter.setPen( Qt::cyan );
        painter.setBrush( Qt::cyan );
        painter.drawRect( getScreenX( center.x ) - 1,
                          getScreenY( center.y ) - 1,
                          3, 3 );
#if 0
        center = rcsc::Triangle2D::incenter( tri->vertex( 0 )->pos(),
                                             tri->vertex( 1 )->pos(),
                                             tri->vertex( 2 )->pos() );

        painter.setPen( Qt::red );
        painter.setBrush( Qt::red );
        painter.drawRect( getScreenX( center.x ) - 1,
                          getScreenY( center.y ) - 1,
                          3, 3 );


        center = rcsc::Triangle2D::circumcenter( tri->vertex( 0 )->pos(),
                                                 tri->vertex( 1 )->pos(),
                                                 tri->vertex( 2 )->pos() );

        painter.setPen( Qt::red );
        painter.setBrush( Qt::red );
        painter.drawRect( getScreenX( center.x ) - 1,
                          getScreenY( center.y ) - 1,
                          3, 3 );
#endif
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawTrainingData( QPainter & painter )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    // if method type is not a FormationDelaunayTriangulation
    // only sample ball points are drawn
    if ( f->methodName() != rcsc::FormationDT::name() )
    {
        painter.setPen( Qt::red );
        painter.setBrush( Qt::NoBrush );
        painter.setFont( M_player_font );
        //painter.setBackgroundMode( Qt::TransparentMode );

        const int r = scale( 0.5 );

        int count = 0;
        const std::list< rcsc::Formation::Snapshot >::const_iterator train_end
            = ptr->trainingData().end();
        for ( std::list< rcsc::Formation::Snapshot >::const_iterator train_it
                  = ptr->trainingData().begin();
              train_it != train_end;
              ++train_it )
        {
            int x = getScreenX( train_it->ball_.x );
            int y = getScreenY( train_it->ball_.y );
            //painter.DrawCircle( x, y, r );
            painter.drawLine( x - r, y - r,
                              x + r, y + r );
            painter.drawLine( x - r, y + r,
                              x + r, y - r );
            //char buf[4];
            //std::snprintf( buf, 4, "%2d", ++count );
            painter.drawText( x + r, y - r,
                              QString().sprintf( "%2d", ++count ) );// QString::fromAscii( buf ) );
        }

        return;
    }

    // draw delaunay triangulation

    if ( ptr->triangulation().vertices().size() < 3 )
    {
        // too few kernel points
        // no valid triangulation
        return;
    }

    painter.setPen( M_triangle_pen );
    painter.setBrush( Qt::NoBrush );

    const std::map< int, rcsc::DelaunayTriangulation::EdgePtr >::const_iterator end
        = ptr->triangulation().edgeMap().end();
    for ( std::map< int, rcsc::DelaunayTriangulation::EdgePtr >::const_iterator it
              = ptr->triangulation().edgeMap().begin();
          it != end;
          ++it )
    {
        painter.drawLine( getScreenX( it->second->vertex( 0 )->pos().x ),
                          getScreenY( it->second->vertex( 0 )->pos().y ),
                          getScreenX( it->second->vertex( 1 )->pos().x ),
                          getScreenY( it->second->vertex( 1 )->pos().y ) );
    }

    painter.setFont( M_player_font );
    painter.setPen( Qt::red );
    painter.setBackgroundMode( Qt::TransparentMode );

    const int radius = scale( 0.7 );
    int count = 0;
    const std::list< rcsc::Formation::Snapshot >::const_iterator v_end
        = ptr->trainingData().end();
    for ( std::list< rcsc::Formation::Snapshot >::const_iterator it
              = ptr->trainingData().begin();
          it != v_end;
          ++it )
    {
        int ix = getScreenX( it->ball_.x );
        int iy = getScreenY( it->ball_.y );
        //char buf[4];
        //std::snprintf( buf, 4, "%2d", ++count );
        painter.drawText( ix + radius, iy - radius,
                          QString().sprintf( "%2d", ++count ) ); //QString::fromAscii( buf ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawPlayers( QPainter & painter )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    const int radius = scale( 1.2 );
    const int diameter = radius * 2;;
    const int selected_radius = radius + 3;

    ////////////////////////////////////////////////////////////
    painter.setFont( M_player_font );
    painter.setBackgroundMode( Qt::TransparentMode );
    ////////////////////////////////////////////////////////////
    // draw right players
#if 0
    {
        painter.setBrush( M_right_team_brush );

        const std::vector< rcsc::Vector2D >::const_iterator
            selection = ( ptr->getSelectType() == FEditData::SELECT_OPP
                          ? ( ptr->getOppPlayersPos().begin()
                              + ptr->getSelectIndex() )
                          : ptr->getOppPlayersPos().end() );
        int unum = 1;
        for ( std::vector< rcsc::Vector2D >::const_iterator
                  it = ptr->getOppPlayersPos().begin();
              it != ptr->getOppPlayersPos().end();
              ++it, ++unum )
        {
            int ix = getScreenX( it->x );
            int iy = getScreenY( it->y );
            if ( selection == it )
            {
                painter.setPen( M_select_pen );
                painter.drawEllipse( ix - selected_radius,
                                     iy - selected_radius,
                                     selected_radius * 2,
                                     selected_radius * 2 );
                painter.setPen( M_player_pen );
            }
            else
            {
                painter.drawEllipse( ix - radius,
                                     iy - radius,
                                     diameter,
                                     diameter );
            }
            //char buf[4];
            //std::snprintf( buf, 4, "%2d", unum );
            painter.DrawText( ix + radius, iy - radius,
                              QString().sprintf( "%2d", unum ) );
                              //QString::fromAscii( buf ) );

        }
    }
#endif
    ////////////////////////////////////////////////////////////
    // draw left players
    {
        const std::vector< rcsc::Vector2D >::const_iterator
            selection = ( ptr->getSelectType() == FEditData::SELECT_OUR
                          ? ( ptr->ourPlayers().begin()
                              + ptr->getSelectIndex() )
                          : ptr->ourPlayers().end() );
        int unum = 1;
        for ( std::vector< rcsc::Vector2D >::const_iterator
                  it = ptr->ourPlayers().begin();
              it != ptr->ourPlayers().end();
              ++it, ++unum )
        {
            int ix = getScreenX( it->x );
            int iy = getScreenY( it->y );
            if ( f->isMirrorType( unum ) )
            {
                painter.setPen( M_player_pen );
                painter.setBrush( M_mirror_brush );
                painter.drawEllipse( ix - radius,
                                     iy - radius,
                                     diameter,
                                     diameter );
            }
            else if ( selection == it )
            {
                painter.setPen( M_select_pen );
                painter.setBrush( M_left_team_brush );
                painter.drawEllipse( ix - selected_radius,
                                     iy - selected_radius,
                                     selected_radius * 2,
                                     selected_radius * 2 );
            }
            else
            {
                painter.setPen( M_player_pen );
                painter.setBrush( M_left_team_brush );
                painter.drawEllipse( ix - radius,
                                     iy - radius,
                                     diameter,
                                     diameter );
            }
            //char buf[4];
            //std::snprintf( buf, 4, "%2d", unum );
            painter.setPen( Qt::white );
            painter.drawText( ix + radius, iy - radius,
                              QString().sprintf( "%2d", unum ) );
                              //QString::fromAscii( buf ) );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawBall( QPainter & painter )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    painter.setPen( M_ball_pen );

    if ( ptr->isBallDraggable() )
    {
        painter.setBrush( M_ball_brush );
    }
    else
    {
        painter.setBrush( Qt::lightGray );
    }

    int radius = ( ptr->getSelectType() == FEditData::SELECT_BALL
                   ? scale( 1.0 )
                   : scale( 0.7 ) );

    painter.drawEllipse( getScreenX( ptr->ball().x ) - radius,
                         getScreenY( ptr->ball().y ) - radius,
                         radius * 2,
                         radius * 2 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::mousePressEvent( QMouseEvent * event )
{
    if ( event->button() == Qt::LeftButton )
    {
        M_mouse_state[0].pressed( event->pos() );

        if ( boost::shared_ptr< FEditData > ptr = M_data_ptr.lock() )
        {
            ptr->selectObject( getFieldX( event->pos().x() ), 
                               getFieldY( event->pos().y() ) );
            this->update();
        }
    }
    else if ( event->button() == Qt::MidButton )
    {
        M_mouse_state[1].pressed( event->pos() );
    }
    else if ( event->button() == Qt::RightButton )
    {
        M_mouse_state[2].pressed( event->pos() );

        if ( boost::shared_ptr< FEditData > ptr = M_data_ptr.lock() )
        {
            if ( ptr->isBallDraggable() )
            {
                ptr->moveBallTo( getFieldX( event->pos().x() ),
                                 getFieldY( event->pos().y() ) );
                this->update();
                emit objectMoved();
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::mouseReleaseEvent( QMouseEvent * event )
{
    if ( event->button() == Qt::LeftButton )
    {
        M_mouse_state[0].released();

        if ( boost::shared_ptr< FEditData > ptr = M_data_ptr.lock() )
        {
            ptr->releaseObject( getFieldX( event->pos().x() ),
                                getFieldY( event->pos().y() ) );
            this->update();
        }
    }
    else if ( event->button() == Qt::MidButton )
    {
        M_mouse_state[1].released();
    }
    else if ( event->button() == Qt::RightButton )
    {
        M_mouse_state[2].released();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::mouseMoveEvent( QMouseEvent * event )
{
    QPoint point = event->pos();
    for ( int i = 0; i < 3; ++i )
    {
        M_mouse_state[i].moved( point );
    }

    double field_x = getFieldX( point.x() );
    double field_y = getFieldY( point.y() );
    //char buf[32];
    //std::snprintf( buf, 32, "(%.2f %.2f)", field_x, field_y );

    if ( M_mouse_state[0].isDragged() )
    {
        boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
        if ( ptr
             && ptr->getSelectType() != FEditData::NO_SELECT )
        {
            ptr->setObjectDragPoint( field_x, field_y );

            this->update();
            emit objectMoved();
        }
    }

    //emit pointerMoved( QString::fromAscii( buf ) );
    emit pointerMoved( QString().sprintf( "(%.2f %.2f)", field_x, field_y ) );

}

