// -*-c++-*-

/*!
	\file feditor_canvas.cpp
	\brief formation editor canvas class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include "fedit_canvas.h"

#include "fedit_frame.h"
#include "fedit_data.h"
#include "fedit_config.h"
#include "id.h"

#include <rcsc/formation/formation_dt.h>

#include <rcsc/geom/triangle_2d.h>

#include <algorithm>
#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
FEditCanvas::FEditCanvas( FEditFrame * parent )
    : wxWindow( parent, -1, wxDefaultPosition, wxDefaultSize,
                wxSIMPLE_BORDER /*| wxNO_FULL_REPAINT_ON_RESIZE*/ )
    , M_parent( parent )
    , M_scale( 1.0 )
    , M_center( 0, 0 )
    , M_field_brush(  wxColor( 31, 160, 31 ), wxSOLID )
    , M_field_dark_brush( wxColor( 15, 143, 15 ), wxSOLID )
    , M_line_pen( wxColor( 255, 255, 255 ), 1, wxSOLID )
    , M_triangle_pen( wxColor( 255, 127, 0 ), 1, wxSOLID )
    , M_triangle_font( 10, wxDEFAULT, wxNORMAL, wxNORMAL, false, wxT( "" ), wxFONTENCODING_ISO8859_1 ) //wxFONTENCODING_SYSTEM )
    , M_area_pen( wxColor( 127, 127, 127 ), 1, wxSOLID )
    , M_ball_pen( wxColor( 255, 255, 255 ), 1, wxSOLID )
    , M_ball_brush( wxColor( 255, 255, 255 ), wxSOLID )
    , M_player_pen( wxColor( 0, 0, 0 ), 1, wxSOLID )
    , M_select_pen( *wxWHITE, 2, wxSOLID )
    , M_left_team_brush( wxColor( 255, 215, 0 ), wxSOLID )
    , M_right_team_brush( wxColor( 240, 20, 20 ), wxSOLID )
    , M_mirror_brush( wxColor( 0, 255, 95 ), wxSOLID )
    , M_player_font( 10, wxDEFAULT, wxNORMAL, wxNORMAL, false, wxT( "" ), wxFONTENCODING_ISO8859_1 ) //wxFONTENCODING_SYSTEM )
{
    assert( parent );

    //parent->SetClientSize( static_cast< int >( pitch_len / M_scale ),
    //static_cast< int >( pitch_width / M_scale ) );

    createWindows();
    connectEvents();

    M_canvas_bmp.Create( this->GetClientSize().GetWidth(),
                         this->GetClientSize().GetHeight() );

    updateSize();
}

/*-------------------------------------------------------------------*/
/*!
  destructor
*/
FEditCanvas::~FEditCanvas()
{
    //std::cerr << "delete FEditCanvas" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::connectEvents()
{
    //////////////////////////////////////////////////////////////
    // paint event
    Connect( wxID_ANY, wxEVT_PAINT,
             wxPaintEventHandler( FEditCanvas::handlePaint ) );
    // mouse event
    Connect( wxID_ANY, wxEVT_LEFT_DOWN,
             wxMouseEventHandler( FEditCanvas::handleLeftDown ) );
    Connect( wxID_ANY, wxEVT_LEFT_UP,
             wxMouseEventHandler( FEditCanvas::handleLeftUp ) );
    Connect( wxID_ANY, wxEVT_MIDDLE_DOWN,
             wxMouseEventHandler( FEditCanvas::handleMiddleDown ) );
    Connect( wxID_ANY, wxEVT_MIDDLE_UP,
             wxMouseEventHandler( FEditCanvas::handleMiddleUp ) );
    Connect( wxID_ANY, wxEVT_RIGHT_DOWN,
             wxMouseEventHandler( FEditCanvas::handleRightDown ) );
    Connect( wxID_ANY, wxEVT_RIGHT_UP,
             wxMouseEventHandler( FEditCanvas::handleRightUp ) );
    Connect( wxID_ANY, wxEVT_MOTION,
             wxMouseEventHandler( FEditCanvas::handleMouseMotion ) );
}


/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::createWindows()
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::updateSize()
{
    M_center.x = this->GetClientSize().x / 2;
    M_center.y = this->GetClientSize().y / 2;

    const double pitch_l = ( FEditConfig::PITCH_LENGTH
                             + FEditConfig::PITCH_MARGIN * 2.0 );
    const double pitch_w = ( FEditConfig::PITCH_WIDTH
                             + FEditConfig::PITCH_MARGIN * 2.0 );

    M_scale = this->GetClientSize().x / pitch_l;
    if ( pitch_w * M_scale > this->GetSize().y )
    {
        M_scale = this->GetClientSize().y / pitch_w;
    }
    M_scale = std::max( 2.0, M_scale );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::handlePaint( wxPaintEvent & WXUNUSED( event ) )
{
    wxPaintDC dc( this );

    draw( dc );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::draw()
{
    wxClientDC dc( this );

    draw( dc );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::draw( wxDC & dc )
{
    // check canvas size
    const wxSize cur_size = this->GetClientSize();
    if ( M_canvas_bmp.GetWidth() != cur_size.GetWidth()
         || M_canvas_bmp.GetHeight() != cur_size.GetHeight() )
    {
        M_canvas_bmp.Create( cur_size.GetWidth(), cur_size.GetHeight() );
        updateSize();
    }

    // create buffer DC
    wxMemoryDC mem_dc;
    mem_dc.SelectObject( M_canvas_bmp );

    drawField( mem_dc );
    drawTrainingData( mem_dc );
    drawPlayers( mem_dc );
    drawBall( mem_dc );

    // blit buffer DC to 'dc'
    dc.Blit( 0, 0,
             cur_size.GetWidth(), cur_size.GetHeight(),
             &mem_dc, 0, 0 );

    mem_dc.SelectObject( wxNullBitmap );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawField( wxDC & dc )
{
    //--------------------------------------
    // fill background
    dc.SetBackground( M_field_brush );
    dc.Clear();

    //--------------------------------------
    drawContainedArea( dc );

    // set screen coordinates of field
    const wxCoord left_x   = getScreenX( - FEditConfig::PITCH_LENGTH * 0.5 );
    const wxCoord right_x  = getScreenX( + FEditConfig::PITCH_LENGTH * 0.5 );
    const wxCoord top_y    = getScreenY( - FEditConfig::PITCH_WIDTH * 0.5 );
    const wxCoord bottom_y = getScreenY( + FEditConfig::PITCH_WIDTH * 0.5 );

    // fill bottom side
    //dc.SetPen( *wxTRANSPARENT_PEN );
    //dc.SetBrush( M_field_dark_brush );
    //dc.DrawRectangle( left_x, M_center.y,
    //right_x - left_x, (bottom_y - top_y)/2 );

    //--------------------------------------
    // draw lines
    dc.SetPen( M_line_pen );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );
    // side lines & goal lines
    dc.DrawLine( left_x, top_y, right_x, top_y );
    dc.DrawLine( right_x, top_y, right_x, bottom_y );
    dc.DrawLine( right_x, bottom_y, left_x, bottom_y );
    dc.DrawLine( left_x, bottom_y, left_x, top_y );
    // center line
    dc.DrawLine( M_center.x, top_y, M_center.x, bottom_y );
    // center circle
    const wxCoord center_radius = scale( FEditConfig::CENTER_CIRCLE_R );
    dc.DrawCircle( M_center, center_radius );
    // draw penalty area box
    const wxCoord pen_top_y    = getScreenY( - FEditConfig::PENALTY_AREA_WIDTH / 2 );
    const wxCoord pen_bottom_y = getScreenY( + FEditConfig::PENALTY_AREA_WIDTH / 2 );
    // left penalty box
    wxCoord pen_x = getScreenX( -( FEditConfig::PITCH_LENGTH*0.5
                                   - FEditConfig::PENALTY_AREA_LENGTH ) );
    dc.DrawLine( left_x, pen_top_y, pen_x, pen_top_y );
    dc.DrawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    dc.DrawLine( pen_x, pen_bottom_y, left_x, pen_bottom_y );
    // right penalty box
    pen_x = getScreenX( +( FEditConfig::PITCH_LENGTH*0.5
                           - FEditConfig::PENALTY_AREA_LENGTH ) );
    dc.DrawLine( right_x, pen_top_y, pen_x, pen_top_y );
    dc.DrawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    dc.DrawLine( pen_x, pen_bottom_y, right_x, pen_bottom_y );
    // draw goal area box
    const wxCoord goal_area_y_abs = scale( FEditConfig::GOAL_AREA_WIDTH*0.5 );
    const wxCoord goal_area_top_y = M_center.y - goal_area_y_abs;
    const wxCoord goal_area_bottom_y = M_center.y + goal_area_y_abs;
    // left goal area
    wxCoord goal_area_x = getScreenX( - FEditConfig::PITCH_LENGTH*0.5
                                      + FEditConfig::GOAL_AREA_LENGTH );
    dc.DrawLine( left_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    dc.DrawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    dc.DrawLine( goal_area_x, goal_area_bottom_y, left_x, goal_area_bottom_y );
    // right goal area
    goal_area_x = getScreenX( FEditConfig::PITCH_LENGTH*0.5
                              - FEditConfig::GOAL_AREA_LENGTH );
    dc.DrawLine( right_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    dc.DrawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    dc.DrawLine( goal_area_x, goal_area_bottom_y, right_x, goal_area_bottom_y );

    //--------------------------------------
    // draw goal boxes
    dc.SetPen( *wxBLACK_PEN );
    dc.SetBrush( *wxBLACK_BRUSH );
    const wxCoord goal_top_y = getScreenY( - FEditConfig::GOAL_WIDTH*0.5 );
    const wxCoord goal_size_x = scale( FEditConfig::GOAL_DEPTH );
    const wxCoord goal_size_y = scale( FEditConfig::GOAL_WIDTH );
    // left goal
    dc.DrawRectangle( getScreenX( - FEditConfig::PITCH_LENGTH*0.5
                                  - FEditConfig::GOAL_DEPTH ),
                      goal_top_y, goal_size_x, goal_size_y );
    // right goal
    dc.DrawRectangle( getScreenX( FEditConfig::PITCH_LENGTH*0.5 ) + 1,
                      goal_top_y, goal_size_x + 1, goal_size_y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawContainedArea( wxDC & dc )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return ;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    if ( f->methodName() != rcsc::FormationDT::name() )
    {
        return;
    }

    if ( ptr->getTriangulation().vertices().size() < 3 )
    {
        return;
    }

    // draw polygon
    const rcsc::DelaunayTriangulation::TrianglePtr tri
        = ptr->getTriangulation().findTriangleContains( ptr->getBall() );
    if ( tri )
    {
        dc.SetPen( *wxTRANSPARENT_PEN );
        dc.SetBrush( M_field_dark_brush );
        wxPoint vertices[3] = { wxPoint( getScreenX( tri->vertex( 0 )->pos().x ),
                                         getScreenY( tri->vertex( 0 )->pos().y ) ),
                                wxPoint( getScreenX( tri->vertex( 1 )->pos().x ),
                                         getScreenY( tri->vertex( 1 )->pos().y ) ),
                                wxPoint( getScreenX( tri->vertex( 2 )->pos().x ),
                                         getScreenY( tri->vertex( 2 )->pos().y ) ),
        };

        dc.DrawPolygon( 3, vertices );

        // draw center point
        rcsc::Vector2D center
            = rcsc::Triangle2D::centroid( tri->vertex( 0 )->pos(),
                                          tri->vertex( 1 )->pos(),
                                          tri->vertex( 2 )->pos() );

        dc.SetPen( *wxCYAN_PEN );
        dc.SetBrush( *wxCYAN_BRUSH );
        dc.DrawRectangle( getScreenX( center.x ) - 1,
                          getScreenY( center.y ) - 1,
                          3, 3 );
#if 0
        center = rcsc::Triangle2D::incenter( tri->vertex( 0 )->pos(),
                                             tri->vertex( 1 )->pos(),
                                             tri->vertex( 2 )->pos() );

        dc.SetPen( *wxRED_PEN );
        dc.SetBrush( *wxRED_BRUSH );
        dc.DrawRectangle( getScreenX( center.x ) - 1,
                          getScreenY( center.y ) - 1,
                          3, 3 );


        center = rcsc::Triangle2D::circumcenter( tri->vertex( 0 )->pos(),
                                                 tri->vertex( 1 )->pos(),
                                                 tri->vertex( 2 )->pos() );

        dc.SetPen( *wxRED_PEN );
        dc.SetBrush( *wxRED_BRUSH );
        dc.DrawRectangle( getScreenX( center.x ) - 1,
                          getScreenY( center.y ) - 1,
                          3, 3 );
#endif
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawTrainingData( wxDC & dc )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    // if method type is not a FormationDelaunayTriangulation
    // only sample ball points are drawn
    if ( f->methodName() != rcsc::FormationDT::name() )
    {
        dc.SetPen( *wxRED_PEN ); //M_triangle_pen );
        dc.SetBrush( *wxTRANSPARENT_BRUSH );
        dc.SetFont( M_player_font );
        dc.SetTextForeground( *wxRED );
        dc.SetBackgroundMode( wxTRANSPARENT );

        const int r = scale( 0.5 );

        int count = 0;
        const std::list< rcsc::Formation::Snapshot >::const_iterator train_end
            = ptr->getTrainingData().end();
        for ( std::list< rcsc::Formation::Snapshot >::const_iterator train_it
                  = ptr->getTrainingData().begin();
              train_it != train_end;
              ++train_it )
        {
            int x = getScreenX( train_it->ball_.x );
            int y = getScreenY( train_it->ball_.y );
            //dc.DrawCircle( x, y, r );
            dc.DrawLine( x - r, y - r,
                         x + r, y + r );
            dc.DrawLine( x - r, y + r,
                         x + r, y - r );
            wxString str;
            str.sprintf( wxT( "%2d" ), ++count );
            dc.DrawText( str,
                         x + r, y - r );
        }

        return;
    }

    // draw delaunay triangulation

    if ( ptr->getTriangulation().vertices().size() < 3 )
    {
        // too few kernel points
        // no valid triangulation
        return;
    }

    dc.SetPen( M_triangle_pen );
    dc.SetBrush( *wxTRANSPARENT_BRUSH );

    const std::map< int, rcsc::DelaunayTriangulation::EdgePtr >::const_iterator end
        = ptr->getTriangulation().edgeMap().end();
    for ( std::map< int, rcsc::DelaunayTriangulation::EdgePtr >::const_iterator it
              = ptr->getTriangulation().edgeMap().begin();
          it != end;
          ++it )
    {
        dc.DrawLine( getScreenX( it->second->vertex( 0 )->pos().x ),
                     getScreenY( it->second->vertex( 0 )->pos().y ),
                     getScreenX( it->second->vertex( 1 )->pos().x ),
                     getScreenY( it->second->vertex( 1 )->pos().y ) );
    }

    dc.SetFont( M_player_font );
    dc.SetTextForeground( *wxRED );
    dc.SetBackgroundMode( wxTRANSPARENT );

    const int radius = scale( 0.7 );
    int count = 0;
    const std::list< rcsc::Formation::Snapshot >::const_iterator v_end
        = ptr->getTrainingData().end();
    for ( std::list< rcsc::Formation::Snapshot >::const_iterator it
              = ptr->getTrainingData().begin();
          it != v_end;
          ++it )
    {
        int ix = getScreenX( it->ball_.x );
        int iy = getScreenY( it->ball_.y );
        wxString str;
        str.sprintf( wxT( "%2d" ), ++count );
        dc.DrawText( str,
                     ix + radius, iy - radius );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawPlayers( wxDC & dc )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    const int radius = scale( 1.2 );

    ////////////////////////////////////////////////////////////
    dc.SetPen( M_player_pen );
    dc.SetFont( M_player_font );
    dc.SetTextForeground( *wxWHITE );
    dc.SetBackgroundMode( wxTRANSPARENT );
    ////////////////////////////////////////////////////////////
    // draw right players
#if 0
    {
        dc.SetBrush( M_right_team_brush );

        const std::vector< rcsc::Vector2D >::const_iterator
            selection = ( ptr->getSelectType() == FEditData::SELECT_OPP
                          ? ( ptr->getOppPlayersPos().begin()
                              + ptr->getSelectIndex() )
                          : ptr->getOppPlayersPos().end() );
        int unum = 1;
        for ( std::vector< rcsc::Vector2D >::const_iterator
                  it = ptr->getOppPlayersPos().begin();
              it != ptr->getOppPlayersPos().end();
              ++it, ++unum )
        {
            int ix = getScreenX( it->x );
            int iy = getScreenY( it->y );
            if ( selection == it )
            {
                dc.SetPen( M_select_pen );
                dc.DrawCircle( ix, iy, radius + 3 );
                dc.SetPen( M_player_pen );
            }
            else
            {
                dc.DrawCircle( ix, iy, radius );
            }
            wxString player_str;
            player_str.Printf( wxT( "%2d" ), unum );
            dc.DrawText( player_str,
                         ix + radius, iy - radius );

        }
    }
#endif
    ////////////////////////////////////////////////////////////
    // draw left players
    {
        const std::vector< rcsc::Vector2D >::const_iterator
            selection = ( ptr->getSelectType() == FEditData::SELECT_OUR
                          ? ( ptr->getOurPlayers().begin()
                              + ptr->getSelectIndex() )
                          : ptr->getOurPlayers().end() );
        int unum = 1;
        for ( std::vector< rcsc::Vector2D >::const_iterator
                  it = ptr->getOurPlayers().begin();
              it != ptr->getOurPlayers().end();
              ++it, ++unum )
        {
            int ix = getScreenX( it->x );
            int iy = getScreenY( it->y );
            if ( f->isMirrorType( unum ) )
            {
                dc.SetBrush( M_mirror_brush );
                dc.DrawCircle( ix, iy, radius );
            }
            else if ( selection == it )
            {
                dc.SetPen( M_select_pen );
                dc.SetBrush( M_left_team_brush );
                dc.DrawCircle( ix, iy, radius + 3 );
                dc.SetPen( M_player_pen );
            }
            else
            {
                dc.SetBrush( M_left_team_brush );
                dc.DrawCircle( ix, iy, radius );
            }
            wxString player_str;
            player_str.Printf( wxT( "%2d" ), unum );
            dc.DrawText( player_str,
                         ix + radius, iy - radius );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawBall( wxDC & dc )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    dc.SetPen( M_ball_pen );

    if ( ptr->isBallDraggable() )
    {
        dc.SetBrush( M_ball_brush );
    }
    else
    {
        dc.SetBrush( *wxLIGHT_GREY_BRUSH );
    }

    int radius = ( ptr->getSelectType() == FEditData::SELECT_BALL
                   ? scale( 1.0 )
                   : scale( 0.7 ) );

    dc.DrawCircle( getScreenX( ptr->getBall().x ),
                   getScreenY( ptr->getBall().y ),
                   radius );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::handleLeftDown( wxMouseEvent & event )
{
    M_mouse_state[0].pressed( event.GetPosition() );

    wxPoint point = event.GetPosition();
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ptr )
    {
        ptr->selectObject( ( point.x - M_center.x ) / M_scale,
                           ( point.y - M_center.y ) / M_scale );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::handleLeftUp( wxMouseEvent & event )
{
    M_mouse_state[0].released();

    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ptr )
    {
        ptr->releaseObject( ( event.GetPosition().x - M_center.x ) / M_scale,
                            ( event.GetPosition().y - M_center.y ) / M_scale );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::handleMiddleDown( wxMouseEvent & event )
{
    M_mouse_state[1].pressed( event.GetPosition() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::handleMiddleUp( wxMouseEvent & event )
{
    M_mouse_state[1].released();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::handleRightDown( wxMouseEvent & event )
{
    M_mouse_state[2].pressed( event.GetPosition() );

    if ( boost::shared_ptr< FEditData > ptr = M_data_ptr.lock() )
    {
        if ( ptr->isBallDraggable() )
        {
            ptr->dropBallTo( ( event.GetPosition().x - M_center.x ) / M_scale,
                             ( event.GetPosition().y - M_center.y ) / M_scale );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::handleRightUp( wxMouseEvent & event )
{
    M_mouse_state[2].released();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::handleMouseMotion( wxMouseEvent & event )
{
    wxPoint point = event.GetPosition();
    for ( int i = 0; i < 3; ++i )
    {
        M_mouse_state[i].moved( point );
    }
    M_parent->updatePositionLabel( ( point.x - M_center.x ) / M_scale,
                                   ( point.y - M_center.y ) / M_scale );

    if ( M_mouse_state[0].isDragged() )
    {
        boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
        if ( ptr
             && ptr->getSelectType() != FEditData::NO_SELECT )
        {
            ptr->setObjectDragPoint( ( point.x - M_center.x ) / M_scale,
                                     ( point.y - M_center.y ) / M_scale );
        }
    }

}
