// -*-c++-*-

/*!
	\file feditor_data.cpp
	\brief formation editor data class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/filename.h>
#endif

#include "fedit_data.h"

#include "id.h"
#include "fedit_frame.h"
#include "fedit_canvas.h"
#include "fedit_dialog.h"
#include "fedit_config.h"

#include <rcsc/formation/formation_bpn.h>
#include <rcsc/formation/formation_dt.h>
#include <rcsc/formation/formation_ngnet.h>
#include <rcsc/formation/formation_rbf.h>
#include <rcsc/formation/formation_sbsp.h>
#include <rcsc/math_util.h>

#include <algorithm>
#include <sstream>
#include <fstream>
#include <iostream>

#include <cassert>
#include <ctime>
#include <cstdio>

const int FEditData::MAX_TRAINING_DATA_SIZE = 128;


/*-------------------------------------------------------------------*/
/*!

*/
FEditData::FEditData( FEditFrame * frame )
    : M_frame( frame )
    , M_canvas( static_cast< FEditCanvas * >( 0 ) )
    , M_dialog( static_cast< FEditDialog * >( 0 ) )
    , M_training_type_name( "" )
    , M_filepath( "" )
    , M_filepath_train( "" )
    , M_modified( false )
    , M_train_data_changed( false )
    , M_our_dragged( 11, 0 )
    , M_our_players( 11 )
    , M_opp_players( 11 )
    , M_select_type( NO_SELECT )
    , M_select_index( 0 )
    , M_ball_draggable( true )
    , M_player_auto_move( true )
{
    assert( frame );
    init();
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditData::~FEditData()
{
    //saveChanged();
    //std::cerr << "delete FEditData" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
boost::shared_ptr< rcsc::Formation >
FEditData::create_formation( const std::string & type_name )
{
    boost::shared_ptr< rcsc::Formation > f;

    if ( type_name == rcsc::FormationBPN::name() )
    {
        f = boost::shared_ptr< rcsc::Formation >( new rcsc::FormationBPN );
    }
    else if ( type_name == rcsc::FormationDT::name() )
    {
        f = boost::shared_ptr< rcsc::Formation >( new rcsc::FormationDT );
    }
    else if ( type_name == rcsc::FormationNGNet::name() )
    {
        f = boost::shared_ptr< rcsc::Formation >( new rcsc::FormationNGNet );
    }
    else if ( type_name == rcsc::FormationRBF::name() )
    {
        f = boost::shared_ptr< rcsc::Formation >( new rcsc::FormationRBF );
    }
    else if ( type_name == rcsc::FormationSBSP::name() )
    {
        f = boost::shared_ptr< rcsc::Formation >( new rcsc::FormationSBSP );
    }
    else
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***RRROR*** Unknown formation method name. ["
                  << type_name << "]"
                  << std::endl;
    }

    return f;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::init()
{
    M_modified = false;
    M_train_data_changed = false;

    M_ball.assign( 0.0, 0.0 );
    M_our_dragged.assign( 11, 0 );
    for ( int i = 1; i <= 11; ++i )
    {
        M_our_players[i-1].setPolar( -3.0 * i, -37.0 );
        M_opp_players[i-1].assign( 3.0 * i, -37.0 );
    }

    M_select_type = NO_SELECT;
    M_select_index = 0;
    M_ball_draggable = true;
    M_player_auto_move = true;

    M_triangulation.init( rcsc::Rect2D( - FEditConfig::PITCH_LENGTH * 0.5,
                                        - FEditConfig::PITCH_WIDTH * 0.5,
                                        FEditConfig::PITCH_LENGTH,
                                        FEditConfig::PITCH_WIDTH ) );
    updateTriangulation();

    printMessageWithTime( "init" );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::printMessageWithTime( char * msg, ... )
{
    if ( FEditConfig::instance().logging() )
    {
        std::time_t t;
        std::time( &t );
        std::tm * lt = std::localtime( &t );

        char time_str[32];
        std::strftime( time_str, 32, "%Y-%m-%d-%H-%M-%S", lt );

        char buffer[512];
        va_list argp;
        va_start( argp, msg );
        vsnprintf( buffer, 512, msg, argp );
        va_end( argp );

        std::fprintf( stdout, "%s : %s\n", time_str, buffer );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setCanvas( FEditCanvas * canvas )
{
    M_canvas = canvas;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setDialog( FEditDialog * dialog )
{
    M_dialog = dialog;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::createDefaultParam()
{
    M_formation = create_formation( M_training_type_name );

    if ( ! M_formation )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** Failed to create formation."
                  << std::endl;
        return;
    }

    rcsc::Formation::Snapshot default_data = M_formation->createDefaultParam();

    M_train_data.push_back( default_data );
    M_frame->setTrainDataIndexMax( static_cast< int >( M_train_data.size() ) );

    printMessageWithTime( "create default params" );

    train();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::open( const std::string & filepath )
{
    if ( M_modified )
    {
        showSaveNotifyDialog();
    }

    std::ifstream fin( filepath.c_str() );
    if ( ! fin.is_open() )
    {
        ::wxMessageBox( wxT( "Failed to open the file!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        M_frame ); // parent frame
        std::cerr << "Failed to open formation file [" << filepath << "]"
                  << std::endl;
        return false;
    }

    M_train_data.clear();

    {
        std::string temp, type;
        fin >> temp >> type; // read training method type name
        fin.seekg( 0 );
        M_formation = create_formation( type );

        if ( ! M_formation )
        {
            ::wxMessageBox( wxT( "Invalid formation format!!" ),
                            wxT( "Error" ),   // title
                            wxOK | wxICON_ERROR, // style
                            M_frame ); // parent frame
            std::cerr << "Invalid formation format in the file ["
                      << filepath << "]\n"
                      << "You must write the training method type name"
                      << " at the top of the formation file"
                      << std::endl;
            return false;
        }
    }

    bool success = M_formation->read( fin );
    if ( success )
    {
        M_filepath = filepath;
        init();
        updatePlayerPosition();
        notifyChanged();

        wxString filepath_wx( filepath.c_str(),
                              *wxConvCurrent );
        wxString only_filename;
        wxFileName::SplitPath( filepath_wx,
                               NULL,
                               &only_filename,
                               NULL );
        wxString title = wxT( "Formation Editor - " );
        title += only_filename;
        title += wxT( " -" );
        M_frame->SetTitle( title );
        printMessageWithTime( "opend formation data [%s]",
                              filepath.c_str() );

        return true;
    }
    else
    {
        ::wxMessageBox( wxT( "Failed to load formation data!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        M_frame ); // parent frame
        M_formation.reset();
    }

    fin.close();

    return success;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::openTrainData( const std::string & filepath )
{
    if ( M_train_data_changed )
    {
        showTrainDataSaveNotifyDialog();
    }

    std::ifstream fin( filepath.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open training data file [" << filepath << "]"
                  << std::endl;
        ::wxMessageBox( wxT( "Failed to open the file!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        M_frame ); // parent frame
        return false;
    }

    // clear current data list
    M_train_data.clear();

    bool success = true;
    std::string line_buf;
    while ( std::getline( fin, line_buf ) )
    {
        rcsc::Formation::Snapshot snap;
        std::istringstream istr( line_buf );
        istr >> snap.ball_.x >> snap.ball_.y;
        for ( int i = 0; i < 11; ++i )
        {
            if ( ! istr.good() )
            {
                success = false;
                break;
            }
            rcsc::Vector2D v;
            istr >> v.x >> v.y;
            snap.players_.push_back( v );
        }

        M_train_data.push_back( snap );
    }

    fin.close();

    if ( success )
    {
        /*
        ::wxMessageBox( wxT( "Please push train button." ),
                        wxT( "Note" ),   // title
                        wxOK, // style
                        M_frame ); // parent frame
        */
        printMessageWithTime( "opened training data. [%s]",
                              filepath.c_str() );
        M_filepath_train = filepath;
        M_frame->setTrainDataIndexMax( static_cast< int >( M_train_data.size() ) );

        updateTriangulation();
        //train();
        M_modified = false;
        M_train_data_changed = false;
        moveBallTo( 0.0, 0.0 );
        updatePlayerPosition();
    }
    else
    {
        ::wxMessageBox( wxT( "Failed to load training data!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        M_frame ); // parent frame
        M_train_data.clear();
    }

    return success;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
FEditData::saveChanged()
{
    //std::cerr << "FEditData::close()" << std::endl;
    int result = 0;
    if ( M_modified )
    {
        result |= showSaveNotifyDialog();
    }

    if ( M_train_data_changed )
    {
        result |= showTrainDataSaveNotifyDialog();
    }

    return result;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::save()
{
    if ( M_filepath.empty() )
    {
        return saveAs();
    }

    // create backup file
    if ( FEditConfig::instance().autoBackup() )
    {
        char time_str[64];
        std::time_t current = std::time( NULL );
        std::tm * local_time = std::localtime( &current );
        if ( local_time )
        {
            std::strftime( time_str, 64, ".%Y%m%d-%H%M", local_time );
        }
        else
        {
            std::snprintf( time_str, 64, ".time-%ld", current );
        }

        std::string backup_filepath = M_filepath + time_str;
        std::ifstream fin( M_filepath.c_str() );
        std::ofstream fout( backup_filepath.c_str() );
        if ( fin && fout )
        {
            std::istreambuf_iterator< char > in_itr( fin );
            std::ostreambuf_iterator< char > out_itr( fout );
            std ::copy( in_itr, std::istreambuf_iterator< char >(),
                        out_itr );

            fin.close();
            fout.close();
        }
    }

    bool success = false;
    {
        std::ofstream fout( M_filepath.c_str() );
        if ( ! fout.is_open() )
        {
            fout.close();
            return false;
        }
        success = save( fout );
        fout.close();
    }

    printMessageWithTime( "save formation" );
    return success;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::save( std::ostream & os )
{
    //std::cerr << "FEditData::save(ostream)" << std::endl;
    if ( ! M_formation )
    {
        return false;
    }

    M_formation->print( os );

    M_modified = false;

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::saveAs()
{
    wxString wild_card( wxT( "Formation file (*.conf)|*.conf"
                             "|All files (*.*)|*.*" ) );
    wxFileDialog file_dialog( M_frame, // parent window
                              wxT( "Save Formation Parameter" ), // dialog name shown on titlebar
                              wxT( "" ), // default dir
                              wxT( "" ), // default file
                              wild_card,
                              wxSAVE
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return false;
    }

#ifdef UNICODE
    std::string filepath = (const char*)(file_dialog.GetPath().mbc_str());
#else
    std::string filepath = (const char*)(file_dialog.GetPath().c_str());
#endif

    if ( filepath.empty() )
    {
        return false;
    }

    if ( filepath.length() <= 5
         || filepath.compare( filepath.length() - 5, 5, ".conf" ) != 0 )
    {
        filepath += ".conf";
    }

    std::ofstream fout( filepath.c_str() );
    if ( ! fout.is_open() )
    {
        ::wxMessageBox( wxT( "Failed to open the file!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        M_frame ); // parent frame
        std::cerr << "Error. Failed to open file output steramr" << std::endl;
        fout.close();
        return false;
    }

    bool success = save( fout );
    fout.close();

    if ( success )
    {
        M_filepath = filepath;

        wxString filepath_wx( filepath.c_str(),
                              *wxConvCurrent );
        wxString only_filename;
        wxFileName::SplitPath( filepath_wx,
                               NULL,
                               &only_filename,
                               NULL );
        wxString title = wxT( "Formation Editor - " );
        title += only_filename;
        title += wxT( " -");
        M_frame->SetTitle( title );

        printMessageWithTime( "save formation as [%s]",
                              filepath.c_str() );
    }

    return success;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::saveTrainData()
{
    if ( M_filepath_train.empty() )
    {
        return saveTrainDataAs();
    }

    // create backup file
    if ( FEditConfig::instance().autoBackup() )
    {
        char time_str[64];
        std::time_t current = std::time( NULL );
        std::tm * local_time = std::localtime( &current );
        if ( local_time )
        {
            std::strftime( time_str, 64, ".%Y%m%d-%H%M", local_time );
        }
        else
        {
            std::snprintf( time_str, 64, ".time-%ld", current );
        }

        std::string backup_filepath = M_filepath_train + time_str;
        std::ifstream fin( M_filepath_train.c_str() );
        std::ofstream backup( backup_filepath.c_str() );
        if ( fin
             && backup )
        {
            std::istreambuf_iterator< char > in_itr( fin );
            std::ostreambuf_iterator< char > out_itr( backup );
            std::copy( in_itr, std::istreambuf_iterator< char >(),
                       out_itr );

            fin.close();
            backup.close();
        }
    }

    std::ofstream fout( M_filepath_train.c_str() );
    if ( ! fout.is_open() )
    {
        fout.close();
        return false;
    }

    bool success = saveTrainData( fout );
    fout.close();
    M_train_data_changed = false;

    printMessageWithTime( "save training data" );
    return success;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::saveTrainData( std::ostream & os )
{
    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_train_data.begin();
          it != M_train_data.end();
          ++it )
    {
        os << it->ball_.x << ' ' << it->ball_.y << ' ';
        for ( std::vector< rcsc::Vector2D >::iterator p = it->players_.begin();
              p != it->players_.end();
              ++p )
        {
            os << p->x << ' ' << p->y << ' ';
        }
        os << std::endl;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::saveTrainDataAs()
{
    if ( M_train_data.empty() )
    {
        ::wxMessageBox( wxT( "Training data is empty!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        M_frame ); // parent frame
        return false;
    }

    wxString wild_card( wxT( "Training data file (*.dat)|*.dat"
                             "|All files (*.*)|*.*" ) );
    wxFileDialog file_dialog( M_frame, // parent window
                              wxT( "Save Training Data" ), // dialog name shown on titlebar
                              wxT( "" ), // default dir
                              wxT( "" ), // default file
                              wild_card,
                              wxSAVE
                              );
    if ( file_dialog.ShowModal() != wxID_OK )
    {
        return false;
    }

#ifdef UNICODE
    std::string filepath = (const char*)(file_dialog.GetPath().mbc_str());
#else
    std::string filepath = (const char*)(file_dialog.GetPath().c_str());
#endif

    if ( filepath.empty() )
    {
        return false;
    }

    if ( filepath.length() <= 4
         || filepath.compare( filepath.length() - 4, 4, ".dat" ) != 0 )
    {
        filepath += ".dat";
    }

    std::ofstream fout( filepath.c_str() );
    if ( ! fout.is_open() )
    {
        ::wxMessageBox( wxT( "Failed to open the file!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        M_frame ); // parent frame
        std::cerr << "Error. Failed to open file output steramr" << std::endl;
        fout.close();
        return false;
    }

    bool success = saveTrainData( fout );
    fout.close();
    if ( success )
    {
        M_filepath_train = filepath;
        M_train_data_changed = false;
    }

    printMessageWithTime( "save training data as [%s]",
                          filepath.c_str() );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
FEditData::showSaveNotifyDialog()
{
    //std::cerr << "FEditData::showSaveNotifyDialog()" << std::endl;
    int answer = ::wxMessageBox( wxT( "Formation parameter are modified. \n"
                                      "Save?" ),
                                 wxT( "Notify" ), // title
                                 wxYES_NO | wxCANCEL,
                                 M_frame );

    if ( answer == wxYES )
    {
        save();
        M_modified = false;
    }

    return answer;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
FEditData::showTrainDataSaveNotifyDialog()
{
    //std::cerr << "FEditData::showTrainDataSaveNotifyDialog()" << std::endl;
    int answer = ::wxMessageBox( wxT( "Train data is modified. \n"
                                      "Save?" ),
                                 wxT( "Notify" ), // title
                                 wxYES_NO | wxCANCEL,
                                 M_frame );

    if ( answer == wxYES )
    {
        saveTrainData();
        M_train_data_changed = false;
    }

    return answer;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::notifyChanged()
{
    if ( M_canvas )
    {
        M_canvas->draw();
    }
    if ( M_dialog )
    {
        M_dialog->update();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::notifyDraw()
{
    if ( M_canvas )
    {
        M_canvas->draw();
    }
    if ( M_dialog )
    {
        M_dialog->update();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::updatePlayerPosition()
{
    if ( ! M_player_auto_move )
    {
        return;
    }

    if ( ! M_formation )
    {
        //std::cerr << "No formation param set" << std::endl;
        return;
    }

    // update all players' position using formation param

    const rcsc::Vector2D & bpos = M_ball;

    for ( int unum = 1; unum <= 11; ++unum )
    {
        rcsc::Vector2D pos = M_formation->getPosition( unum, bpos );
        pos.x = rcsc::min_max( -53.0, pos.x, 53.0 );
        pos.y = rcsc::min_max( -35.0, pos.y, 35.0 );

        M_our_players[unum - 1] = pos;
    }

    M_our_dragged.assign( 11, 0 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::updateTriangulation()
{
    rcsc::Rect2D pitch( - FEditConfig::PITCH_LENGTH * 0.5,
                        - FEditConfig::PITCH_WIDTH * 0.5,
                        FEditConfig::PITCH_LENGTH,
                        FEditConfig::PITCH_WIDTH );
    M_triangulation.init( pitch );
    /*
    M_triangulation.addVertex( 0.0, 0.0 );
    M_triangulation.addVertex( pitch.topLeft() );
    M_triangulation.addVertex( pitch.bottomLeft() );
    M_triangulation.addVertex( pitch.bottomRight() );
    M_triangulation.addVertex( pitch.topRight() );
    */
    const std::list< rcsc::Formation::Snapshot >::const_iterator end
        = getTrainingData().end();
    for ( std::list< rcsc::Formation::Snapshot >::const_iterator it
              = getTrainingData().begin();
          it != end;
          ++it )
    {
        M_triangulation.addVertex( it->ball_ );
        /*
        if ( it->ball_.absY() > 1.0 )
        {
            M_triangulation.addVertex( it->ball_.x,
                                       it->ball_.y * -1.0 );
        }
        */
    }

    M_triangulation.compute();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::reverseY()
{
    if ( ! M_formation )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** No formation!!"
                  << std::endl;
        return;
    }

    M_ball.y *= -1.0;

    std::vector< rcsc::Vector2D > old_positions = M_our_players;

    int unum = 1;
    for ( std::vector< rcsc::Vector2D >::iterator it = M_our_players.begin();
          it != M_our_players.end();
          ++it )
    {
        if ( M_formation->isCenterType( unum ) )
        {
            it->y *= -1.0;
        }
        else if ( M_formation->isMirrorType( unum ) )
        {
            int mirror_unum = M_formation->getMirrorNumber( unum );
            if ( mirror_unum == 0 ) continue;
            it->x = old_positions[mirror_unum - 1].x;
            it->y = old_positions[mirror_unum - 1].y * -1.0;
        }
        else if ( M_formation->isSideType( unum ) )
        {
            it->y *= -1.0;
            for ( int iunum = 1; iunum <= 11; ++iunum )
            {
                if ( M_formation->getMirrorNumber( iunum ) == unum )
                {
                    it->x = old_positions[iunum - 1].x;
                    it->y = old_positions[iunum - 1].y * -1.0;
                }
            }
        }
        ++unum;
    }

    /*
    for ( std::vector< rcsc::Vector2D >::iterator it = M_opp_players.begin();
          it != M_opp_players.end();
          ++it )
    {
        it->y *= -1.0;
    }
    */
}


/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::moveBallTo( const double & x,
                         const double & y )
{
    rcsc::Vector2D pos( x, y );
    pos.y = std::max( - FEditConfig::PITCH_WIDTH * 0.5 - 2.0, pos.y );
    pos.y = std::min( + FEditConfig::PITCH_WIDTH * 0.5 + 2.0, pos.y );
    pos.x = std::max( - FEditConfig::PITCH_LENGTH * 0.5 - 2.0, pos.x );
    pos.x = std::min( + FEditConfig::PITCH_LENGTH * 0.5 + 2.0, pos.x );

    M_ball = pos;
    if ( pos.absY() < 1.0 )
    {
        M_ball.y = 0.0;
    }

    updatePlayerPosition();
    notifyDraw();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::dropBallTo( const double & x,
                         const double & y )
{
    printMessageWithTime( "drop ball to (%.2f %.2f)", x, y );

    moveBallTo( x, y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::selectObject( const double & x,
                           const double & y )
{
    const rcsc::Vector2D pos( x, y );
    const double dist2_thr = 1.5 * 1.5;

    SelectType old_selection = M_select_type;

    double mindist2 = 200.0 * 200.0;

    if ( M_ball_draggable )
    {
        double d2 = M_ball.dist2( pos );
        if ( d2 < dist2_thr )
        {
            //std::cerr << "selection update ball" << std::endl;
            M_select_type = SELECT_BALL;
            mindist2 = d2;
        }
    }
    //mindist2 = M_ball.dist2( pos );

    // check oppnent
#if 0
    for ( std::vector< rcsc::Vector2D >::iterator it = M_opp_players.begin();
          it != M_opp_players.end();
          ++it )
    {
        double d2 = it->dist2( pos );
        if ( d2 < dist2_thr
             && d2 < mindist2 )
        {
            M_select_type = SELECT_OPP;
            M_select_index = std::distance( M_opp_players.begin(), it );
            mindist2 = d2;
        }
    }
#endif

    // check our players
    int unum = 1;
    for ( std::vector< rcsc::Vector2D >::iterator it = M_our_players.begin();
          it != M_our_players.end();
          ++it, ++unum )
    {
        //if ( M_formation
        //&& M_formation->isUsingMirror( unum ) )
        //{
        //continue;
        //}

        double d2 = it->dist2( pos );
        if ( d2 < dist2_thr
             && d2 < mindist2 )
        {
            M_select_type = SELECT_OUR;
            M_select_index = std::distance( M_our_players.begin(), it );
            mindist2 = d2;
        }
    }

    if ( old_selection != M_select_type )
    {
        if ( M_select_type == SELECT_BALL )
        {
            printMessageWithTime( "select ball at (%.2f %.2f)",
                                  M_ball.x, M_ball.y );
            M_ball = pos;
        }
        else if ( M_select_type == SELECT_OUR )
        {
            printMessageWithTime( "select our player %d (%.2f %.2f)",
                                  M_select_index + 1,
                                  M_our_players[M_select_index].x,
                                  M_our_players[M_select_index].y );
            M_our_players[M_select_index] = pos;
        }
        else if ( M_select_type == SELECT_OPP )
        {
            M_opp_players[M_select_index] = pos;
        }

        notifyDraw();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::releaseObject( const double & x,
                            const double & y )
{
    if ( M_select_type == SELECT_BALL )
    {
        printMessageWithTime( "release ball at (%.2f %.2f)", x, y );
    }
    else if ( M_select_type == SELECT_OUR )
    {
        printMessageWithTime( "release our player at (%.2f %.2f)", x, y );
    }
    else if ( M_select_type == SELECT_OPP )
    {
        printMessageWithTime( "release opp player at (%.2f %.2f)", x, y );
    }

    M_select_type = NO_SELECT;
    notifyDraw();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setObjectDragPoint( double x,
                                 double y )
{
    if ( M_select_type == SELECT_BALL )
    {
        moveBallTo( x, y );
        return;
    }

    y = std::max( - FEditConfig::PITCH_WIDTH * 0.5 - 2.0, y );
    y = std::min( + FEditConfig::PITCH_WIDTH * 0.5 + 2.0, y );
    x = std::max( - FEditConfig::PITCH_LENGTH * 0.5 - 2.0, x );
    x = std::min( + FEditConfig::PITCH_LENGTH * 0.5 + 2.0, x );

    switch ( M_select_type ) {
    case SELECT_OUR:
        M_our_dragged[M_select_index] = 1;
        M_our_players[M_select_index].assign( x, y );
        notifyDraw();
        break;
    case SELECT_OPP:
        M_opp_players[M_select_index].assign( x, y );
        notifyDraw();
        break;
    default:
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::toggleBallDraggable()
{
    printMessageWithTime( "toggleBallDraggable" );
    M_ball_draggable = ! M_ball_draggable;
    if ( !! M_ball_draggable )
    {
        notifyDraw();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::togglePlayerAutoMove()
{
    printMessageWithTime( "togglePlayerAutoMove" );
    M_player_auto_move = ! M_player_auto_move;
    if ( M_player_auto_move )
    {
        notifyDraw();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setTrainingTypeName( const std::string & type_name )
{
    printMessageWithTime( "setFormationName. [%s]",
                          type_name.c_str() );

    M_training_type_name = type_name;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::updateRoleData( const int unum,
                             const int mirror_unum,
                             const std::string & role_name )
{
    if ( ! M_formation )
    {
        return;
    }

    if ( M_formation->updateRole( unum, mirror_unum, role_name ) )
    {
        M_modified = true;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setBallPosition( const double & x,
                              const double & y )
{
    printMessageWithTime( "setBallPosition (%.2f %.2f)", x, y );

    M_ball.assign( x, y );
}


/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setPlayerPosition( const int unum,
                                const double & x,
                                const double & y )
{
    printMessageWithTime( "setPlayerPosition unum=%d (%.2f %.2f)", unum, x, y );

    M_our_players[unum - 1].assign( x, y );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::recordCurrentPosition()
{
    printMessageWithTime( "recordCurrentPosition" );

    if ( ! M_formation )
    {
        printMessageWithTime( "recordCurrentPosition. no formation" );
        return false;
    }

    if ( static_cast< int >( M_train_data.size() ) >= MAX_TRAINING_DATA_SIZE )
    {
        ::wxMessageBox( wxT( "Too many recorded positin!!" ),
                        wxT( "Warning" ),   // title
                        wxOK | wxICON_ERROR, // style
                        M_frame ); // parent frame
        printMessageWithTime( "recordCurrentPosition. too many." );
        return false;
    }

    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_train_data.begin();
          it != M_train_data.end();
          ++it )
    {
        if ( it->ball_.dist( M_ball ) < 2.0 )
        {
            std::cerr << "*** WARNING ***"
                      << " Exist too near recorded position!" << std::endl;
            ::wxMessageBox( wxT( "Exist too near recorded positin!!" ),
                            wxT( "Warning" ),   // title
                            wxOK | wxICON_ERROR, // style
                            M_frame ); // parent frame
            printMessageWithTime( "recordCurrentPosition. exist too near previous data." );
            return false;
        }
    }


    M_train_data.push_back( rcsc::Formation::Snapshot( M_ball, M_our_players ) );
    std::cerr << "Save positin. current data size = " << M_train_data.size()
              << std::endl;

    M_train_data_changed = true;

    M_frame->setTrainDataIndexMax( static_cast< int >( M_train_data.size() ) );

    updateTriangulation();

    printMessageWithTime( "recordCurrentPosition. done. size=%d",
                          M_train_data.size() );

    notifyDraw();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::focusTrainData( const int idx )
{
    printMessageWithTime( "focusTrainData index=%d", idx );

    std::list< rcsc::Formation::Snapshot >::iterator it = M_train_data.begin();

    int count = 1;
    while ( it != M_train_data.end() )
    {
        if ( idx == count )
        {
            M_ball = it->ball_;
            for ( std::size_t i = 0; i < 11; ++i )
            {
                M_our_players[i] = it->players_[i];
            }
            notifyDraw();
            break;
        }
        ++count;
        ++it;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::replaceTrainData( const int idx )
{
    printMessageWithTime( "replaceTrainData index=%d", idx );
    if ( idx == 0 )
    {
        return false;
    }

    std::list< rcsc::Formation::Snapshot >::iterator it = M_train_data.begin();
    int count = 1;
    while ( it != M_train_data.end() )
    {
        if ( idx == count )
        {
            *it = rcsc::Formation::Snapshot( M_ball, M_our_players );
            M_train_data_changed = true;
            break;
        }
        ++count;
        ++it;
    }

    updateTriangulation();
    notifyDraw();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::deleteTrainData( const int idx )
{
    printMessageWithTime( "deleteTrainData index=%d", idx );
    if ( idx == 0 )
    {
        return false;
    }

    std::list< rcsc::Formation::Snapshot >::iterator it = M_train_data.begin();
    int count = 1;
    while ( it != M_train_data.end() )
    {
        if ( idx == count )
        {
            M_train_data.erase( it );
            M_frame->setTrainDataIndexMax( static_cast< int >( M_train_data.size() ) );
            M_train_data_changed = true;
            break;
        }
        ++count;
        ++it;
    }

    updateTriangulation();
    notifyDraw();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::train()
{
    printMessageWithTime( "train" );

    if ( ! M_formation )
    {
        printMessageWithTime( "train. no formation" );
        return;
    }

    M_formation->train( M_train_data );

    M_modified = true;
}
