// -*-c++-*-

/*!
  \file debug_log_data.cpp
  \brief debug log data Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "debug_log_data.h"

#include <iostream>
#include <cstdio>

/*-------------------------------------------------------------------*/
/*!
  \param buf debug message string. but original input string contains
  <Time> info on top of the string.
*/
bool
DebugLogData::parse( const char * buf )
{
    // <Level> <Type> <Content>

    int level;
    char type;
    int n_read = 0;

    if ( std::sscanf( buf, " %d %c %n",
                      &level, &type,
                      &n_read ) != 2
         || n_read == 0 )
    {
        std::cerr << "DebugLogData. Illegal format [" << buf
                  << "]  read level =" << level
                  << "  read type = " << type
                  << "  read length = " << n_read
                  << std::endl;
        return false;
    }

    buf += n_read;

    switch ( type ) {
    case 'T':
        M_text_list.push_back
            ( std::pair< int, std::string >( level, std::string( buf ) ) );
        M_text_list.back().second += '\n';
        return true;
    case 'C':
        return addCircle( level, buf );
    case 'M':
        return addMessage( level, buf );
    case 'L':
        return addLine( level, buf );
    case 'P':
        return addPoint( level, buf );
    default:
        break;
    }

    std::cerr << "DebugLogData. Unsupported type message " << buf << std::endl;
    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
DebugLogData::addCircle( const int level,
                         const char * buf )
{
    // <x> <y> <r>
    CircleT c;
    if ( std::sscanf( buf, " %lf %lf %lf ",
                      &c.x_, &c.y_, &c.r_ ) != 3 )
    {
        std::cerr << "DebugLogData. Illegal Circle" << std::endl;
        return false;
    }
    c.level_ = level;

    M_circle_cont.push_back( c );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
DebugLogData::addMessage( const int level,
                          const char * buf )
{
    // <x> <y> <str>
    MessageT mes;
    int n_read = 0;
    if ( std::sscanf( buf, " %lf %lf ",
                      &mes.x_, &mes.y_ ) != 2
         || n_read == 0 )
    {
        std::cerr << "DebugLogData. Illegal Message" << std::endl;
        return false;
    }
    buf += n_read;

    mes.message_ = buf;
    mes.level_ = level;

    M_message_cont.push_back( mes );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
DebugLogData::addLine( const int level,
                       const char * buf )
{
    // <x1> <y1> <x2> <y2>
    LineT line;
    if ( std::sscanf( buf, " %lf %lf %lf %lf ",
                      &line.x1_, &line.y1_,
                      &line.x2_, &line.y2_ ) != 4 )
    {
        std::cerr << "DebugLogData. Illegal Line" << std::endl;
        return false;
    }

    line.level_ = level;

    M_line_cont.push_back( line );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
DebugLogData::addPoint( const int level,
                        const char * buf )
{
    // <x> <y>
    PointT point;
    if ( std::sscanf( buf, " %lf %lf ",
                      &point.x_, &point.y_ ) != 2 )
    {
        std::cerr << "DebugLogData. Illegal Point" << std::endl;
        return false;
    }

    point.level_ = level;

    M_point_cont.push_back( point );
    return true;
}
