// -*-c++-*-

/*!
  \file player.h
  \brief player data class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCERWINDOW2_MODEL_PLAYER_H
#define SOCCERWINDOW2_MODEL_PLAYER_H

#include <rcsc/types.h>
#include <rcsc/rcg/types.h>

/*!
  \brief ball info class

  This class contains ball info by RCG data format.
  But ball_t has local byte order variables.
*/
class Player {
private:
    enum {
        HAVE_TYPE    = 0x0001, //!< bit hetero id
        HAVE_DELTA   = HAVE_TYPE << 1, //!< bit speed info
        HAVE_STAMINA = HAVE_TYPE << 2, //!< bit stamina info
        HAVE_VIEW    = HAVE_TYPE << 3, //!< bit view mode & width info
        HAVE_COUNT   = HAVE_TYPE << 4, //!< bit action count info
    };

    //! RCG format player data. But variables are recorded by local byte order.
    rcsc::rcg::player_t M_data;

    //! side info
    rcsc::SideID M_side;
    //! uniform number
    int M_unum;
    //! information level flag
    int M_bits;

    //! real body direction
    double M_body;
    //! real head direction
    double M_head;
    //! real view width
    double M_view_width;

public:
    //! default construct.
    Player();

    /*!
      \brief construct from v3 protocol
      \param s side ID
      \param unum uniform number
      \param p v2 RCD data
    */
    Player( const rcsc::SideID s,
            const int unum,
            const rcsc::rcg::pos_t & p )
      {
          assign( s, unum, p );
      }

    /*!
      \brief construct from v3 protocol
      \param s side ID
      \param unum uniform number
      \param p v3 RCD data
    */
    Player( const rcsc::SideID side,
            const int unum,
            const rcsc::rcg::player_t & p )
      {
          assign( side, unum, p );
      }

    //! assign new data from v2 RCG data
    void assign( const rcsc::SideID side,
                 const int unum,
                 const rcsc::rcg::pos_t & p );

    //! assign new data from v3 RCG data
    void assign( const rcsc::SideID side,
                 const int unum,
                 const rcsc::rcg::player_t & p );

    void convertTo( rcsc::rcg::player_t & to ) const;

    void convertTo( rcsc::rcg::pos_t & to ) const;

    /*!
      \brief returns side ID
      \return side ID
    */
    rcsc::SideID side() const
      {
          return M_side;
      }

    /*!
      \brief returns uniform number
      \return uniform number
    */
    int unum() const
      {
          return M_unum;
      }

private:
    /*!
      \brief returns status flag bit
      \return short int status flag bit
     */
    boost::int16_t mode() const
      {
          return M_data.mode;
      }

    //! assign real body direction
    void setBody();
    //! assign real head direction
    void setHead();
    //! assign real view width
    void setViewWidth();

public:
    /*!
      \brief returns heterogeneous player type ID
      \return int type heterogeneous player type ID
     */
    int type() const
      {
          return static_cast< int >( M_data.type );
      }

    /*!
      \brief check if player has hetero type ID.
      \retval true player has hetero type ID.
      \retval false no hetero type ID.
    */
    bool hasType() const
      {
          return M_bits & HAVE_TYPE;
      }

    /*!
      \brief check if player has velocity info.
      \retval true player has velocity info.
      \retval false no velocity info.
    */
    bool hasDelta() const
      {
          return M_bits & HAVE_DELTA;
      }

    /*!
      \brief check if player has stamina info.
      \retval true player has stamina info.
      \retval false no stamina info.
    */
    bool hasStamina() const
      {
          return M_bits & HAVE_STAMINA;
      }

    /*!
      \brief check if player has view mode info and neck angle.
      \retval true player has view mode info and neck angle.
      \retval false no view mode or neck angle.
    */
    bool hasView() const
      {
          return M_bits & HAVE_VIEW;
      }

    /*!
      \brief check if player has command count info.
      \retval true player has command count info.
      \retval false no command count info.
    */
    bool hasCommandCount() const
      {
          return M_bits & HAVE_COUNT;
      }

    /*!
      \brief check if player is stand on.
      \retval true player is in field.
      \retval false player is out of pitch.
     */
    bool isAlive() const
      {
          return mode() & rcsc::rcg::STAND;
      }

    /*!
      \brief check if player is goalie
      \retval true player is goalie.
      \retval false player is not goalie.
     */
    bool isGoalie() const
      {
          return mode() & rcsc::rcg::GOALIE;
      }

    /*!
      \brief check if player is kicking now.
      \retval true player is kicking.
      \retval false no kicks.
     */
    bool isKick() const
      {
          return mode() & rcsc::rcg::KICK;
      }

   /*!
      \brief check if player failed to kick.
      \retval true player failed to kick.
      \retval false successfully kicked or no kicks.
     */
    bool isKickFault() const
      {
          return mode() & rcsc::rcg::KICK_FAULT;
      }

   /*!
      \brief check if player is catching the ball.
      \retval true player is catching.
      \retval false no catching.
     */
    bool isCatch() const
      {
          return mode() & rcsc::rcg::CATCH;
      }

   /*!
      \brief check if player failed to catch the ball.
      \retval true player failed to catch.
      \retval false successfully catched or no catchings.
     */
    bool isCatchFault() const
      {
          return mode() & rcsc::rcg::CATCH_FAULT;
      }

    /*!
      \brief check if player collide with ball.
      \retval true player collide with ball.
      \retval false no collisions with ball.
     */
    bool isCollidedBall() const
      {
          return ( mode() & rcsc::rcg::BALL_COLLIDE
                   || mode() & rcsc::rcg::BALL_TO_PLAYER
                   || mode() & rcsc::rcg::PLAYER_TO_BALL );
      }

    /*!
      \brief check if player collide with other players.
      \retval true player collide with other players.
      \retval false no collisions with other players.
     */
    bool isCollidedPlayer() const
      {
          return mode() & rcsc::rcg::PLAYER_COLLIDE;
      }

    /*!
      \brief check if player is tackling.
      \retval true player is tackling.
      \retval false no tackle.
     */
    bool isTackle() const
      {
          return mode() & rcsc::rcg::TACKLE;
      }

    /*!
      \brief check if player failed to tackle.
      \retval true player failed to tackle.
      \retval false successfully tackled or no tackle.
     */
    bool isTackleFault() const
      {
          return mode() & rcsc::rcg::TACKLE_FAULT;
      }

    /*!
      \brief returns real coordinate value.
      \return real coordinate X
     */
    double x() const
      {
          return static_cast< double >( M_data.x )
              / rcsc::rcg::SHOWINFO_SCALE2;
      }

    /*!
      \brief returns real coordinate value.
      \return real coordinate Y
     */
    double y() const
      {
          return static_cast< double >( M_data.y )
              / rcsc::rcg::SHOWINFO_SCALE2;
      }

    /*!
      \brief returns velocity value.
      \return velocity X
     */
    double deltaX() const
      {
          return static_cast< double >( M_data.deltax )
              / rcsc::rcg::SHOWINFO_SCALE2;
      }

    /*!
      \brief returns velocity value.
      \return velocity Y
     */
    double deltaY() const
      {
          return static_cast< double >( M_data.deltay )
              / rcsc::rcg::SHOWINFO_SCALE2;
      }

    /*!
      \brief returns body angle.
      \return body angle by degree
     */
    const
    double & body() const
      {
          return M_body;
      }

    /*!
      \brief returns global head angle.
      \return global head angle by degree
     */
    const
    double & head() const
      {
          return M_head;
      }

    /*!
      \brief returns view cone angle.
      \return view cone angle by degree.
     */
    const
    double & viewWidth() const
      {
          return M_view_width;
      }

    /*!
      \brief returns stamina value
      \return real stamina value.
     */
    double stamina() const
      {
          return static_cast< double >( M_data.stamina )
              / rcsc::rcg::SHOWINFO_SCALE2;
      }

    /*!
      \brief returns effort value
      \return real effort value.
     */
    double effort() const
      {
          return static_cast< double >( M_data.effort )
              / rcsc::rcg::SHOWINFO_SCALE2;
      }

    /*!
      \brief returns recovery value
      \return real recovery value.
     */
    double recovery() const
      {
          return static_cast< double >( M_data.recovery )
              / rcsc::rcg::SHOWINFO_SCALE2;
      }

    /*!
      \brief returns command count
      \return kick count
     */
    rcsc::rcg::Int16 kickCount() const
      {
          return M_data.kick_count;
      }

    /*!
      \brief returns command count
      \return dash count
     */
    rcsc::rcg::Int16 dashCount() const
      {
          return M_data.dash_count;
      }

    /*!
      \brief returns command count
      \return turn count
     */
    rcsc::rcg::Int16 turnCount() const
      {
          return M_data.turn_count;
      }

    /*!
      \brief returns command count
      \return say count
     */
    rcsc::rcg::Int16 sayCount() const
      {
          return M_data.say_count;
      }

    /*!
      \brief returns command count
      \return turn_neck count
     */
    rcsc::rcg::Int16 turnNeckCount() const
      {
          return M_data.turn_neck_count;
      }

    /*!
      \brief returns command count
      \return catch count
     */
    rcsc::rcg::Int16 catchCount() const
      {
          return M_data.catch_count;
      }

    /*!
      \brief returns command count
      \return move count
     */
    rcsc::rcg::Int16 moveCount() const
      {
          return M_data.move_count;
      }

    /*!
      \brief returns command count
      \return change_view count
     */
    rcsc::rcg::Int16 changeViewCount() const
      {
          return M_data.change_view_count;
      }

    /*!
      \brief check if player has full effort.
      \param max_effort real maximal effort value.
      \retval true player has max effort value.
      \retval false player's effort value is decayed.
     */
    bool hasFullEffort( const double & max_effort ) const
      {
          return ( static_cast< rcsc::rcg::Int32 >( max_effort
                                                    * rcsc::rcg::SHOWINFO_SCALE2 )
                   == M_data.effort );
      }

    /*!
      \brief check if player has full recovery.
      \retval true player has max recovery value.
      \retval false player's recovery value is decayed.
     */
    bool hasFullRecovery() const
      {
          // real recovery == 1.0
          return rcsc::rcg::SHOWINFO_SCALE2 == M_data.recovery;
      }
};

#endif
