// -*-c++-*-

/*!
  \file debug_painter.cpp
  \brief debug info painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "debug_painter.h"

#include "draw_config.h"
// model
#include "main_data.h"

#include <rcsc/common/server_param.h>

#include <iostream>
#include <cmath>

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::draw( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    if ( ! vconf.isShownDebugView() )
    {
        return;
    }

    int number = vconf.selectedNumber();
    if ( number == 0 )
    {
        return;
    }

    const rcsc::SideID player_side = ( number < 0 ? rcsc::RIGHT : rcsc::LEFT );

    const std::map< long, DebugViewCont > & view_map
        = ( player_side == rcsc::RIGHT
            ? M_main_data.viewHolder().rightDebugView()
            : M_main_data.viewHolder().leftDebugView() );

    MonitorViewConstPtr monitor_view
        = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! monitor_view )
    {
        return;
    }

    std::map< long, DebugViewCont >::const_iterator it
        = view_map.find( monitor_view->cycle() );

    if ( it == view_map.end() )
    {
        return;
    }

    number = std::abs( number );

    if ( static_cast< int >( it->second.size() ) < number )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " Error debug view container size is small."
                  << std::endl;
        return;
    }

    const DebugViewPtr view = it->second[ number - 1 ];
    if ( ! view )
    {
        // no data!
        //std::cerr << __FILE__ << ":" << __LINE__
        //          << " Error. No debug view data of " << number
        //          << std::endl;
        return;
    }

    const DrawConfig & dconf = DrawConfig::instance();

    // draw self
    if ( vconf.isShownDebugViewSelf() )
    {
        drawSelf( painter, player_side, *view );
    }

    // draw players
    if ( vconf.isShownDebugViewPlayers() )
    {
        drawPlayers( painter,
                     player_side,
                     view->teammates(),
                     ( vconf.isShownDebugViewTarget()
                       ? view->targetTeammate()
                       : -1 ),
                     dconf.debugTeammateBrush() );
        drawPlayers( painter,
                     player_side,
                     view->opponents(),
                     -1,
                     dconf.debugOpponentBrush() );
        drawPlayers( painter,
                     player_side,
                     view->unknownTeammates(),
                     -1,
                     dconf.debugUnknownTeammateBrush() );
        drawPlayers( painter,
                     player_side,
                     view->unknownOpponents(),
                     -1,
                     dconf.debugUnknownOpponentBrush() );
        drawPlayers( painter,
                     player_side,
                     view->unknownPlayers(),
                     -1,
                     dconf.debugUnknownPlayerBrush() );
    }

    // draw ball
    if ( vconf.isShownDebugViewBall()
         && view->ball() )
    {
        drawBall( painter, player_side, *view );
    }

    // draw lines
    if ( vconf.isShownDebugViewFigure() )
    {
        drawLines( painter, player_side, *view );
        drawTriangles( painter, player_side, *view );
        drawRectangles( painter, player_side, *view );
        drawCircles( painter, player_side, *view );
    }

    if ( vconf.isShownDebugViewMessage()
         && ! view->message().empty() )
    {
        drawMessage( painter, view->message() );
    }
}

/*-------------------------------------------------------------------*/
/*!
  pos, vel, body, neck, comment
*/
void
DebugPainter::drawSelf( QPainter & painter,
                        const rcsc::SideID player_side,
                        const DebugViewData & view ) const
{
    const boost::shared_ptr< DebugViewData::SelfT > self = view.self();
    if ( ! self )
    {
        std::cerr << "DebugPainter::drawSelf. No self data!" << std::endl;
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    const int ix = vconf.getScreenX( (double)self->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    const int iy = vconf.getScreenY( (double)self->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    const int r = std::max( 2,
                            ( vconf.isEnlarged()
                              ? vconf.scale( 1.0 )
                              : vconf.scale( 0.3 ) ) );

    // draw targets
    if ( vconf.isShownDebugViewTarget()
         &&  view.targetPoint() )
    {
        int tx = vconf.getScreenX( view.targetPoint()->x * reverse );
        int ty = vconf.getScreenY( view.targetPoint()->y * reverse );

        painter.setPen( dconf.debugTargetPen() );
        painter.setBrush( dconf.transparentBrush() );

        painter.drawLine( tx - r, ty - r,
                          tx + r, ty + r );
        painter.drawLine( tx - r, ty + r,
                          tx + r, ty - r );
    }

    // draw body half circle
    double body_start_dir = - ( (double)self->body_ + 90.0 );
    if ( player_side == rcsc::RIGHT ) body_start_dir -= 180.0;

    painter.setPen( dconf.transparentPen() );
    painter.setBrush( dconf.debugSelfBrush() );

    painter.drawPie( ix - r,
                     iy - r,
                     r * 2,
                     r * 2,
                     static_cast< int >( rint( body_start_dir * 16 ) ),
                     180 * 16 );

    // draw edge

    painter.setPen( dconf.debugPlayerPen() );
    painter.setBrush( dconf.transparentBrush() );

    painter.drawEllipse( ix - r,
                         iy - r,
                         r * 2,
                         r * 2 );

    // draw comment
    if ( vconf.isShownDebugViewComment()
         && ! self->comment_.empty() )
    {
        painter.setFont( dconf.debugCommentFont() );
        painter.setPen( dconf.debugCommentFontPen() );

        painter.drawText( ix - r,
                          iy + r + painter.fontMetrics().ascent(),
                          QString::fromStdString( self->comment_ ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawBall( QPainter & painter,
                        const rcsc::SideID player_side,
                        const DebugViewData & view ) const
{
    const boost::shared_ptr< DebugViewData::BallT > & ball = view.ball();

    if ( ! ball )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    rcsc::Vector2D bpos( (double)ball->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse,
                         (double)ball->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
    rcsc::Vector2D bvel( (double)ball->vx_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse,
                         (double)ball->vy_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

    const double ball_decay = rcsc::ServerParam::i().ballDecay();
    const int ix = vconf.getScreenX( bpos.x );
    const int iy = vconf.getScreenY( bpos.y );
    const int r = std::max( 2,
                            ( vconf.isEnlarged()
                              ? vconf.scale( 0.25 )
                              : vconf.scale( 0.07 ) ) );

    painter.setPen( dconf.transparentPen() );
    painter.setBrush( dconf.debugBallBrush() );

    painter.drawEllipse( ix - r,
                         iy - r,
                         r * 2,
                         r * 2 );

    {
        QPainterPath path;
        painter.setPen( dconf.debugBallTracePen() );
        painter.setBrush( dconf.debugBallTraceBrush() );

        int bx = ix;
        int by = iy;
        for ( int i = 0; i < 10; ++i )
        {
            bpos += bvel;
            bvel *= ball_decay;
            bx = vconf.getScreenX( bpos.x );
            by = vconf.getScreenY( bpos.y );
            path.addEllipse( bx, by, 4, 4 );
        }
        path.moveTo( ix, iy );
        path.lineTo( bx, by );

        painter.drawPath( path );
    }

    // draw comment
    if ( vconf.isShownDebugViewComment()
         && ! ball->comment_.empty() )
    {
        painter.setFont( dconf.debugCommentFont() );
        painter.setPen( dconf.debugCommentFontPen() );

        painter.drawText( ix - r,
                          iy + r + painter.fontMetrics().ascent(),
                          QString::fromStdString( ball->comment_ ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawPlayers( QPainter & painter,
                           const rcsc::SideID player_side,
                           const DebugViewData::PlayerCont & players,
                           const int target_number,
                           const QBrush & body_brush ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    const bool comment = vconf.isShownDebugViewComment();
    const int r = std::max( 2,
                            ( vconf.isEnlarged()
                              ? vconf.scale( 1.0 )
                              : vconf.scale( 0.3 ) ) );

    const DebugViewData::PlayerCont::const_iterator end = players.end();
    for ( DebugViewData::PlayerCont::const_iterator it = players.begin();
          it != end;
          ++it )
    {
        const int ix = vconf.getScreenX( (double)(*it)->x_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );
        const int iy = vconf.getScreenY( (double)(*it)->y_ / rcsc::rcg::SHOWINFO_SCALE2 * reverse );

        if ( (*it)->body_ != -360 )
        {
            double body_start_dir = - ( (double)(*it)->body_ + 90.0 );
            if ( player_side == rcsc::RIGHT ) body_start_dir -= 180.0;
            //double body_end_dir = body_start_dir + 180.0;
            // draw body half circle
            painter.setPen( dconf.transparentPen() );
            painter.setBrush( body_brush );
            painter.drawPie( ix - r, iy - r,
                             r * 2, r * 2,
                             static_cast< int >( rint( body_start_dir * 16 ) ),
                             180 * 16 );
            // draw edge
            painter.setPen( dconf.debugPlayerPen() );
            painter.setBrush( dconf.transparentBrush() );
            painter.drawEllipse( ix - r, iy - r, r * 2, r * 2 );
        }
        else
        {
            // draw simple circle
            painter.setPen( dconf.transparentPen() );
            painter.setBrush( body_brush );
            painter.drawEllipse( ix - r, iy - r, r * 2, r * 2 );
        }

        if ( (*it)->unum_ > 0 )
        {
            painter.setFont( dconf.debugCommentFont() );
            painter.setPen( dconf.debugCommentFontPen() );

            painter.drawText( ix + r, iy + 4,
            QString::number( (*it)->unum_ ) );
        }

        if ( target_number == (*it)->unum_ )
        {
            painter.setPen( dconf.debugTargetPen() );
            painter.setBrush( dconf.transparentBrush() );
            int rr = r + 2;
            painter.drawEllipse( ix - rr, iy - rr,
                                 rr * 2, rr * 2 );
        }

        if ( comment
             && ! (*it)->comment_.empty() )
        {
            painter.setFont( dconf.debugCommentFont() );
            painter.setPen( dconf.debugCommentFontPen() );

            painter.drawText( ix - r,
                              iy + r + painter.fontMetrics().ascent(),
                              QString::fromStdString( (*it)->comment_ ) );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawLines( QPainter & painter,
                         const rcsc::SideID player_side,
                         const DebugViewData & view ) const
{
    if ( view.lines().empty() )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );

    painter.setPen( dconf.debugFigurePen() );
    painter.setBrush( dconf.transparentBrush() );

    QPainterPath path;

    const std::list< DebugViewData::LineT >::const_iterator end = view.lines().end();
    for ( std::list< DebugViewData::LineT >::const_iterator it = view.lines().begin();
          it != end;
          ++it )
    {
        path.moveTo( vconf.getScreenX( it->x1_ * reverse ),
                     vconf.getScreenY( it->y1_ * reverse ) );
        path.lineTo( vconf.getScreenX( it->x2_ * reverse ),
                     vconf.getScreenY( it->y2_ * reverse ) );
    }

    painter.drawPath( path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawTriangles( QPainter & painter,
                             const rcsc::SideID player_side,
                             const DebugViewData & view ) const
{
    if ( view.triangles().empty() )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );

    painter.setPen( dconf.debugFigurePen() );
    painter.setBrush( dconf.transparentBrush() );

    QPainterPath path;

    const std::list< DebugViewData::TriangleT >::const_iterator end = view.triangles().end();
    for ( std::list< DebugViewData::TriangleT >::const_iterator it = view.triangles().begin();
          it != end;
          ++it )
    {
        int x1 = vconf.getScreenX( it->x1_ * reverse );
        int y1 = vconf.getScreenY( it->y1_ * reverse );
        int x2 = vconf.getScreenX( it->x2_ * reverse );
        int y2 = vconf.getScreenY( it->y2_ * reverse );
        int x3 = vconf.getScreenX( it->x3_ * reverse );
        int y3 = vconf.getScreenY( it->y3_ * reverse );

        path.moveTo( x1, y1 );
        path.lineTo( x2, y2 );
        path.lineTo( x3, y3 );
        path.lineTo( x1, y1 );
    }

    painter.drawPath( path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawRectangles( QPainter & painter,
                              const rcsc::SideID player_side,
                              const DebugViewData & view ) const
{
    if ( view.rectangles().empty() )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );
    painter.setPen( dconf.debugFigurePen() );
    painter.setBrush( dconf.transparentBrush() );

    QPainterPath path;

    const std::list< DebugViewData::RectT >::const_iterator end = view.rectangles().end();
    for ( std::list< DebugViewData::RectT >::const_iterator it = view.rectangles().begin();
          it != end;
          ++it )
    {
        int left_x = vconf.getScreenX( it->left_x_ * reverse );
        int top_y = vconf.getScreenY( it->top_y_ * reverse );
        int right_x = vconf.getScreenX( it->right_x_ * reverse );
        int bottom_y = vconf.getScreenY( it->bottom_y_ * reverse );

        path.addRect( left_x,
                      top_y,
                      ( right_x - left_x ),
                      ( bottom_y - top_y ) );
    }

    painter.drawPath( path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawCircles( QPainter & painter,
                           const rcsc::SideID player_side,
                           const DebugViewData & view ) const
{
    if ( view.circles().empty() )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::RIGHT ? -1.0 : 1.0 );

    painter.setPen( dconf.debugFigurePen() );
    painter.setBrush( dconf.transparentBrush() );

    QPainterPath path;

    const std::list< DebugViewData::CircleT >::const_iterator end = view.circles().end();
    for ( std::list< DebugViewData::CircleT >::const_iterator it = view.circles().begin();
          it != end;
          ++it )
    {
        int r = vconf.scale( it->radius_ );
        path.addEllipse( vconf.getScreenX( it->center_x_ * reverse ) - r,
                         vconf.getScreenY( it->center_y_ * reverse ) - r,
                         r * 2,
                         r * 2 );
    }

    painter.drawPath( path );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugPainter::drawMessage( QPainter & painter,
                           const std::string & message ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    painter.setFont( dconf.debugMessageFont() );
    painter.setPen( dconf.debugMessageFontPen() );

    painter.drawText( 10,
                      M_main_data.viewConfig().scoreBoardHeight()
                      + painter.fontMetrics().ascent() + 2,
                      QString::fromStdString( message ) );
}
